"""
Tracing - Session tracing and replay for agent execution.

Provides:
- Span and Trace models for execution tracking
- Tracer for recording agent operations
- Storage backends (in-memory and SQLite)
- Session replay for debugging and analysis

Example:
    from agent_orchestrator.tracing import Tracer, SpanKind

    tracer = Tracer()
    trace = tracer.start_trace("task-123", "claude-code")

    async with tracer.span("generate_code", SpanKind.LLM_CALL) as span:
        result = await llm.generate(prompt)
        span.output_tokens = 500

    tracer.end_trace()
"""

from .models import (
    SpanKind,
    SpanStatus,
    Span,
    Trace,
    TimelineEvent,
    DecisionPoint,
)
from .tracer import (
    Tracer,
    TracerContext,
    get_tracer,
    start_trace,
    end_trace,
    span,
    get_global_tracer,
    set_global_tracer,
)
from .storage import (
    TraceStorage,
    InMemoryTraceStorage,
    SQLiteTraceStorage,
    get_trace_storage,
    set_trace_storage,
)
from .replay import (
    SessionReplay,
    get_session_replay,
    set_session_replay,
)

__all__ = [
    # Models
    "SpanKind",
    "SpanStatus",
    "Span",
    "Trace",
    "TimelineEvent",
    "DecisionPoint",
    # Tracer
    "Tracer",
    "TracerContext",
    "get_tracer",
    "start_trace",
    "end_trace",
    "span",
    "get_global_tracer",
    "set_global_tracer",
    # Storage
    "TraceStorage",
    "InMemoryTraceStorage",
    "SQLiteTraceStorage",
    "get_trace_storage",
    "set_trace_storage",
    # Replay
    "SessionReplay",
    "get_session_replay",
    "set_session_replay",
]
