"""
Reliability - Production hardening utilities.

This module provides:
- Rate limiting with exponential backoff
- Retry logic for transient failures
- Fallback model configuration
- Circuit breaker pattern
- Graceful shutdown handling

Usage:
    from agent_orchestrator.reliability import (
        RateLimiter,
        retry,
        RetryManager,
        CircuitBreaker,
        GracefulShutdown,
    )

    # Rate limiting
    limiter = RateLimiter(provider="anthropic")
    wait_time = await limiter.acquire(tokens=1000)
    if wait_time > 0:
        await asyncio.sleep(wait_time)

    # Retry decorator
    @retry(max_retries=3, initial_delay=1.0)
    async def call_api():
        return await api.request()

    # Graceful shutdown
    shutdown = GracefulShutdown()
    shutdown.register_signals()
    shutdown.register_task("stop_agents", stop_agents)
"""

from .rate_limiter import (
    RateLimiter,
    RateLimitConfig,
    RateLimitState,
    RateLimiterRegistry,
    DEFAULT_RATE_LIMITS,
    get_rate_limiter_registry,
    with_rate_limit,
)

from .retry import (
    RetryConfig,
    RetryManager,
    FallbackModel,
    FallbackConfig,
    FallbackRequired,
    CircuitBreaker,
    retry,
    is_transient_exception,
    is_rate_limit_exception,
    get_retry_manager,
    set_retry_manager,
    DEFAULT_FALLBACKS,
    TRANSIENT_EXCEPTIONS,
)

from .shutdown import (
    GracefulShutdown,
    ShutdownConfig,
    ShutdownTask,
    ShutdownResult,
    ShutdownState,
    get_shutdown_manager,
    set_shutdown_manager,
    stop_agents,
    cancel_pending_tasks,
    close_connections,
    persist_state,
    flush_logs,
)

__all__ = [
    # Rate Limiter
    "RateLimiter",
    "RateLimitConfig",
    "RateLimitState",
    "RateLimiterRegistry",
    "DEFAULT_RATE_LIMITS",
    "get_rate_limiter_registry",
    "with_rate_limit",
    # Retry
    "RetryConfig",
    "RetryManager",
    "FallbackModel",
    "FallbackConfig",
    "FallbackRequired",
    "CircuitBreaker",
    "retry",
    "is_transient_exception",
    "is_rate_limit_exception",
    "get_retry_manager",
    "set_retry_manager",
    "DEFAULT_FALLBACKS",
    "TRANSIENT_EXCEPTIONS",
    # Shutdown
    "GracefulShutdown",
    "ShutdownConfig",
    "ShutdownTask",
    "ShutdownResult",
    "ShutdownState",
    "get_shutdown_manager",
    "set_shutdown_manager",
    "stop_agents",
    "cancel_pending_tasks",
    "close_connections",
    "persist_state",
    "flush_logs",
]
