# Workspace Setup Runbook

## Purpose

This runbook describes how to set up and manage agent workspaces using git worktrees and tmux.

## Prerequisites

- git 2.20+ (for worktree support)
- tmux 3.0+
- The main repository is a git repository

## Initial Setup

### 1. Verify Dependencies

```bash
# Check git version
git --version  # Should be 2.20+

# Check tmux version
tmux -V  # Should be 3.0+

# Verify in a git repo
git rev-parse --is-inside-work-tree
```

### 2. Create Agent Workspaces

```bash
# Using the provided script
./scripts/create_worktrees.sh -p /path/to/repo -b main

# Or manually for a single agent
git branch feature/agent-claude-code main
git worktree add ../agent-claude-code feature/agent-claude-code
tmux new-session -d -s claude-code -c ../agent-claude-code
```

### 3. Verify Setup

```bash
# List worktrees
git worktree list

# List tmux sessions
tmux list-sessions
```

## Workspace Structure

Each agent gets:
- **Worktree**: `../agent-<agent-id>/` (sibling directory)
- **Branch**: `feature/<agent-id>` or custom branch name
- **tmux session**: Named `<agent-id>`

```
/path/to/main-repo/          # Main repository
/path/to/agent-claude-code/  # Claude Code worktree
/path/to/agent-gemini/       # Gemini worktree
/path/to/agent-codex/        # Codex worktree
```

## Managing Workspaces

### Switch to an Agent

```bash
# Attach to tmux session
tmux attach -t claude-code

# Or switch within tmux
tmux switch-client -t claude-code
```

### Run Command in Workspace

```bash
# Send keys to tmux
tmux send-keys -t claude-code "pytest tests/" Enter
```

### Capture Agent Output

```bash
# Get last 100 lines
tmux capture-pane -t claude-code -p -S -100
```

### Check Git Status

```bash
# From main repo, check worktree
cd ../agent-claude-code && git status
```

## Cleanup

### Remove Single Agent

```bash
# Kill tmux session
tmux kill-session -t claude-code

# Remove worktree
git worktree remove ../agent-claude-code

# Optionally delete branch
git branch -d feature/agent-claude-code
```

### Remove All Agents

```bash
./scripts/cleanup_worktrees.sh -p /path/to/repo
```

## Troubleshooting

### Worktree Already Exists

```bash
# Check if it's registered
git worktree list

# If orphaned, prune
git worktree prune

# Then recreate
git worktree add ../agent-<id> <branch>
```

### tmux Session Won't Start

```bash
# Check if already exists
tmux has-session -t <agent-id>

# Kill and recreate
tmux kill-session -t <agent-id>
tmux new-session -d -s <agent-id> -c <path>
```

### Branch Conflicts

```bash
# If branch diverged from main
cd ../agent-<id>
git fetch origin
git rebase origin/main

# Or merge
git merge origin/main
```

## Best Practices

1. **One agent per worktree**: Prevents conflicts
2. **Descriptive branch names**: `feature/<agent-id>-<task>`
3. **Regular cleanup**: Remove stale worktrees weekly
4. **Commit before switching**: Avoid losing work
5. **Use the orchestrator**: Let it manage workspaces when possible
