#!/bin/bash
# verify_deployment.sh - Post-deployment verification
#
# Usage: ./verify_deployment.sh [local|remote] <file_path>
#   Checks file exists, compares checksums, tails error logs

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
SSH_KEY="/tmp/aei_key"
REMOTE_USER="Julian"
REMOTE_HOST="18.225.0.90"
REMOTE_SCHEDULER="/var/www/vhosts/aeihawaii.com/httpdocs/scheduler"
LOCAL_DEV="/var/www/html/dev_scheduler/SCHEDULER"

usage() {
    echo "Usage: $0 [local|remote] <file_path>"
    echo ""
    echo "Verify a deployment by checking file existence, checksums, and error logs."
    echo ""
    echo "Arguments:"
    echo "  local|remote   Target environment to verify"
    echo "  file_path      Relative path within the scheduler application"
    echo ""
    echo "Examples:"
    echo "  $0 local system/application/controllers/acschedule.php"
    echo "  $0 remote system/application/controllers/acschedule.php"
}

if [ $# -lt 2 ] || [ "$1" = "--help" ]; then
    usage
    exit 0
fi

ENV="$1"
FILE_PATH="$2"
SSH_CMD="ssh -i $SSH_KEY -o StrictHostKeyChecking=no"

verify_local() {
    local target="$LOCAL_DEV/$FILE_PATH"
    echo "=== Local Verification ==="
    echo "File: $target"
    echo ""

    # Check file exists
    if [ -f "$target" ]; then
        echo "[OK] File exists"
        ls -la "$target"
    else
        echo "[FAIL] File not found!"
        return 1
    fi
    echo ""

    # Checksum
    echo "Checksum:"
    md5sum "$target"
    echo ""

    # Check for recent errors in Apache logs
    echo "Recent PHP errors (last 10 lines):"
    tail -10 /var/log/apache2/error.log 2>/dev/null || echo "(no error log found)"
    echo ""

    echo "Local site: http://aeihawaii.com/scheduler/"
}

verify_remote() {
    local target="$REMOTE_SCHEDULER/$FILE_PATH"
    echo "=== Remote Verification ==="
    echo "File: $REMOTE_USER@$REMOTE_HOST:$target"
    echo ""

    # Ensure SSH key
    if [ ! -f "$SSH_KEY" ]; then
        echo "SSH key not found. Run scripts/ssh_connect.sh first."
        exit 1
    fi

    # Check file exists
    echo "File check:"
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
        "if [ -f '$target' ]; then echo '[OK] File exists'; ls -la '$target'; else echo '[FAIL] File not found!'; fi"
    echo ""

    # Checksum
    echo "Remote checksum:"
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "md5sum '$target'"
    echo ""

    # Local checksum for comparison
    local local_file="$LOCAL_DEV/$FILE_PATH"
    if [ -f "$local_file" ]; then
        echo "Local checksum (for comparison):"
        md5sum "$local_file"
        echo ""

        # Compare
        local_md5=$(md5sum "$local_file" | cut -d' ' -f1)
        remote_md5=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "md5sum '$target'" | cut -d' ' -f1)
        if [ "$local_md5" = "$remote_md5" ]; then
            echo "[OK] Checksums MATCH"
        else
            echo "[WARN] Checksums DIFFER"
        fi
    fi
    echo ""

    # Check ownership
    echo "File ownership:"
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "ls -la '$target'"
    echo ""

    # Tail error logs
    echo "Recent Apache errors (last 10 lines):"
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "tail -10 /var/log/httpd/error_log 2>/dev/null || echo '(no errors)'"
    echo ""

    echo "Production site: https://aeihawaii.com/dev/scheduler/"
}

case "$ENV" in
    local)
        verify_local
        ;;
    remote)
        verify_remote
        ;;
    *)
        echo "ERROR: First argument must be 'local' or 'remote'"
        usage
        exit 1
        ;;
esac
