#!/bin/bash
# sync_from_remote.sh - Pull latest code from remote -> backup mirror -> local dev
#
# Wraps the 3-step sync process with config file protection.
#
# Usage: ./sync_from_remote.sh [--dry-run] [--skip-db]

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
SYNC_SCRIPT="/mnt/e/remote_scheduler_backup/scripts/sync_source.sh"
BACKUP_MIRROR="/mnt/e/remote_scheduler_backup/source"
LOCAL_DEV="/var/www/html/dev_scheduler/SCHEDULER"

DRY_RUN=""
SKIP_DB=false

usage() {
    echo "Usage: $0 [--dry-run] [--skip-db]"
    echo ""
    echo "Sync scheduler code from remote production to local development."
    echo ""
    echo "Steps:"
    echo "  1. Pull code from remote server to backup mirror"
    echo "  2. Copy from backup mirror to local dev (protects config files)"
    echo "  3. (Optional) Sync database"
    echo ""
    echo "Options:"
    echo "  --dry-run   Show what would change without making changes"
    echo "  --skip-db   Skip database sync step"
    echo "  --help      Show this help message"
}

for arg in "$@"; do
    case "$arg" in
        --dry-run)  DRY_RUN="--dry-run" ;;
        --skip-db)  SKIP_DB=true ;;
        --help)     usage; exit 0 ;;
        *)          echo "Unknown option: $arg"; usage; exit 1 ;;
    esac
done

echo "========================================="
echo "  AEI Scheduler Sync from Remote"
echo "  $(date)"
echo "========================================="
echo ""

# Step 1: Sync remote -> backup mirror
echo "=== Step 1: Remote -> Backup Mirror ==="
if [ -f "$SYNC_SCRIPT" ]; then
    if [ -n "$DRY_RUN" ]; then
        echo "(dry run)"
        "$SYNC_SCRIPT" --dry-run
    else
        "$SYNC_SCRIPT"
    fi
else
    echo "ERROR: Sync script not found at $SYNC_SCRIPT"
    echo "You can sync manually with:"
    echo "  rsync -avz -e 'ssh -i /tmp/aei_key' \\"
    echo "    --exclude='s3uploads/' --exclude='pdftemp/' --exclude='temppdf/' \\"
    echo "    --exclude='mail_sent/' --exclude='uploads/' --exclude='*.log' \\"
    echo "    Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/ \\"
    echo "    $BACKUP_MIRROR/"
    exit 1
fi
echo ""

# Step 2: Copy backup mirror -> local dev
echo "=== Step 2: Backup Mirror -> Local Dev ==="
echo "PROTECTED: database.php and config.php will NOT be overwritten"
echo ""

RSYNC_CMD="rsync -avz $DRY_RUN --stats \
  --exclude='system/application/config/database.php' \
  --exclude='system/application/config/config.php' \
  --exclude='.sync.log' \
  '$BACKUP_MIRROR/' \
  '$LOCAL_DEV/'"

if [ -n "$DRY_RUN" ]; then
    echo "(dry run)"
fi

eval $RSYNC_CMD
echo ""

# Step 3: Database sync (optional)
if [ "$SKIP_DB" = true ]; then
    echo "=== Step 3: Database Sync (SKIPPED) ==="
else
    echo "=== Step 3: Database Sync ==="
    echo "Latest weekly backups:"
    ls -lht /mnt/e/remote_scheduler_backup/database/full_backup_*.sql.gz 2>/dev/null | head -3
    echo ""
    echo "To import the latest backup:"
    echo "  zcat /mnt/e/remote_scheduler_backup/database/full_backup_YYYY-MM-DD.sql.gz | mysql -u root -pscheduler mandhdesign_schedular"
    echo ""
    echo "To pull a fresh dump (takes ~40 min):"
    echo "  See CONNECTORS/DATABASE/README.md or LOCAL_SYNC/README.md"
fi

echo ""
echo "========================================="
if [ -n "$DRY_RUN" ]; then
    echo "  DRY RUN COMPLETE - no changes made"
else
    echo "  SYNC COMPLETE"
fi
echo "========================================="
echo ""
echo "Verify local site: http://aeihawaii.com/scheduler/"
