#!/bin/bash
# deploy_remote.sh - Deploy file(s) to remote production server
#
# Usage: ./deploy_remote.sh <file_path> [file_path2 ...]
#   file_path: relative path within the scheduler application
#              e.g., system/application/controllers/acschedule.php
#
# Process:
#   1. Creates backup on remote (.bak.YYYYMMDD)
#   2. Uploads to /tmp via SCP
#   3. Copies to destination
#   4. Restores ec2-user:ec2-user ownership
#   5. Verifies deployment

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
SSH_KEY="/tmp/aei_key"
REMOTE_USER="Julian"
REMOTE_HOST="18.225.0.90"
REMOTE_SCHEDULER="/var/www/vhosts/aeihawaii.com/httpdocs/scheduler"
LOCAL_DEV="/var/www/html/dev_scheduler/SCHEDULER"
DATE_STAMP=$(date +%Y%m%d)

usage() {
    echo "Usage: $0 <file_path> [file_path2 ...]"
    echo ""
    echo "Deploy file(s) to the remote production server."
    echo ""
    echo "Arguments:"
    echo "  file_path   Relative path within the scheduler application"
    echo "              e.g., system/application/controllers/acschedule.php"
    echo ""
    echo "Options:"
    echo "  --from DIR  Source directory (default: $LOCAL_DEV)"
    echo "  --help      Show this help message"
    echo ""
    echo "Protected files (will NOT be deployed):"
    echo "  system/application/config/database.php"
    echo "  system/application/config/config.php"
}

if [ $# -eq 0 ] || [ "$1" = "--help" ]; then
    usage
    exit 0
fi

# Ensure SSH key exists
if [ ! -f "$SSH_KEY" ]; then
    echo "SSH key not found at $SSH_KEY. Running ssh_connect.sh to convert..."
    "$SCRIPT_DIR/ssh_connect.sh" "echo 'Key ready'"
fi

# Parse --from option
SOURCE_DIR="$LOCAL_DEV"
if [ "$1" = "--from" ]; then
    SOURCE_DIR="$2"
    shift 2
fi

PROTECTED_FILES=(
    "system/application/config/database.php"
    "system/application/config/config.php"
)

SSH_CMD="ssh -i $SSH_KEY -o StrictHostKeyChecking=no"
SCP_CMD="scp -i $SSH_KEY -o StrictHostKeyChecking=no"

deploy_file() {
    local file_path="$1"
    local source="$SOURCE_DIR/$file_path"
    local remote_target="$REMOTE_SCHEDULER/$file_path"
    local tmp_name="deploy_$(echo "$file_path" | tr '/' '_')"

    # Check for protected files
    for protected in "${PROTECTED_FILES[@]}"; do
        if [ "$file_path" = "$protected" ]; then
            echo "BLOCKED: $file_path is a protected config file. Skipping."
            return 1
        fi
    done

    # Check source exists
    if [ ! -f "$source" ]; then
        echo "ERROR: Source file not found: $source"
        return 1
    fi

    echo "============================================"
    echo "File:   $file_path"
    echo "Source: $source"
    echo "Target: $REMOTE_USER@$REMOTE_HOST:$remote_target"
    echo "============================================"

    # Show local file info
    echo "Local file:"
    ls -la "$source"
    echo ""

    # Show remote file info (if exists)
    echo "Remote file:"
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "ls -la $remote_target 2>/dev/null || echo '(does not exist)'"
    echo ""

    # Confirm
    read -p "Deploy $file_path to PRODUCTION? This affects live users. [y/N] " confirm
    if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
        echo "Skipped: $file_path"
        return 0
    fi

    # Step 1: Create backup on remote
    echo "1. Creating remote backup..."
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
        "if [ -f '$remote_target' ]; then cp '$remote_target' '${remote_target}.bak.${DATE_STAMP}'; echo 'Backup created'; else echo 'No existing file to backup'; fi"

    # Step 2: Upload to /tmp
    echo "2. Uploading to remote /tmp..."
    $SCP_CMD "$source" "$REMOTE_USER@$REMOTE_HOST:/tmp/$tmp_name"

    # Step 3: Copy to destination
    echo "3. Deploying to destination..."
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
        "mkdir -p \$(dirname '$remote_target') && cp '/tmp/$tmp_name' '$remote_target'"

    # Step 4: Restore ownership
    echo "4. Restoring file ownership..."
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
        "sudo chown ec2-user:ec2-user '$remote_target' '${remote_target}.bak.${DATE_STAMP}' 2>/dev/null; echo 'Ownership restored'"

    # Step 5: Verify
    echo "5. Verifying deployment..."
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "ls -la '$remote_target'"

    # Step 6: Cleanup /tmp
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "rm -f '/tmp/$tmp_name'"

    echo "DEPLOYED: $file_path"
    echo ""
}

echo "========================================="
echo "  REMOTE DEPLOYMENT"
echo "  Target: $REMOTE_HOST"
echo "  Date:   $(date)"
echo "========================================="
echo ""

for file in "$@"; do
    deploy_file "$file"
done

echo "Done. Verify at https://aeihawaii.com/dev/scheduler/"
