#!/bin/bash
# deploy_local.sh - Deploy file(s) from working directory to local test site
#
# Usage: ./deploy_local.sh <file_path> [file_path2 ...]
#   file_path: relative path within the scheduler (e.g., system/application/controllers/acschedule.php)
#
# The script preserves relative paths, so the file lands in the correct location
# under the local dev site.

set -e

LOCAL_DEV="/var/www/html/dev_scheduler/SCHEDULER"
WORKING_DIR="/var/www/html/dev_scheduler/SCHEDULER"  # Default source

usage() {
    echo "Usage: $0 <file_path> [file_path2 ...]"
    echo ""
    echo "Deploy file(s) to the local test site at $LOCAL_DEV"
    echo ""
    echo "Arguments:"
    echo "  file_path   Relative path within the scheduler application"
    echo "              e.g., system/application/controllers/acschedule.php"
    echo ""
    echo "Options:"
    echo "  --from DIR  Source directory (default: current directory)"
    echo "  --help      Show this help message"
    echo ""
    echo "Protected files (will NOT be deployed):"
    echo "  system/application/config/database.php"
    echo "  system/application/config/config.php"
}

if [ $# -eq 0 ] || [ "$1" = "--help" ]; then
    usage
    exit 0
fi

# Parse --from option
SOURCE_DIR="$(pwd)"
if [ "$1" = "--from" ]; then
    SOURCE_DIR="$2"
    shift 2
fi

PROTECTED_FILES=(
    "system/application/config/database.php"
    "system/application/config/config.php"
)

deploy_file() {
    local file_path="$1"
    local source="$SOURCE_DIR/$file_path"
    local target="$LOCAL_DEV/$file_path"

    # Check for protected files
    for protected in "${PROTECTED_FILES[@]}"; do
        if [ "$file_path" = "$protected" ]; then
            echo "BLOCKED: $file_path is a protected config file. Skipping."
            return 1
        fi
    done

    # Check source exists
    if [ ! -f "$source" ]; then
        echo "ERROR: Source file not found: $source"
        return 1
    fi

    # Show diff if target exists
    if [ -f "$target" ]; then
        echo "--- Changes for $file_path ---"
        diff "$target" "$source" || true
        echo "---"
    else
        echo "NEW FILE: $file_path"
    fi

    # Confirm
    read -p "Deploy $file_path to local dev? [y/N] " confirm
    if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
        echo "Skipped: $file_path"
        return 0
    fi

    # Create target directory if needed
    mkdir -p "$(dirname "$target")"

    # Create backup if target exists
    if [ -f "$target" ]; then
        cp "$target" "${target}.bak.$(date +%Y%m%d%H%M%S)"
    fi

    # Deploy
    cp "$source" "$target"
    echo "DEPLOYED: $file_path"
}

echo "Source:  $SOURCE_DIR"
echo "Target:  $LOCAL_DEV"
echo ""

for file in "$@"; do
    deploy_file "$file"
    echo ""
done

echo "Done. Test at http://aeihawaii.com/scheduler/"
