#!/bin/bash
# deploy_enh033.sh - Bulk deploy ENH-033 (489 files) to production
#
# Usage:
#   deploy_enh033.sh                Run full deployment
#   deploy_enh033.sh --dry-run      Show what would happen, no changes
#   deploy_enh033.sh --dirs-only    Only create/verify directories
#
# What it does:
#   1. Ensures required directories exist on production with correct permissions
#   2. Creates a timestamped backup of all files that will be overwritten
#   3. Transfers all 489 modified files via tar+scp (single transfer)
#   4. Restores ownership (ec2-user:ec2-user)
#   5. Deploys cleanup_pdftemp.sh and updates crontab
#   6. Verifies deployment with checksum comparison

set -euo pipefail

# ─── Configuration ────────────────────────────────────────────────────
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
SSH_KEY="/tmp/aei_key"
REMOTE_USER="Julian"
REMOTE_HOST="18.225.0.90"
REMOTE_SCHEDULER="/var/www/vhosts/aeihawaii.com/httpdocs/scheduler"
LOCAL_DEV="/var/www/html/dev_scheduler/SCHEDULER"
ENH_DIR="/var/www/html/dev_scheduler/ENHANCEMENTS/ENH-033_dev_scheduler_cleanup"
MODIFIED_DIR="$ENH_DIR/modified"
DATE_STAMP=$(date +%Y%m%d_%H%M)

SSH_CMD="ssh -i $SSH_KEY -o StrictHostKeyChecking=no -o ConnectTimeout=10"
SCP_CMD="scp -i $SSH_KEY -o StrictHostKeyChecking=no -o ConnectTimeout=10"

DRY_RUN=false
DIRS_ONLY=false

# ─── Colors ──────────────────────────────────────────────────────────
if [ -t 1 ]; then
    RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'
    BLUE='\033[0;34m'; BOLD='\033[1m'; NC='\033[0m'
else
    RED=''; GREEN=''; YELLOW=''; BLUE=''; BOLD=''; NC=''
fi

info()   { echo -e "${BLUE}[INFO]${NC}  $*"; }
ok()     { echo -e "${GREEN}[OK]${NC}    $*"; }
warn()   { echo -e "${YELLOW}[WARN]${NC}  $*"; }
err()    { echo -e "${RED}[ERROR]${NC} $*"; }
header() { echo -e "\n${BOLD}═══ $* ═══${NC}"; }

# ─── Parse args ──────────────────────────────────────────────────────
while [[ $# -gt 0 ]]; do
    case "$1" in
        --dry-run)  DRY_RUN=true; shift ;;
        --dirs-only) DIRS_ONLY=true; shift ;;
        --help)
            echo "Usage: deploy_enh033.sh [--dry-run] [--dirs-only]"
            echo "  --dry-run    Show what would happen without making changes"
            echo "  --dirs-only  Only create/verify directories on production"
            exit 0
            ;;
        *) err "Unknown option: $1"; exit 1 ;;
    esac
done

# ─── Pre-flight ──────────────────────────────────────────────────────
header "ENH-033 PRODUCTION DEPLOYMENT"
info "Modified files: $(find "$MODIFIED_DIR" -type f | wc -l)"
info "Date: $(date)"
info "Dry run: $DRY_RUN"
echo ""

# Verify modified directory exists
if [ ! -d "$MODIFIED_DIR" ]; then
    err "Modified directory not found: $MODIFIED_DIR"
    exit 1
fi

# Ensure SSH key
if [ ! -f "$SSH_KEY" ]; then
    info "SSH key not found, converting from PPK..."
    if [ -f "$SCRIPT_DIR/ssh_connect.sh" ]; then
        "$SCRIPT_DIR/ssh_connect.sh" "echo 'Key ready'" 2>&1
    else
        err "No SSH key at $SSH_KEY and no ssh_connect.sh"
        exit 1
    fi
fi

# Test SSH
info "Testing SSH connection..."
if ! $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "echo 'connected'" &>/dev/null; then
    err "Cannot connect to $REMOTE_HOST"
    exit 1
fi
ok "SSH connected"

# ─── Step 1: Create/verify directories on production ─────────────────
header "STEP 1: Ensure directories exist on production"

REQUIRED_DIRS=(
    "pdftemp"
    "temppdf"
    "mail_sent"
    "uploads"
    "logs"
    "scripts"
    "pdfwriting"
)

for dir in "${REQUIRED_DIRS[@]}"; do
    remote_path="$REMOTE_SCHEDULER/$dir"
    if $DRY_RUN; then
        info "[DRY RUN] Would ensure: $remote_path"
    else
        result=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "
            if [ -d '$remote_path' ]; then
                perms=\$(stat -c '%a' '$remote_path' 2>/dev/null || echo 'unknown')
                echo \"EXISTS:\$perms\"
            else
                sudo mkdir -p '$remote_path'
                sudo chmod 777 '$remote_path'
                sudo chown ec2-user:ec2-user '$remote_path'
                echo 'CREATED:777'
            fi
        " 2>/dev/null)
        status=$(echo "$result" | cut -d: -f1)
        perms=$(echo "$result" | cut -d: -f2)
        if [ "$status" = "EXISTS" ]; then
            if [ "$perms" = "777" ] || [ "$perms" = "775" ] || [ "$perms" = "755" ]; then
                ok "$dir/ exists (permissions: $perms)"
            else
                warn "$dir/ exists but permissions are $perms — fixing..."
                $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "sudo chmod 777 '$remote_path'" 2>/dev/null
                ok "$dir/ permissions set to 777"
            fi
        elif [ "$status" = "CREATED" ]; then
            ok "$dir/ created with permissions 777"
        else
            err "Unexpected result for $dir/: $result"
        fi
    fi
done

if $DIRS_ONLY; then
    echo ""
    ok "Directory verification complete (--dirs-only mode)"
    exit 0
fi

# ─── Step 2: Create backup archive on production ─────────────────────
header "STEP 2: Backup current files on production"

BACKUP_NAME="enh033_backup_${DATE_STAMP}.tar.gz"
BACKUP_PATH="/tmp/$BACKUP_NAME"

# Generate the file list (relative to REMOTE_SCHEDULER)
FILELIST_TMP="/tmp/enh033_deploy_files.txt"
cd "$MODIFIED_DIR"
find . -type f | sed 's|^\./||' | sort > "$FILELIST_TMP"
FILE_COUNT=$(wc -l < "$FILELIST_TMP")

info "Files to deploy: $FILE_COUNT"
info "Creating backup on production..."

if $DRY_RUN; then
    info "[DRY RUN] Would create backup of $FILE_COUNT files"
else
    # Upload file list
    $SCP_CMD "$FILELIST_TMP" "$REMOTE_USER@$REMOTE_HOST:/tmp/enh033_filelist.txt"

    # Create backup of existing files on production
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "
        cd '$REMOTE_SCHEDULER'
        # Only backup files that actually exist
        existing_files=\$(while read f; do [ -f \"\$f\" ] && echo \"\$f\"; done < /tmp/enh033_filelist.txt)
        if [ -n \"\$existing_files\" ]; then
            echo \"\$existing_files\" | tar czf '$BACKUP_PATH' -T - 2>/dev/null
            backup_size=\$(du -h '$BACKUP_PATH' | cut -f1)
            backup_count=\$(echo \"\$existing_files\" | wc -l)
            echo \"Backed up \$backup_count files (\$backup_size) to $BACKUP_PATH\"
        else
            echo 'No existing files to backup (all new)'
        fi
    "
    ok "Backup created: $BACKUP_PATH"
fi

# ─── Step 3: Transfer files ─────────────────────────────────────────
header "STEP 3: Transfer modified files to production"

DEPLOY_TAR="/tmp/enh033_deploy_${DATE_STAMP}.tar.gz"

if $DRY_RUN; then
    info "[DRY RUN] Would create tar of $FILE_COUNT files and transfer"
else
    # Create tar from modified directory
    info "Creating deployment archive..."
    cd "$MODIFIED_DIR"
    tar czf "$DEPLOY_TAR" -T "$FILELIST_TMP"
    deploy_size=$(du -h "$DEPLOY_TAR" | cut -f1)
    info "Archive size: $deploy_size"

    # Upload
    info "Uploading to production..."
    $SCP_CMD "$DEPLOY_TAR" "$REMOTE_USER@$REMOTE_HOST:/tmp/enh033_deploy.tar.gz"
    ok "Upload complete"

    # Extract on production
    info "Extracting on production..."
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "
        cd '$REMOTE_SCHEDULER'
        sudo tar xzf /tmp/enh033_deploy.tar.gz
        sudo chown -R ec2-user:ec2-user .
        rm -f /tmp/enh033_deploy.tar.gz /tmp/enh033_filelist.txt
    "
    ok "Extracted $FILE_COUNT files to $REMOTE_SCHEDULER"

    # Clean up local temp files
    rm -f "$DEPLOY_TAR" "$FILELIST_TMP"
fi

# ─── Step 4: Verify deployment ──────────────────────────────────────
header "STEP 4: Verify deployment"

if $DRY_RUN; then
    info "[DRY RUN] Would verify checksums and test grep"
else
    # Spot-check a few key files
    info "Spot-checking key files..."
    SPOT_CHECK_FILES=(
        "system/application/controllers/cron.php"
        "system/application/controllers/esign.php"
        "admin.php"
        "scripts/cleanup_pdftemp.sh"
        "assets/js/web.js"
    )

    check_pass=0
    check_fail=0
    for f in "${SPOT_CHECK_FILES[@]}"; do
        local_file="$LOCAL_DEV/$f"
        if [ ! -f "$local_file" ]; then
            warn "Local file not found for spot check: $f"
            continue
        fi
        local_md5=$(md5sum "$local_file" | cut -d' ' -f1)
        remote_md5=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "md5sum '$REMOTE_SCHEDULER/$f' 2>/dev/null | cut -d' ' -f1" 2>/dev/null || echo "missing")

        if [ "$local_md5" = "$remote_md5" ]; then
            ok "Match: $f"
            check_pass=$((check_pass + 1))
        else
            err "MISMATCH: $f (local=$local_md5 remote=$remote_md5)"
            check_fail=$((check_fail + 1))
        fi
    done

    echo ""
    info "Spot check: $check_pass passed, $check_fail failed"

    # Verify no /dev/scheduler/ references remain on production
    info "Checking for remaining /dev/scheduler/ references on production..."
    remaining=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "
        cd '$REMOTE_SCHEDULER'
        grep -rl '/dev/scheduler/' --include='*.php' --include='*.js' --include='*.css' --include='*.sh' . 2>/dev/null | grep -v '_svn' | grep -v '.bak.' | wc -l
    " 2>/dev/null || echo "error")

    if [ "$remaining" = "0" ]; then
        ok "Zero /dev/scheduler/ references on production"
    else
        warn "$remaining files still have /dev/scheduler/ references on production"
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "
            cd '$REMOTE_SCHEDULER'
            grep -rl '/dev/scheduler/' --include='*.php' --include='*.js' --include='*.css' --include='*.sh' . 2>/dev/null | grep -v '_svn' | grep -v '.bak.' | head -10
        " 2>/dev/null
    fi
fi

# ─── Step 5: Update crontab ─────────────────────────────────────────
header "STEP 5: Crontab update"

if $DRY_RUN; then
    info "[DRY RUN] Would check/update crontab for cleanup_pdftemp.sh"
else
    info "Checking current crontab on production..."
    current_cron=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "sudo crontab -l 2>/dev/null || echo '(no crontab)'" 2>/dev/null)

    if echo "$current_cron" | grep -q "cleanup_pdftemp"; then
        if echo "$current_cron" | grep -q "/dev/scheduler/"; then
            info "Found old crontab entry with /dev/scheduler/ path — needs update"
            info "Current: $(echo "$current_cron" | grep cleanup_pdftemp)"
            info "New:     0 2 * * * $REMOTE_SCHEDULER/scripts/cleanup_pdftemp.sh"
            info ""
            info "To update manually on production:"
            info "  sudo crontab -e"
            info "  Change: .../dev/scheduler/scripts/cleanup_pdftemp.sh"
            info "  To:     $REMOTE_SCHEDULER/scripts/cleanup_pdftemp.sh"
        else
            ok "Crontab already uses correct path"
        fi
    else
        info "No cleanup_pdftemp.sh in crontab yet"
        info "To add on production:"
        info "  sudo crontab -e"
        info "  Add: 0 2 * * * $REMOTE_SCHEDULER/scripts/cleanup_pdftemp.sh"
    fi
fi

# ─── Summary ─────────────────────────────────────────────────────────
echo ""
header "DEPLOYMENT COMPLETE"
echo ""
if $DRY_RUN; then
    info "This was a DRY RUN — no changes were made"
else
    ok "$FILE_COUNT files deployed to production"
    ok "Backup saved at: $BACKUP_PATH (on production)"
    echo ""
    info "Post-deployment checklist:"
    echo "  1. Test PDF generation: proposals page → generate PDF"
    echo "  2. Test photo tab: lightbox CSS loads correctly"
    echo "  3. Test esign: document generation works"
    echo "  4. Check error logs: tail /var/log/httpd/error_log"
    echo "  5. Update crontab if flagged above"
    echo ""
    info "To rollback if needed:"
    echo "  ssh Julian@18.225.0.90"
    echo "  cd $REMOTE_SCHEDULER"
    echo "  sudo tar xzf $BACKUP_PATH"
fi
