#!/bin/bash
# aei_deploy.sh - Unified deployment pipeline for AEI Scheduler
#
# Usage:
#   aei_deploy.sh local <file1> [file2 ...]       Deploy + test locally
#   aei_deploy.sh production <file1> [file2 ...]   Deploy to production
#   aei_deploy.sh rollback <file1> [file2 ...]     Rollback production files
#
# Non-interactive by default (works from Claude). Add --confirm for prompts.
#
# Files are specified as relative paths within the scheduler, e.g.:
#   system/application/controllers/admin.php
#   admin.php  (auto-resolved to system/application/controllers/admin.php)

set -euo pipefail

# ─── Configuration ────────────────────────────────────────────────────
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
SSH_KEY="/tmp/aei_key"
REMOTE_USER="Julian"
REMOTE_HOST="18.225.0.90"
REMOTE_SCHEDULER="/var/www/vhosts/aeihawaii.com/httpdocs/scheduler"
LOCAL_DEV="/var/www/html/dev_scheduler/SCHEDULER"
LOCAL_TEST_URL="http://dev.aeihawaii.com/scheduler"
PROD_TEST_URL="https://aeihawaii.com/scheduler"
DATE_STAMP=$(date +%Y%m%d_%H%M)
PHP_BIN="/usr/bin/php5.6"
FPM_ERROR_LOG="/var/log/php5.6-fpm.log"
APACHE_ERROR_LOG="/var/log/apache2/error.log"

SSH_CMD="ssh -i $SSH_KEY -o StrictHostKeyChecking=no -o ConnectTimeout=10"
SCP_CMD="scp -i $SSH_KEY -o StrictHostKeyChecking=no -o ConnectTimeout=10"

PROTECTED_FILES=(
    "system/application/config/database.php"
    "system/application/config/config.php"
    ".htaccess"
)

# PHP 5.3 incompatible patterns
PHP53_PATTERNS=(
    '\[.*\].*='          # [] array syntax (rough check)
    '::class'            # ::class constant
    'yield '             # generators
    '\.\.\.'             # splat operator
    'finally\s*{'        # finally blocks
)

INTERACTIVE=false
VERBOSE=false
ERRORS=0
SUCCESSES=0

# ─── Colors (if terminal) ────────────────────────────────────────────
if [ -t 1 ]; then
    RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'
    BLUE='\033[0;34m'; BOLD='\033[1m'; NC='\033[0m'
else
    RED=''; GREEN=''; YELLOW=''; BLUE=''; BOLD=''; NC=''
fi

# ─── Helpers ──────────────────────────────────────────────────────────
info()  { echo -e "${BLUE}[INFO]${NC}  $*"; }
ok()    { echo -e "${GREEN}[OK]${NC}    $*"; }
warn()  { echo -e "${YELLOW}[WARN]${NC}  $*"; }
err()   { echo -e "${RED}[ERROR]${NC} $*"; }
header() { echo -e "\n${BOLD}═══ $* ═══${NC}"; }

usage() {
    cat <<'EOF'
Usage: aei_deploy.sh <mode> [options] <file1> [file2 ...]

Modes:
  local        Deploy files to local test site and run pre-flight checks
  production   Deploy files to production (run 'local' first!)
  rollback     Rollback production files to most recent backup

Options:
  --confirm    Enable interactive confirmation prompts
  --from DIR   Source directory (default: local dev site)
  --verbose    Show detailed output
  --help       Show this help message

File paths:
  Relative to scheduler root, e.g.:
    system/application/controllers/admin.php
    admin.php  (auto-resolved to system/application/controllers/)

Protected files (always skipped):
  system/application/config/database.php
  system/application/config/config.php
  .htaccess

Examples:
  aei_deploy.sh local admin.php cancel_job.php
  aei_deploy.sh production system/application/controllers/admin.php
  aei_deploy.sh rollback admin.php
EOF
}

# Resolve shorthand file paths to full scheduler-relative paths
resolve_path() {
    local file="$1"

    # Already a full relative path
    if [[ "$file" == system/* ]] || [[ "$file" == assets/* ]] || [[ "$file" == index.php ]]; then
        echo "$file"
        return
    fi

    # Check if it's a controller name (e.g., "admin.php")
    if [[ "$file" == *.php ]]; then
        # Check controllers first (most common)
        local controller_path="system/application/controllers/$file"
        if [ -f "$LOCAL_DEV/$controller_path" ]; then
            echo "$controller_path"
            return
        fi
        # Check config
        local config_path="system/application/config/$file"
        if [ -f "$LOCAL_DEV/$config_path" ]; then
            echo "$config_path"
            return
        fi
        # Check views
        local view_path="system/application/views/$file"
        if [ -f "$LOCAL_DEV/$view_path" ]; then
            echo "$view_path"
            return
        fi
    fi

    # Return as-is if no resolution found
    echo "$file"
}

# Check if file is protected
is_protected() {
    local file="$1"
    for protected in "${PROTECTED_FILES[@]}"; do
        if [ "$file" = "$protected" ]; then
            return 0
        fi
    done
    return 1
}

# PHP syntax check
check_php_syntax() {
    local file="$1"
    if [[ "$file" != *.php ]]; then
        return 0
    fi

    # Use short_open_tag=On since the codebase uses <? short tags everywhere
    if [ -f "$PHP_BIN" ]; then
        if ! "$PHP_BIN" -d short_open_tag=On -l "$file" 2>&1 | grep -q "No syntax errors"; then
            err "PHP syntax error in $file:"
            "$PHP_BIN" -d short_open_tag=On -l "$file" 2>&1
            return 1
        fi
    else
        # Fallback to any available php
        if command -v php &>/dev/null; then
            if ! php -d short_open_tag=On -l "$file" 2>&1 | grep -q "No syntax errors"; then
                err "PHP syntax error in $file:"
                php -d short_open_tag=On -l "$file" 2>&1
                return 1
            fi
        else
            warn "No PHP binary found, skipping syntax check"
        fi
    fi
    return 0
}

# PHP 5.3 compatibility check
check_php53_compat() {
    local file="$1"
    if [[ "$file" != *.php ]]; then
        return 0
    fi

    local issues=0

    # Check for [] array syntax (most common issue)
    # Look for $var = [...] or [...] => or function(...) patterns BUT NOT $array[index]
    if grep -Pn '(?<!\$\w{0,30})\[(?!\d)(?!.*\]).*[,\]]' "$file" 2>/dev/null | grep -v '//' | head -5 | grep -q .; then
        # More precise: look for lines starting arrays with [ or = [
        if grep -Pn '=\s*\[|^\s*\[.*[,\]]|\(\s*\[' "$file" 2>/dev/null | grep -v '//' | grep -v '\$.*\[' | head -5 | grep -q .; then
            warn "Possible PHP 5.3 incompatible [] array syntax in $file:"
            grep -Pn '=\s*\[|^\s*\[.*[,\]]|\(\s*\[' "$file" 2>/dev/null | grep -v '//' | grep -v '\$.*\[' | head -3
            issues=1
        fi
    fi

    # Check for ::class
    if grep -Pn '::class\b' "$file" 2>/dev/null | head -3 | grep -q .; then
        warn "PHP 5.3 incompatible ::class syntax in $file"
        issues=1
    fi

    return $issues
}

# Ensure SSH key is available
ensure_ssh_key() {
    if [ -f "$SSH_KEY" ]; then
        return 0
    fi
    info "SSH key not found at $SSH_KEY, converting from PPK..."
    if [ -f "$SCRIPT_DIR/ssh_connect.sh" ]; then
        "$SCRIPT_DIR/ssh_connect.sh" "echo 'Key ready'" 2>&1
    else
        err "No SSH key and no ssh_connect.sh script found"
        return 1
    fi
}

# Get error log lines after a timestamp
get_new_errors() {
    local log_file="$1"
    local since_time="$2"  # epoch seconds

    if [ ! -f "$log_file" ]; then
        return 0
    fi

    # Get lines from the log that are newer than since_time
    local new_errors
    new_errors=$(awk -v since="$since_time" '
        {
            # Try to extract timestamp from common log formats
            if (match($0, /\[.*\]/)) {
                print $0
            }
        }
    ' "$log_file" 2>/dev/null | tail -10)

    if [ -n "$new_errors" ]; then
        warn "New log entries after deployment:"
        echo "$new_errors"
    fi
}

# ─── LOCAL MODE ───────────────────────────────────────────────────────
deploy_local() {
    local source_dir="$1"
    shift
    local files=("$@")

    header "LOCAL DEPLOYMENT"
    info "Source:  $source_dir"
    info "Target:  $LOCAL_DEV"
    info "Files:   ${#files[@]}"
    info "Date:    $(date)"
    echo ""

    # Record pre-deploy error log position
    local pre_deploy_lines=0
    if [ -f "$APACHE_ERROR_LOG" ]; then
        pre_deploy_lines=$(wc -l < "$APACHE_ERROR_LOG")
    fi

    for file_arg in "${files[@]}"; do
        local file_path
        file_path=$(resolve_path "$file_arg")
        local source="$source_dir/$file_path"
        local target="$LOCAL_DEV/$file_path"

        echo "────────────────────────────────────────"
        info "File: $file_path"

        # Protected file check
        if is_protected "$file_path"; then
            err "BLOCKED - protected config file. Skipping."
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Source exists check
        if [ ! -f "$source" ]; then
            # If source_dir IS local_dev, the file is already in place
            if [ "$source_dir" = "$LOCAL_DEV" ]; then
                if [ -f "$target" ]; then
                    ok "File already in local dev (source=target)"
                    # Still run checks
                    if ! check_php_syntax "$target"; then
                        ERRORS=$((ERRORS + 1))
                        continue
                    fi
                    check_php53_compat "$target" || true
                    SUCCESSES=$((SUCCESSES + 1))
                    continue
                fi
            fi
            err "Source file not found: $source"
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Pre-flight: PHP syntax
        if ! check_php_syntax "$source"; then
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Pre-flight: PHP 5.3 compatibility (warning only, not blocking)
        check_php53_compat "$source" || true

        # Interactive confirmation
        if [ "$INTERACTIVE" = true ]; then
            read -p "Deploy $file_path to local dev? [y/N] " confirm
            if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
                info "Skipped: $file_path"
                continue
            fi
        fi

        # Create target directory if needed
        mkdir -p "$(dirname "$target")"

        # Create backup if target exists and differs
        if [ -f "$target" ]; then
            if ! diff -q "$source" "$target" &>/dev/null; then
                cp "$target" "${target}.bak.${DATE_STAMP}"
                info "Backup: ${target}.bak.${DATE_STAMP}"
            else
                info "File unchanged, no backup needed"
            fi
        fi

        # Deploy (only if source != target)
        if [ "$source" != "$target" ]; then
            cp "$source" "$target"
        fi

        ok "Deployed: $file_path"
        SUCCESSES=$((SUCCESSES + 1))
    done

    # Restart PHP-FPM to clear OPcache
    echo ""
    info "Restarting PHP 5.6 FPM..."
    if sudo systemctl restart php5.6-fpm 2>&1; then
        ok "PHP 5.6 FPM restarted"
    else
        warn "Failed to restart PHP 5.6 FPM (may need manual restart)"
    fi

    # Check for new errors
    if [ -f "$APACHE_ERROR_LOG" ]; then
        local post_deploy_lines
        post_deploy_lines=$(wc -l < "$APACHE_ERROR_LOG")
        if [ "$post_deploy_lines" -gt "$pre_deploy_lines" ]; then
            local new_count=$((post_deploy_lines - pre_deploy_lines))
            warn "$new_count new lines in Apache error log:"
            tail -n "$new_count" "$APACHE_ERROR_LOG" | tail -10
        else
            ok "No new Apache errors"
        fi
    fi

    # Summary
    echo ""
    header "LOCAL DEPLOYMENT SUMMARY"
    echo "  Succeeded: $SUCCESSES"
    echo "  Failed:    $ERRORS"
    echo ""
    info "Test URLs:"
    echo "  ${LOCAL_TEST_URL}/admin"
    echo "  ${LOCAL_TEST_URL}/login"

    if [ "$ERRORS" -gt 0 ]; then
        return 1
    fi
    return 0
}

# ─── PRODUCTION MODE ──────────────────────────────────────────────────
deploy_production() {
    local source_dir="$1"
    shift
    local files=("$@")

    header "PRODUCTION DEPLOYMENT"
    info "Source:  $source_dir"
    info "Target:  $REMOTE_USER@$REMOTE_HOST:$REMOTE_SCHEDULER"
    info "Files:   ${#files[@]}"
    info "Date:    $(date)"
    echo ""

    # Ensure SSH key
    ensure_ssh_key || { err "Cannot connect to production without SSH key"; return 1; }

    # Test SSH connectivity
    info "Testing SSH connection..."
    if ! $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "echo 'connected'" &>/dev/null; then
        err "Cannot connect to $REMOTE_HOST. Check SSH key and network."
        return 1
    fi
    ok "SSH connection verified"

    for file_arg in "${files[@]}"; do
        local file_path
        file_path=$(resolve_path "$file_arg")
        local source="$source_dir/$file_path"
        local remote_target="$REMOTE_SCHEDULER/$file_path"
        local tmp_name="deploy_$(echo "$file_path" | tr '/' '_')"

        echo ""
        echo "────────────────────────────────────────"
        info "File: $file_path"

        # Protected file check
        if is_protected "$file_path"; then
            err "BLOCKED - protected config file. Skipping."
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Source exists
        if [ ! -f "$source" ]; then
            err "Source file not found: $source"
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Pre-flight: PHP syntax
        if ! check_php_syntax "$source"; then
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Pre-flight: PHP 5.3 compatibility
        check_php53_compat "$source" || true

        # Check if local and remote actually differ
        local local_md5 remote_md5
        local_md5=$(md5sum "$source" | cut -d' ' -f1)
        remote_md5=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "md5sum '$remote_target' 2>/dev/null | cut -d' ' -f1" 2>/dev/null || echo "none")

        if [ "$local_md5" = "$remote_md5" ]; then
            info "File is identical on production (md5: $local_md5). Skipping."
            SUCCESSES=$((SUCCESSES + 1))
            continue
        fi

        info "Local  md5: $local_md5"
        info "Remote md5: $remote_md5"

        # Interactive confirmation
        if [ "$INTERACTIVE" = true ]; then
            read -p "Deploy $file_path to PRODUCTION? This affects live users. [y/N] " confirm
            if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
                info "Skipped: $file_path"
                continue
            fi
        fi

        # Step 1: Create backup on remote
        info "1/5 Creating remote backup..."
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
            "if [ -f '$remote_target' ]; then cp '$remote_target' '${remote_target}.bak.${DATE_STAMP}'; echo 'Backup: ${remote_target}.bak.${DATE_STAMP}'; else echo 'No existing file to backup'; fi"

        # Step 2: Upload to /tmp
        info "2/5 Uploading via SCP..."
        $SCP_CMD "$source" "$REMOTE_USER@$REMOTE_HOST:/tmp/$tmp_name"

        # Step 3: Copy to destination
        info "3/5 Deploying to destination..."
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
            "sudo mkdir -p \$(dirname '$remote_target') && sudo cp '/tmp/$tmp_name' '$remote_target'"

        # Step 4: Restore ownership
        info "4/5 Restoring file ownership..."
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
            "sudo chown ec2-user:ec2-user '$remote_target' 2>/dev/null; echo 'Ownership: ec2-user:ec2-user'"

        # Step 5: Verify checksum
        info "5/5 Verifying checksum..."
        remote_md5=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "md5sum '$remote_target'" | cut -d' ' -f1)
        if [ "$local_md5" = "$remote_md5" ]; then
            ok "Checksum MATCH: $local_md5"
        else
            err "Checksum MISMATCH! Local=$local_md5 Remote=$remote_md5"
            ERRORS=$((ERRORS + 1))
            continue
        fi

        # Cleanup /tmp
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "rm -f '/tmp/$tmp_name'"

        ok "Deployed: $file_path"
        SUCCESSES=$((SUCCESSES + 1))
    done

    # Check remote error logs
    echo ""
    info "Checking remote error log..."
    $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "tail -5 /var/log/httpd/error_log 2>/dev/null || echo '(no error log)'"

    # Summary
    echo ""
    header "PRODUCTION DEPLOYMENT SUMMARY"
    echo "  Succeeded: $SUCCESSES"
    echo "  Failed:    $ERRORS"
    echo ""
    info "Test URLs:"
    echo "  ${PROD_TEST_URL}/admin"

    if [ "$ERRORS" -gt 0 ]; then
        return 1
    fi
    return 0
}

# ─── ROLLBACK MODE ────────────────────────────────────────────────────
rollback_production() {
    shift 2>/dev/null || true  # consume source_dir if passed
    local files=("$@")

    header "PRODUCTION ROLLBACK"
    info "Target:  $REMOTE_USER@$REMOTE_HOST:$REMOTE_SCHEDULER"
    info "Files:   ${#files[@]}"
    info "Date:    $(date)"
    echo ""

    # Ensure SSH key
    ensure_ssh_key || { err "Cannot connect to production without SSH key"; return 1; }

    for file_arg in "${files[@]}"; do
        local file_path
        file_path=$(resolve_path "$file_arg")
        local remote_target="$REMOTE_SCHEDULER/$file_path"

        echo "────────────────────────────────────────"
        info "File: $file_path"

        # Find most recent backup
        info "Looking for backups..."
        local backup_file
        backup_file=$($SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
            "ls -t ${remote_target}.bak.* 2>/dev/null | head -1" 2>/dev/null)

        if [ -z "$backup_file" ]; then
            err "No backup found for $file_path"
            ERRORS=$((ERRORS + 1))
            continue
        fi

        info "Found backup: $backup_file"

        # Interactive confirmation
        if [ "$INTERACTIVE" = true ]; then
            read -p "Restore $file_path from backup? [y/N] " confirm
            if [ "$confirm" != "y" ] && [ "$confirm" != "Y" ]; then
                info "Skipped: $file_path"
                continue
            fi
        fi

        # Restore
        info "Restoring from backup..."
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" \
            "cp '$backup_file' '$remote_target' && sudo chown ec2-user:ec2-user '$remote_target'"

        # Verify
        $SSH_CMD "$REMOTE_USER@$REMOTE_HOST" "ls -la '$remote_target'"

        ok "Rolled back: $file_path (from $backup_file)"
        SUCCESSES=$((SUCCESSES + 1))
    done

    # Summary
    echo ""
    header "ROLLBACK SUMMARY"
    echo "  Succeeded: $SUCCESSES"
    echo "  Failed:    $ERRORS"

    if [ "$ERRORS" -gt 0 ]; then
        return 1
    fi
    return 0
}

# ─── Main ─────────────────────────────────────────────────────────────
if [ $# -eq 0 ] || [ "$1" = "--help" ] || [ "$1" = "-h" ]; then
    usage
    exit 0
fi

MODE="$1"
shift

# Parse options
SOURCE_DIR="$LOCAL_DEV"
FILES=()

while [ $# -gt 0 ]; do
    case "$1" in
        --confirm)
            INTERACTIVE=true
            shift
            ;;
        --verbose)
            VERBOSE=true
            shift
            ;;
        --from)
            SOURCE_DIR="$2"
            shift 2
            ;;
        --help|-h)
            usage
            exit 0
            ;;
        -*)
            err "Unknown option: $1"
            usage
            exit 1
            ;;
        *)
            FILES+=("$1")
            shift
            ;;
    esac
done

if [ ${#FILES[@]} -eq 0 ]; then
    err "No files specified"
    usage
    exit 1
fi

case "$MODE" in
    local)
        deploy_local "$SOURCE_DIR" "${FILES[@]}"
        ;;
    production|prod|remote)
        deploy_production "$SOURCE_DIR" "${FILES[@]}"
        ;;
    rollback)
        rollback_production "$SOURCE_DIR" "${FILES[@]}"
        ;;
    *)
        err "Unknown mode: $MODE (use: local, production, rollback)"
        usage
        exit 1
        ;;
esac
