import 'package:flutter/material.dart';

import '../models/upload_queue_item.dart';
import '../services/upload_queue_service.dart';

/// Compact upload queue indicator for use in app bars or banners.
///
/// Shows a badge with the number of pending uploads. Tapping opens a
/// bottom sheet with the full queue list and retry/remove actions.
class UploadStatusWidget extends StatelessWidget {
  const UploadStatusWidget({super.key});

  @override
  Widget build(BuildContext context) {
    return ValueListenableBuilder<int>(
      valueListenable: UploadQueueService.instance.queueCount,
      builder: (context, count, _) {
        if (count == 0) return const SizedBox.shrink();

        return ValueListenableBuilder<bool>(
          valueListenable: UploadQueueService.instance.isProcessing,
          builder: (context, processing, _) {
            return GestureDetector(
              onTap: () => _showQueueSheet(context),
              child: Container(
                padding: const EdgeInsets.symmetric(horizontal: 10, vertical: 5),
                margin: const EdgeInsets.only(right: 8),
                decoration: BoxDecoration(
                  color: processing ? Colors.blue.shade50 : Colors.orange.shade50,
                  borderRadius: BorderRadius.circular(16),
                  border: Border.all(
                    color: processing ? Colors.blue.shade200 : Colors.orange.shade200,
                  ),
                ),
                child: Row(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    if (processing)
                      const SizedBox(
                        width: 12,
                        height: 12,
                        child: CircularProgressIndicator(strokeWidth: 1.5),
                      )
                    else
                      Icon(Icons.cloud_upload_outlined, size: 14,
                          color: Colors.orange.shade700),
                    const SizedBox(width: 4),
                    Text(
                      '$count',
                      style: TextStyle(
                        fontSize: 12,
                        fontWeight: FontWeight.bold,
                        color: processing ? Colors.blue.shade700 : Colors.orange.shade700,
                      ),
                    ),
                  ],
                ),
              ),
            );
          },
        );
      },
    );
  }

  void _showQueueSheet(BuildContext context) {
    showModalBottomSheet(
      context: context,
      isScrollControlled: true,
      builder: (_) => const _QueueBottomSheet(),
    );
  }
}

class _QueueBottomSheet extends StatefulWidget {
  const _QueueBottomSheet();

  @override
  State<_QueueBottomSheet> createState() => _QueueBottomSheetState();
}

class _QueueBottomSheetState extends State<_QueueBottomSheet> {
  List<UploadQueueItem> _items = [];
  bool _loading = true;

  @override
  void initState() {
    super.initState();
    _loadItems();
  }

  Future<void> _loadItems() async {
    final items = await UploadQueueService.instance.getActiveItems();
    if (mounted) setState(() { _items = items; _loading = false; });
  }

  @override
  Widget build(BuildContext context) {
    return DraggableScrollableSheet(
      initialChildSize: 0.45,
      minChildSize: 0.25,
      maxChildSize: 0.7,
      expand: false,
      builder: (context, scrollController) {
        return Column(
          children: [
            // Handle
            Container(
              margin: const EdgeInsets.only(top: 8),
              width: 40,
              height: 4,
              decoration: BoxDecoration(
                color: Colors.grey.shade300,
                borderRadius: BorderRadius.circular(2),
              ),
            ),

            // Header
            Padding(
              padding: const EdgeInsets.fromLTRB(16, 12, 16, 8),
              child: Row(
                mainAxisAlignment: MainAxisAlignment.spaceBetween,
                children: [
                  const Text(
                    'Upload Queue',
                    style: TextStyle(fontSize: 16, fontWeight: FontWeight.bold),
                  ),
                  TextButton.icon(
                    onPressed: () async {
                      await UploadQueueService.instance.retryFailed();
                      _loadItems();
                    },
                    icon: const Icon(Icons.refresh, size: 16),
                    label: const Text('Retry All'),
                  ),
                ],
              ),
            ),

            const Divider(height: 1),

            // List
            Expanded(
              child: _loading
                  ? const Center(child: CircularProgressIndicator())
                  : _items.isEmpty
                      ? const Center(child: Text('No pending uploads'))
                      : ListView.builder(
                          controller: scrollController,
                          itemCount: _items.length,
                          itemBuilder: (context, index) {
                            final item = _items[index];
                            return _QueueItemTile(
                              item: item,
                              onRemove: () async {
                                if (item.id != null) {
                                  await UploadQueueService.instance.removeItem(item.id!);
                                  _loadItems();
                                }
                              },
                            );
                          },
                        ),
            ),
          ],
        );
      },
    );
  }
}

class _QueueItemTile extends StatelessWidget {
  final UploadQueueItem item;
  final VoidCallback onRemove;

  const _QueueItemTile({required this.item, required this.onRemove});

  @override
  Widget build(BuildContext context) {
    final isFailed = item.status == UploadStatus.failed;
    final isUploading = item.status == UploadStatus.uploading;

    return ListTile(
      leading: Icon(
        isFailed
            ? Icons.error_outline
            : isUploading
                ? Icons.cloud_upload
                : Icons.schedule,
        color: isFailed
            ? Colors.red
            : isUploading
                ? Colors.blue
                : Colors.grey,
      ),
      title: Text(
        item.fileName,
        maxLines: 1,
        overflow: TextOverflow.ellipsis,
        style: const TextStyle(fontSize: 13),
      ),
      subtitle: Text(
        isFailed
            ? 'Failed: ${item.errorMessage ?? "Unknown error"}'
            : isUploading
                ? 'Uploading...'
                : 'Queued (Job ${item.jobId})',
        maxLines: 1,
        overflow: TextOverflow.ellipsis,
        style: TextStyle(
          fontSize: 11,
          color: isFailed ? Colors.red.shade700 : Colors.grey,
        ),
      ),
      trailing: isFailed
          ? IconButton(
              icon: const Icon(Icons.close, size: 18),
              onPressed: onRemove,
              tooltip: 'Remove',
            )
          : null,
    );
  }
}
