import 'dart:async';
import 'package:flutter/material.dart';
import '../services/timesheet_service.dart';

/// Compact AppBar chip showing current clock-in status.
///
/// - When clocked in: shows "Clocked In: [Customer] (2h 15m)" chip
/// - When not clocked in: renders SizedBox.shrink() (invisible)
/// - Tap opens bottom sheet with quick clock-out option
class ClockStatusBanner extends StatefulWidget {
  const ClockStatusBanner({Key? key}) : super(key: key);

  @override
  _ClockStatusBannerState createState() => _ClockStatusBannerState();
}

class _ClockStatusBannerState extends State<ClockStatusBanner> {
  final _ts = TimesheetService.instance;
  Timer? _elapsedTimer;
  String _elapsed = '';

  @override
  void initState() {
    super.initState();
    _ts.clockedIn.addListener(_onChanged);
    _ts.activeClockInRaw.addListener(_onChanged);
    _ts.getStatus();
    _startTimer();
  }

  @override
  void dispose() {
    _ts.clockedIn.removeListener(_onChanged);
    _ts.activeClockInRaw.removeListener(_onChanged);
    _elapsedTimer?.cancel();
    super.dispose();
  }

  void _onChanged() {
    if (mounted) {
      setState(() {});
      _startTimer();
    }
  }

  void _startTimer() {
    _elapsedTimer?.cancel();
    if (_ts.clockedIn.value && _ts.activeClockInRaw.value != null) {
      _tick();
      _elapsedTimer = Timer.periodic(const Duration(seconds: 30), (_) => _tick());
    }
  }

  void _tick() {
    final raw = _ts.activeClockInRaw.value;
    if (raw == null) return;
    final start = DateTime.tryParse(raw);
    if (start == null) return;
    final diff = DateTime.now().difference(start);
    final h = diff.inHours;
    final m = diff.inMinutes % 60;
    if (mounted) {
      setState(() => _elapsed = h > 0 ? '${h}h ${m}m' : '${m}m');
    }
  }

  void _showClockOutSheet() {
    final customer = _ts.activeCustomerName.value ?? 'Unknown';
    final since = _ts.activeClockInTime.value ?? '';

    showModalBottomSheet(
      context: context,
      builder: (ctx) => SafeArea(
        child: Padding(
          padding: const EdgeInsets.all(20),
          child: Column(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                'Currently Clocked In',
                style: Theme.of(ctx).textTheme.titleMedium?.copyWith(fontWeight: FontWeight.bold),
              ),
              const SizedBox(height: 8),
              Text('Customer: $customer'),
              Text('Since: $since${_elapsed.isNotEmpty ? " ($_elapsed)" : ""}'),
              Text('Today total: ${_ts.todayTotal.value}'),
              const SizedBox(height: 16),
              // Today's entries
              if (_ts.todayEntries.isNotEmpty) ...[
                Text('Today\'s Entries:', style: Theme.of(ctx).textTheme.titleSmall),
                const SizedBox(height: 4),
                ..._ts.todayEntries.map((e) => Padding(
                      padding: const EdgeInsets.symmetric(vertical: 2),
                      child: Text(
                        '${e["clock_in"]} - ${e["clock_out"] ?? "now"}  '
                        '${e["customer_name"] ?? ""}'
                        '${e["duration"] != null ? "  (${e["duration"]})" : ""}',
                        style: const TextStyle(fontSize: 13),
                      ),
                    )),
                const SizedBox(height: 12),
              ],
              SizedBox(
                width: double.infinity,
                child: ElevatedButton.icon(
                  icon: const Icon(Icons.stop),
                  label: const Text('Clock Out'),
                  style: ElevatedButton.styleFrom(backgroundColor: Colors.red.shade700),
                  onPressed: () async {
                    Navigator.pop(ctx);
                    final result = await _ts.clockOut();
                    if (result != null && result['status'] == 200 && mounted) {
                      ScaffoldMessenger.of(context).showSnackBar(
                        SnackBar(content: Text(result['message'] ?? 'Clocked out')),
                      );
                    }
                  },
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    if (!_ts.clockedIn.value) return const SizedBox.shrink();

    final name = _ts.activeCustomerName.value ?? '';
    // Truncate long names for the chip
    final displayName = name.length > 15 ? '${name.substring(0, 15)}...' : name;

    return Padding(
      padding: const EdgeInsets.only(right: 4),
      child: ActionChip(
        avatar: Icon(Icons.access_time, size: 16, color: Colors.red.shade700),
        label: Text(
          '$displayName${_elapsed.isNotEmpty ? " ($_elapsed)" : ""}',
          style: TextStyle(fontSize: 12, color: Colors.red.shade700, fontWeight: FontWeight.w600),
        ),
        backgroundColor: Colors.red.shade50,
        side: BorderSide(color: Colors.red.shade200),
        onPressed: _showClockOutSheet,
      ),
    );
  }
}
