import 'dart:async';
import 'package:flutter/material.dart';
import '../services/timesheet_service.dart';

/// Job-contextual clock in/out banner.
///
/// Three visual states:
///   1. Not clocked in → green "Clock In for [Customer]"
///   2. Clocked in for THIS job → red "Clocked In — [Customer] since 9:15 AM" + elapsed
///   3. Clocked in for DIFFERENT job → orange "Switch to [Customer]?"
class ClockInButton extends StatefulWidget {
  final int jobId;
  final int customerId;
  final String customerName;

  const ClockInButton({
    Key? key,
    required this.jobId,
    required this.customerId,
    required this.customerName,
  }) : super(key: key);

  @override
  _ClockInButtonState createState() => _ClockInButtonState();
}

class _ClockInButtonState extends State<ClockInButton> {
  final _ts = TimesheetService.instance;
  bool _loading = true;
  bool _actionInProgress = false;
  Timer? _elapsedTimer;
  String _elapsed = '';

  @override
  void initState() {
    super.initState();
    _ts.clockedIn.addListener(_onStatusChange);
    _ts.activeJobId.addListener(_onStatusChange);
    _refresh();
  }

  @override
  void dispose() {
    _ts.clockedIn.removeListener(_onStatusChange);
    _ts.activeJobId.removeListener(_onStatusChange);
    _elapsedTimer?.cancel();
    super.dispose();
  }

  void _onStatusChange() {
    if (mounted) {
      setState(() {});
      _updateElapsedTimer();
    }
  }

  Future<void> _refresh() async {
    await _ts.getStatus();
    if (mounted) {
      setState(() => _loading = false);
      _updateElapsedTimer();
    }
  }

  void _updateElapsedTimer() {
    _elapsedTimer?.cancel();
    if (_ts.clockedIn.value && _ts.activeJobId.value == widget.jobId && _ts.activeClockInRaw.value != null) {
      _tickElapsed();
      _elapsedTimer = Timer.periodic(const Duration(seconds: 30), (_) => _tickElapsed());
    }
  }

  void _tickElapsed() {
    final raw = _ts.activeClockInRaw.value;
    if (raw == null) return;
    final start = DateTime.tryParse(raw);
    if (start == null) return;
    final diff = DateTime.now().difference(start);
    final h = diff.inHours;
    final m = diff.inMinutes % 60;
    if (mounted) {
      setState(() => _elapsed = h > 0 ? '${h}h ${m}m' : '${m}m');
    }
  }

  // ----------------------------------------------------------------
  // Actions
  // ----------------------------------------------------------------

  Future<void> _doClockIn() async {
    setState(() => _actionInProgress = true);
    final result = await _ts.clockIn(
      jobId: widget.jobId,
      customerId: widget.customerId,
    );
    if (mounted) {
      setState(() => _actionInProgress = false);
      if (result != null && result['status'] != 200) {
        _showSnack(result['message'] ?? 'Clock in failed');
      }
    }
  }

  Future<void> _doClockOut() async {
    setState(() => _actionInProgress = true);
    final result = await _ts.clockOut();
    if (mounted) {
      setState(() => _actionInProgress = false);
      if (result != null && result['status'] != 200) {
        _showSnack(result['message'] ?? 'Clock out failed');
      }
    }
  }

  Future<void> _doSwitchJob() async {
    // Confirmation dialog
    final confirmed = await showDialog<bool>(
      context: context,
      builder: (ctx) => AlertDialog(
        title: const Text('Switch Job?'),
        content: Text(
          'Clock out of ${_ts.activeCustomerName.value ?? "current job"} '
          'and clock in for ${widget.customerName}?',
        ),
        actions: [
          TextButton(onPressed: () => Navigator.pop(ctx, false), child: const Text('Cancel')),
          ElevatedButton(onPressed: () => Navigator.pop(ctx, true), child: const Text('Switch')),
        ],
      ),
    );
    if (confirmed != true) return;

    setState(() => _actionInProgress = true);
    final result = await _ts.switchJob(
      jobId: widget.jobId,
      customerId: widget.customerId,
    );
    if (mounted) {
      setState(() => _actionInProgress = false);
      if (result != null && result['status'] != 200) {
        _showSnack(result['message'] ?? 'Switch failed');
      }
    }
  }

  void _showSnack(String msg) {
    ScaffoldMessenger.of(context).showSnackBar(SnackBar(content: Text(msg)));
  }

  // ----------------------------------------------------------------
  // Build
  // ----------------------------------------------------------------

  @override
  Widget build(BuildContext context) {
    if (_loading) {
      return const SizedBox(
        height: 48,
        child: Center(child: SizedBox(width: 20, height: 20, child: CircularProgressIndicator(strokeWidth: 2))),
      );
    }

    final isClockedIn = _ts.clockedIn.value;
    final activeJob = _ts.activeJobId.value;

    // State 1: Not clocked in — green banner
    if (!isClockedIn) {
      return _banner(
        color: Colors.green.shade700,
        icon: Icons.play_arrow,
        label: 'Clock In for ${widget.customerName}',
        onTap: _actionInProgress ? null : _doClockIn,
      );
    }

    // State 2: Clocked in for THIS job — red banner with elapsed
    if (activeJob == widget.jobId) {
      final since = _ts.activeClockInTime.value ?? '';
      return _banner(
        color: Colors.red.shade700,
        icon: Icons.stop,
        label: 'Clocked In — ${widget.customerName} since $since${_elapsed.isNotEmpty ? " ($_elapsed)" : ""}',
        onTap: _actionInProgress ? null : _doClockOut,
      );
    }

    // State 3: Clocked in for DIFFERENT job — orange banner
    return _banner(
      color: Colors.orange.shade800,
      icon: Icons.swap_horiz,
      label: 'At ${_ts.activeCustomerName.value ?? "another job"}. Switch to ${widget.customerName}?',
      onTap: _actionInProgress ? null : _doSwitchJob,
    );
  }

  Widget _banner({
    required Color color,
    required IconData icon,
    required String label,
    VoidCallback? onTap,
  }) {
    return GestureDetector(
      onTap: onTap,
      child: Container(
        width: double.infinity,
        padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
        margin: const EdgeInsets.only(bottom: 8),
        decoration: BoxDecoration(
          color: color,
          borderRadius: BorderRadius.circular(8),
        ),
        child: Row(
          children: [
            if (_actionInProgress)
              const SizedBox(
                width: 20,
                height: 20,
                child: CircularProgressIndicator(strokeWidth: 2, color: Colors.white),
              )
            else
              Icon(icon, color: Colors.white, size: 22),
            const SizedBox(width: 12),
            Expanded(
              child: Text(
                label,
                style: const TextStyle(color: Colors.white, fontWeight: FontWeight.w600, fontSize: 14),
                overflow: TextOverflow.ellipsis,
                maxLines: 2,
              ),
            ),
          ],
        ),
      ),
    );
  }
}
