import 'dart:async';
import 'dart:convert';
import 'dart:io';

import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:flutter/foundation.dart';
import 'package:http/http.dart' as http;
import 'package:path_provider/path_provider.dart';
import 'package:sqflite/sqflite.dart';
import 'package:path/path.dart' as p;

import '../models/upload_queue_item.dart';

/// Persistent foreground upload queue.
///
/// Photos are copied to app storage on enqueue, then uploaded one-at-a-time
/// with retry/backoff. Queue state survives app restart via SQLite.
class UploadQueueService with WidgetsBindingObserver {
  UploadQueueService._();
  static final UploadQueueService instance = UploadQueueService._();

  static const String _uploadUrl = 'https://aeihawaii.com/photoapi/upload.php';
  static const int _maxRetries = 5;

  Database? _db;
  bool _isProcessing = false;
  bool _isPaused = false;
  StreamSubscription? _connectivitySub;

  // --- Public notifiers for UI ---
  final ValueNotifier<int> queueCount = ValueNotifier(0);
  final ValueNotifier<bool> isProcessing = ValueNotifier(false);
  final ValueNotifier<String?> currentFileName = ValueNotifier(null);

  /// Emits jobId when an upload completes — listeners can refresh their grid.
  final StreamController<int> onUploadComplete = StreamController<int>.broadcast();

  /// Emits when a 401 is received and the queue cannot continue.
  final StreamController<void> onAuthRequired = StreamController<void>.broadcast();

  // Token accessor — set by main.dart after login
  String Function()? tokenGetter;

  // ---------- Lifecycle ----------

  Future<void> init() async {
    final dbPath = await getDatabasesPath();
    _db = await openDatabase(
      p.join(dbPath, 'upload_queue.db'),
      version: 1,
      onCreate: (db, version) async {
        await db.execute(UploadQueueItem.createTableSQL);
      },
    );

    // Reset any items stuck in 'uploading' state (from a previous crash)
    await _db!.update(
      'upload_queue',
      {'status': UploadStatus.queued.name},
      where: 'status = ?',
      whereArgs: [UploadStatus.uploading.name],
    );

    await _updateCount();

    // Watch connectivity
    _connectivitySub = Connectivity().onConnectivityChanged.listen((results) {
      final hasConnection = results.any((r) => r != ConnectivityResult.none);
      if (hasConnection && _isPaused) {
        resume();
      } else if (!hasConnection) {
        pause();
      }
    });

    // Observe app lifecycle
    WidgetsBinding.instance.addObserver(this);

    // Start processing any pending items
    _processNext();
  }

  @override
  void didChangeAppLifecycleState(AppLifecycleState state) {
    if (state == AppLifecycleState.paused) {
      pause();
    } else if (state == AppLifecycleState.resumed) {
      resume();
    }
  }

  void pause() {
    _isPaused = true;
  }

  void resume() {
    _isPaused = false;
    if (!_isProcessing) _processNext();
  }

  Future<void> dispose() async {
    _connectivitySub?.cancel();
    WidgetsBinding.instance.removeObserver(this);
    await _db?.close();
    _db = null;
  }

  // ---------- Enqueue ----------

  /// Add a photo to the upload queue. Returns true if added, false if duplicate.
  /// Copies the file to permanent app storage before queuing.
  Future<bool> enqueue({
    required String filePath,
    required int jobId,
    required String fileName,
  }) async {
    if (_db == null) return false;

    final file = File(filePath);
    if (!file.existsSync()) return false;

    // Build dedup key from original file stats
    final stat = file.statSync();
    final dedupKey = '$filePath:${stat.size}:${stat.modified.millisecondsSinceEpoch}';

    // Check if already queued
    final existing = await _db!.query(
      'upload_queue',
      where: 'dedup_key = ?',
      whereArgs: [dedupKey],
      limit: 1,
    );
    if (existing.isNotEmpty) return false;

    // Copy to permanent app storage (picker temp files can be cleaned by OS)
    final appDir = await getApplicationDocumentsDirectory();
    final queueDir = Directory('${appDir.path}/upload_queue');
    if (!queueDir.existsSync()) queueDir.createSync(recursive: true);
    final permanentPath = '${queueDir.path}/${DateTime.now().millisecondsSinceEpoch}_$fileName';
    await file.copy(permanentPath);

    // Insert into SQLite
    final item = UploadQueueItem(
      filePath: permanentPath,
      jobId: jobId,
      fileName: fileName,
      dedupKey: dedupKey,
      createdAt: DateTime.now(),
    );

    await _db!.insert(
      'upload_queue',
      item.toMap(),
      conflictAlgorithm: ConflictAlgorithm.ignore,
    );

    await _updateCount();

    // Start processing if idle
    if (!_isProcessing) _processNext();

    return true;
  }

  // ---------- Queue Processing ----------

  Future<void> _processNext() async {
    if (_db == null || _isProcessing || _isPaused) return;

    // Get next queued item
    final rows = await _db!.query(
      'upload_queue',
      where: 'status = ?',
      whereArgs: [UploadStatus.queued.name],
      orderBy: 'id ASC',
      limit: 1,
    );

    if (rows.isEmpty) {
      _isProcessing = false;
      isProcessing.value = false;
      currentFileName.value = null;
      return;
    }

    _isProcessing = true;
    isProcessing.value = true;

    final item = UploadQueueItem.fromMap(rows.first);
    currentFileName.value = item.fileName;

    // Mark as uploading
    await _db!.update(
      'upload_queue',
      {'status': UploadStatus.uploading.name},
      where: 'id = ?',
      whereArgs: [item.id],
    );

    try {
      // Verify file still exists
      final file = File(item.filePath);
      if (!file.existsSync()) {
        await _markFailed(item, 'File no longer exists');
        _isProcessing = false;
        _processNext();
        return;
      }

      // Base64 encode and upload (one file at a time — controlled RAM)
      final bytes = await file.readAsBytes();
      final base64Image = base64Encode(bytes);

      final token = tokenGetter?.call() ?? '';
      final resp = await http
          .post(
            Uri.parse(_uploadUrl),
            headers: {
              'Content-Type': 'application/json',
              if (token.isNotEmpty) 'authorization': token,
            },
            body: jsonEncode({
              'auth_token': 'aei@89806849',
              'file_name': item.fileName,
              'image_data': base64Image,
              'job_id': item.jobId.toString(),
            }),
          )
          .timeout(const Duration(minutes: 3));

      if (resp.statusCode == 401) {
        // Token expired — pause queue, notify UI
        await _db!.update(
          'upload_queue',
          {'status': UploadStatus.queued.name},
          where: 'id = ?',
          whereArgs: [item.id],
        );
        _isPaused = true;
        _isProcessing = false;
        isProcessing.value = false;
        onAuthRequired.add(null);
        return;
      }

      if (resp.statusCode != 200) {
        throw HttpException('HTTP ${resp.statusCode}');
      }

      final decoded = jsonDecode(resp.body);
      final bool ok = decoded is Map &&
          (decoded['success'] == true || decoded['status'] == 'ok');

      if (!ok) {
        final msg = decoded is Map ? (decoded['message'] ?? 'Unknown').toString() : 'Bad response';
        throw StateError(msg);
      }

      // Success — mark uploaded, delete local copy
      await _db!.update(
        'upload_queue',
        {'status': UploadStatus.uploaded.name},
        where: 'id = ?',
        whereArgs: [item.id],
      );
      try { file.deleteSync(); } catch (_) {}

      await _updateCount();
      onUploadComplete.add(item.jobId);
    } catch (e) {
      await _handleFailure(item, e.toString());
    }

    _isProcessing = false;
    // Continue to next item
    if (!_isPaused) _processNext();
  }

  Future<void> _handleFailure(UploadQueueItem item, String error) async {
    final newRetryCount = item.retryCount + 1;

    if (newRetryCount >= _maxRetries) {
      await _markFailed(item, error);
    } else {
      // Retry with exponential backoff
      await _db!.update(
        'upload_queue',
        {
          'status': UploadStatus.queued.name,
          'retry_count': newRetryCount,
          'error_message': error,
        },
        where: 'id = ?',
        whereArgs: [item.id],
      );

      // Wait before retrying: 2, 4, 8, 16, 32 seconds
      final delay = Duration(seconds: 1 << newRetryCount);
      await Future.delayed(delay);
    }

    await _updateCount();
  }

  Future<void> _markFailed(UploadQueueItem item, String error) async {
    await _db!.update(
      'upload_queue',
      {
        'status': UploadStatus.failed.name,
        'error_message': error,
      },
      where: 'id = ?',
      whereArgs: [item.id],
    );
  }

  // ---------- Public Helpers ----------

  /// Retry all failed items.
  Future<void> retryFailed() async {
    if (_db == null) return;
    await _db!.update(
      'upload_queue',
      {'status': UploadStatus.queued.name, 'retry_count': 0, 'error_message': null},
      where: 'status = ?',
      whereArgs: [UploadStatus.failed.name],
    );
    await _updateCount();
    if (!_isProcessing) _processNext();
  }

  /// Remove a failed item from the queue and delete its local file.
  Future<void> removeItem(int id) async {
    if (_db == null) return;
    final rows = await _db!.query('upload_queue', where: 'id = ?', whereArgs: [id]);
    if (rows.isNotEmpty) {
      final item = UploadQueueItem.fromMap(rows.first);
      try { File(item.filePath).deleteSync(); } catch (_) {}
    }
    await _db!.delete('upload_queue', where: 'id = ?', whereArgs: [id]);
    await _updateCount();
  }

  /// Clear all completed uploads from the database.
  Future<void> clearCompleted() async {
    if (_db == null) return;
    await _db!.delete(
      'upload_queue',
      where: 'status = ?',
      whereArgs: [UploadStatus.uploaded.name],
    );
  }

  /// Get count of items by status.
  Future<int> countByStatus(UploadStatus status) async {
    if (_db == null) return 0;
    final result = await _db!.rawQuery(
      'SELECT COUNT(*) as cnt FROM upload_queue WHERE status = ?',
      [status.name],
    );
    return Sqflite.firstIntValue(result) ?? 0;
  }

  /// Get all items for display (queued + uploading + failed).
  Future<List<UploadQueueItem>> getActiveItems() async {
    if (_db == null) return [];
    final rows = await _db!.query(
      'upload_queue',
      where: 'status IN (?, ?, ?)',
      whereArgs: [
        UploadStatus.queued.name,
        UploadStatus.uploading.name,
        UploadStatus.failed.name,
      ],
      orderBy: 'id ASC',
    );
    return rows.map(UploadQueueItem.fromMap).toList();
  }

  Future<void> _updateCount() async {
    if (_db == null) return;
    final result = await _db!.rawQuery(
      'SELECT COUNT(*) as cnt FROM upload_queue WHERE status IN (?, ?)',
      [UploadStatus.queued.name, UploadStatus.uploading.name],
    );
    queueCount.value = Sqflite.firstIntValue(result) ?? 0;
  }
}
