import 'dart:async';
import 'dart:convert';
import 'package:flutter/foundation.dart';
import 'package:http/http.dart' as http;
import 'package:shared_preferences/shared_preferences.dart';

/// Singleton service for timesheet clock in/out API calls.
/// Mirrors the web timesheet (ENH-032) but authenticates via token header.
class TimesheetService {
  TimesheetService._();
  static final instance = TimesheetService._();

  static const _baseUrl = 'https://aeihawaii.com/photoappsch/loginapinew';

  /// Reactive notifiers — widgets listen to these for live updates.
  final clockedIn = ValueNotifier<bool>(false);
  final activeJobId = ValueNotifier<int?>(null);
  final activeCustomerName = ValueNotifier<String?>(null);
  final activeClockInTime = ValueNotifier<String?>(null);
  final activeClockInRaw = ValueNotifier<String?>(null);
  final todayTotal = ValueNotifier<String>('0h 0m');

  int? _activeEntryId;
  int? _activeCustomerId;
  List<Map<String, dynamic>> _todayEntries = [];

  int? get activeEntryId => _activeEntryId;
  int? get activeCustomerId => _activeCustomerId;
  List<Map<String, dynamic>> get todayEntries => List.unmodifiable(_todayEntries);

  /// Retrieve the stored auth token.
  Future<String> _getToken() async {
    final prefs = await SharedPreferences.getInstance();
    return prefs.getString('token') ?? '';
  }

  Map<String, String> _headers(String token) => {
        'authorization': token,
        'Content-Type': 'application/json',
      };

  // ------------------------------------------------------------------
  // API: Get current clock status
  // ------------------------------------------------------------------

  /// Fetches current status from server and updates all notifiers.
  /// Call this on app start, after actions, and on job detail page init.
  Future<Map<String, dynamic>?> getStatus() async {
    try {
      final token = await _getToken();
      final response = await http
          .get(
            Uri.parse('$_baseUrl/ts_get_status'),
            headers: _headers(token),
          )
          .timeout(const Duration(seconds: 10));

      if (response.statusCode == 200) {
        final data = json.decode(response.body) as Map<String, dynamic>;
        _applyStatus(data);
        return data;
      }
    } catch (e) {
      debugPrint('TimesheetService.getStatus error: $e');
    }
    return null;
  }

  void _applyStatus(Map<String, dynamic> data) {
    final isIn = data['clocked_in'] == true;
    clockedIn.value = isIn;
    activeJobId.value = isIn ? data['job_id'] as int? : null;
    activeCustomerName.value = isIn ? data['customer_name'] as String? : null;
    activeClockInTime.value = isIn ? data['clock_in_time'] as String? : null;
    activeClockInRaw.value = isIn ? data['clock_in_raw'] as String? : null;
    todayTotal.value = (data['today_total'] as String?) ?? '0h 0m';
    _activeEntryId = isIn ? data['entry_id'] as int? : null;
    _activeCustomerId = isIn ? data['customer_id'] as int? : null;
    _todayEntries = (data['today_entries'] as List<dynamic>?)
            ?.cast<Map<String, dynamic>>() ??
        [];
  }

  // ------------------------------------------------------------------
  // API: Clock In
  // ------------------------------------------------------------------

  /// Clock in to a specific job. Returns the API response map or null on error.
  Future<Map<String, dynamic>?> clockIn({
    required int jobId,
    required int customerId,
    String notes = '',
  }) async {
    try {
      final token = await _getToken();
      final response = await http
          .post(
            Uri.parse('$_baseUrl/ts_clock_in'),
            headers: _headers(token),
            body: json.encode({
              'job_id': jobId,
              'customer_id': customerId,
              'notes': notes,
            }),
          )
          .timeout(const Duration(seconds: 10));

      final data = json.decode(response.body) as Map<String, dynamic>;
      if (data['status'] == 200) {
        await getStatus(); // refresh full state
      }
      return data;
    } catch (e) {
      debugPrint('TimesheetService.clockIn error: $e');
      return null;
    }
  }

  // ------------------------------------------------------------------
  // API: Clock Out
  // ------------------------------------------------------------------

  /// Clock out of the current active entry.
  Future<Map<String, dynamic>?> clockOut() async {
    try {
      final token = await _getToken();
      final response = await http
          .post(
            Uri.parse('$_baseUrl/ts_clock_out'),
            headers: _headers(token),
            body: '{}',
          )
          .timeout(const Duration(seconds: 10));

      final data = json.decode(response.body) as Map<String, dynamic>;
      if (data['status'] == 200) {
        await getStatus(); // refresh full state
      }
      return data;
    } catch (e) {
      debugPrint('TimesheetService.clockOut error: $e');
      return null;
    }
  }

  // ------------------------------------------------------------------
  // API: Switch Job (atomic clock-out + clock-in)
  // ------------------------------------------------------------------

  /// Atomically switch from current job to a new job.
  Future<Map<String, dynamic>?> switchJob({
    required int jobId,
    required int customerId,
    String notes = '',
  }) async {
    try {
      final token = await _getToken();
      final response = await http
          .post(
            Uri.parse('$_baseUrl/ts_switch_job'),
            headers: _headers(token),
            body: json.encode({
              'job_id': jobId,
              'customer_id': customerId,
              'notes': notes,
            }),
          )
          .timeout(const Duration(seconds: 10));

      final data = json.decode(response.body) as Map<String, dynamic>;
      if (data['status'] == 200) {
        await getStatus(); // refresh full state
      }
      return data;
    } catch (e) {
      debugPrint('TimesheetService.switchJob error: $e');
      return null;
    }
  }
}
