import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import 'package:shared_preferences/shared_preferences.dart';
import 'customer_details_full_view.dart';
import 'package:provider/provider.dart';
import '../token_provider.dart';
// Import the JobDetailsPage
import 'package:intl/intl.dart'; // For formatting dates
import 'job_details_full_view.dart';

class TodayJobPage extends StatefulWidget {
  const TodayJobPage({super.key});

  @override
  _TodayJobPageState createState() => _TodayJobPageState();
}

class _TodayJobPageState extends State<TodayJobPage> {
  List<dynamic> todayJobs = [];
  bool isLoading = true;

  @override
  void initState() {
    super.initState();
    fetchTodayJobs();
  }

  Future<void> fetchTodayJobs() async {
    final token = Provider.of<TokenProvider>(context, listen: false).token;
    SharedPreferences prefs = await SharedPreferences.getInstance();
    String? userId = prefs.getString('user_id');

    try {
      final response = await http.get(
        Uri.parse(
            'https://aeihawaii.com/photoappsch/loginapinew/getjob/$userId'),
        headers: {
          'Content-Type': 'application/json',
          'authorization': token,
        },
      );

      if (response.statusCode == 200) {
        setState(() {
          todayJobs = jsonDecode(response.body);
          isLoading = false;
        });
      } else {
        throw Exception('Failed to load today jobs');
      }
    } catch (e) {
      setState(() {
        isLoading = false;
      });
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('No Jobs Today')),
      );
    }
  }

  // ✅ Group jobs by formatted date
  Map<String, List<Map<String, dynamic>>> groupJobsByDate(List<dynamic> jobs) {
    final groupedJobs = <String, List<Map<String, dynamic>>>{};
    for (var job in jobs) {
      String formattedDate = 'No Date';
      if (job['jobdate'] != null) {
        try {
          final timestamp = int.tryParse(job['jobdate'].toString()) ?? 0;
          final date = DateTime.fromMillisecondsSinceEpoch(
            timestamp < 10000000000 ? timestamp * 1000 : timestamp,
          );
          formattedDate = DateFormat('MMMM d, yyyy').format(date);
        } catch (e) {
          formattedDate = 'Invalid Date';
        }
      }
      groupedJobs.putIfAbsent(formattedDate, () => []);
      groupedJobs[formattedDate]!.add(job as Map<String, dynamic>);
    }
    return groupedJobs;
  }

  @override
  Widget build(BuildContext context) {
    if (isLoading) {
      return Center(child: CircularProgressIndicator());
    }

    if (todayJobs.isEmpty) {
      return Center(
        child:
            Text('No jobs available for today', style: TextStyle(fontSize: 18)),
      );
    }

    final groupedJobs = groupJobsByDate(todayJobs);
    final dateKeys = groupedJobs.keys.toList();

    return ListView.builder(
      itemCount: dateKeys.length,
      itemBuilder: (context, index) {
        final date = dateKeys[index];
        final dateJobs = groupedJobs[date]!;

        return Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            // Date Header
            Padding(
              padding:
                  const EdgeInsets.symmetric(vertical: 8.0, horizontal: 16),
              child: Text(
                date,
                style:
                    const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
              ),
            ),

            // Jobs for that date
            Column(
              children: dateJobs.asMap().entries.map((entry) {
                final job = entry.value;
                final colors = [
                  Colors.orange.shade100,
                  Colors.blue.shade100,
                  Colors.purple.shade100,
                  Colors.green.shade100,
                  Colors.yellow.shade100,
                  Colors.pink.shade100,
                ];
                // final bgColor = colors[entry.key % colors.length];
                Color toColor(dynamic v) {
                  if (v == null) return const Color(0x00000000);
                  if (v is int) return Color(v); // JSON int
                  if (v is num) return Color(v.toInt()); // JSON num
                  if (v is String) {
                    // e.g. "#5BC4D4" or "0xFF5BC4D4"
                    var s = v.trim().toUpperCase().replaceAll('#', '');
                    if (s.startsWith('0X')) s = s.substring(2);
                    if (s.length == 6) s = 'FF$s'; // add alpha if missing
                    return Color(int.parse(s, radix: 16));
                  }
                  throw ArgumentError(
                      'Unsupported color: $v (${v.runtimeType})');
                }

// Usage
                final Color bgColor = toColor(job['color']);

                return GestureDetector(
                  onTap: () {
                    final cid = int.tryParse(job['customerId'] ?? '');
                    final jid = int.tryParse(job['id'] ?? '');
                    if (cid != null && jid != null) {
                      Navigator.push(
                        context,
                        MaterialPageRoute(
                          builder: (context) => JobDetailsFullView(jobId: jid),
                        ),
                      );
                    }
                  },
                  child: Container(
                    color: bgColor,
                    padding: const EdgeInsets.symmetric(
                        vertical: 12, horizontal: 16),
                    margin: const EdgeInsets.symmetric(vertical: 2),
                    child: Row(
                      mainAxisAlignment: MainAxisAlignment.spaceBetween,
                      children: [
                        Expanded(
                            flex: 3,
                            child: Text(job['custname'] ?? '',
                                overflow: TextOverflow.ellipsis)),
                        Expanded(
                            flex: 2,
                            child: Text(job['address'] ?? '',
                                overflow: TextOverflow.ellipsis)),
                        Expanded(flex: 1, child: Text(job['type'] ?? '')),
                      ],
                    ),
                  ),
                );
              }).toList(),
            ),
          ],
        );
      },
    );
  }
}
