import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:intl/intl.dart';
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';

import '../token_provider.dart';
import 'job_details_full_view.dart';

/// =========================
/// API ENDPOINTS
/// =========================
const String TECHNICIANS_URL =
    "https://aeihawaii.com/photoappsch/loginapinew/technicians";

/// We will use SAME style as your JobPage:
/// https://aeihawaii.com/photoappsch/loginapinew/getmysearchjob/{userId}/{date}
String jobsByUserAndDateUrl(String userId, String date) =>
    "https://aeihawaii.com/photoappsch/loginapinew/getmysearchjobtech/$userId/$date";

class TechniciansTab extends StatefulWidget {
  const TechniciansTab({super.key});

  @override
  State<TechniciansTab> createState() => _TechniciansTabState();
}

class _TechniciansTabState extends State<TechniciansTab> {
  late Future<List<Technician>> _future;
  List<Technician> _all = [];
  List<Technician> _filtered = [];
  final TextEditingController _search = TextEditingController();

  @override
  void initState() {
    super.initState();
    _future = _loadTechnicians();
    _search.addListener(_applyFilter);
  }

  @override
  void dispose() {
    _search.removeListener(_applyFilter);
    _search.dispose();
    super.dispose();
  }

  void _applyFilter() {
    final q = _search.text.toLowerCase().trim();
    if (q.isEmpty) {
      setState(() => _filtered = List.from(_all));
      return;
    }
    setState(() {
      _filtered = _all
          .where((t) =>
              t.name.toLowerCase().contains(q) ||
              t.email.toLowerCase().contains(q))
          .toList(growable: false);
    });
  }

  Future<bool> _isAdminFromPrefs() async {
    final prefs = await SharedPreferences.getInstance();

    final roleRaw = prefs.getString('role_tech');
    final role = (roleRaw ?? '').trim().toLowerCase();
    return role == 'admin' || role == 'techm'; // show for admin/sales
  }

  Future<List<Technician>> _loadTechnicians() async {
    // extra safety
    final allowed = await _isAdminFromPrefs();
    if (!allowed) {
      throw Exception("Not authorized (admin/sales only).");
    }

    final token = Provider.of<TokenProvider>(context, listen: false).token;
    if (token.trim().isEmpty) {
      throw Exception("Token not found. Please login again.");
    }

    final res = await http.get(
      Uri.parse(TECHNICIANS_URL),
      headers: {
        "authorization": token, // IMPORTANT: no Bearer
        "accept": "application/json",
      },
    );

    if (res.statusCode == 403) throw Exception("Not authorized (403).");
    if (res.statusCode != 200) {
      throw Exception("Technicians API failed: ${res.statusCode}");
    }

    final decoded = jsonDecode(res.body);
    final list = _extractList(decoded);

    final techs = list.map((e) => Technician.fromJson(e)).toList();

    techs.sort((a, b) => a.name.toLowerCase().compareTo(b.name.toLowerCase()));

    _all = techs;
    _filtered = List.from(_all);

    return techs;
  }

  List<dynamic> _extractList(dynamic decoded) {
    if (decoded is List) return decoded;
    if (decoded is Map<String, dynamic>) {
      final data = decoded["data"];
      if (data is List) return data;

      final result = decoded["result"];
      if (result is List) return result;

      final items = decoded["items"];
      if (items is List) return items;
    }
    throw Exception("Unexpected technicians response format.");
  }

  Future<void> _refresh() async {
    setState(() {
      _future = _loadTechnicians();
    });
    await _future;
  }

  @override
  Widget build(BuildContext context) {
    return FutureBuilder<List<Technician>>(
      future: _future,
      builder: (context, snap) {
        if (snap.connectionState == ConnectionState.waiting) {
          return const Center(child: CircularProgressIndicator());
        }

        if (snap.hasError) {
          return _ErrorView(
            message: snap.error.toString(),
            onRetry: _refresh,
          );
        }

        final list = _filtered;

        return Column(
          children: [
            Padding(
              padding: const EdgeInsets.fromLTRB(12, 12, 12, 8),
              child: TextField(
                controller: _search,
                decoration: InputDecoration(
                  hintText: "Search technician...",
                  prefixIcon: const Icon(Icons.search),
                  border: OutlineInputBorder(
                    borderRadius: BorderRadius.circular(10),
                  ),
                ),
              ),
            ),
            Expanded(
              child: RefreshIndicator(
                onRefresh: _refresh,
                child: list.isEmpty
                    ? const Center(child: Text("No technicians found"))
                    : ListView.separated(
                        physics: const AlwaysScrollableScrollPhysics(),
                        itemCount: list.length,
                        separatorBuilder: (_, __) => const Divider(height: 1),
                        itemBuilder: (context, i) {
                          final t = list[i];
                          return ListTile(
                            title: Text(t.name.isEmpty ? "—" : t.name),
                            subtitle: Text(
                              t.email.isNotEmpty ? t.email : "ID: ${t.id}",
                            ),
                            trailing: const Icon(Icons.chevron_right),
                            onTap: () {
                              Navigator.push(
                                context,
                                MaterialPageRoute(
                                  builder: (_) => TechnicianJobsLikeJobPage(
                                    technicianId: t.id.toString(),
                                    technicianName: t.name,
                                  ),
                                ),
                              );
                            },
                          );
                        },
                      ),
              ),
            ),
          ],
        );
      },
    );
  }
}

/// ===========================================
/// Jobs screen that behaves like your JobPage
/// ===========================================
class TechnicianJobsLikeJobPage extends StatefulWidget {
  final String technicianId; // using string like your prefs user_id
  final String technicianName;

  const TechnicianJobsLikeJobPage({
    super.key,
    required this.technicianId,
    required this.technicianName,
  });

  @override
  State<TechnicianJobsLikeJobPage> createState() =>
      _TechnicianJobsLikeJobPageState();
}

class _TechnicianJobsLikeJobPageState extends State<TechnicianJobsLikeJobPage> {
  late DateTime _currentDate;
  final TextEditingController _dateController = TextEditingController();

  Map<String, List<Map<String, String>>> groupedJobs = {};
  bool _isLoading = false;

  @override
  void initState() {
    super.initState();
    _currentDate = DateTime.now();
    _dateController.text = DateFormat('yyyy-MM-dd').format(_currentDate);
    _loadJobs();
  }

  Future<void> _loadJobs() async {
    setState(() => _isLoading = true);

    final token = Provider.of<TokenProvider>(context, listen: false).token;
    final formattedDate = DateFormat('yyyy-MM-dd').format(_currentDate);

    final url =
        Uri.parse(jobsByUserAndDateUrl(widget.technicianId, formattedDate));

    try {
      final response = await http.get(
        url,
        headers: {
          'authorization': token,
          'Content-Type': 'application/json',
        },
      );

      if (response.statusCode == 200) {
        final List<dynamic> data = json.decode(response.body);

        Map<String, List<Map<String, String>>> grouped = {};
        for (var job in data) {
          final jobDate = job['job_date'] ?? 'Unknown Date';
          final header = _formatDateForHeader(jobDate);

          grouped.putIfAbsent(header, () => []);
          grouped[header]!.add({
            'name': job['customername']?.toString() ?? '',
            'location': job['joblocation']?.toString() ?? '',
            'code': job['jobcode']?.toString() ?? '',
            'customerId': job['customerid']?.toString() ?? '',
            'Id': job['id']?.toString() ?? '',
            'color': job['color']?.toString() ?? '',
          });
        }

        setState(() {
          groupedJobs = grouped;
        });
      } else {
        // show error in UI
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text("Jobs API error: ${response.statusCode}")),
        );
      }
    } catch (e) {
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text("Error fetching jobs: $e")),
      );
    } finally {
      setState(() => _isLoading = false);
    }
  }

  String _formatDateForHeader(String rawDate) {
    try {
      final date = DateTime.parse(rawDate);
      return DateFormat.yMMMMd().format(date);
    } catch (_) {
      return rawDate;
    }
  }

  Future<void> _pickDate() async {
    final picked = await showDatePicker(
      context: context,
      initialDate: _currentDate,
      firstDate: DateTime(2020),
      lastDate: DateTime(2100),
    );

    if (picked != null && picked != _currentDate) {
      setState(() {
        _currentDate = picked;
        _dateController.text = DateFormat('yyyy-MM-dd').format(_currentDate);
      });
      await _loadJobs();
    }
  }

  // ===== color helpers (same as your JobPage) =====
  Color parseColorSafe(dynamic v, {Color fallback = const Color(0xFFE0E0E0)}) {
    try {
      if (v == null) return fallback;

      if (v is int) {
        int argb = v;
        if ((argb >>> 24) == 0x00) {
          argb = (0xFF << 24) | (argb & 0x00FFFFFF);
        }
        return Color(argb);
      }

      if (v is num) {
        int argb = v.toInt();
        if ((argb >>> 24) == 0x00) {
          argb = (0xFF << 24) | (argb & 0x00FFFFFF);
        }
        return Color(argb);
      }

      if (v is String) {
        var s = v.trim();

        if (RegExp(r'^\d+$').hasMatch(s)) {
          int argb = int.parse(s);
          if ((argb >>> 24) == 0x00) {
            argb = (0xFF << 24) | (argb & 0x00FFFFFF);
          }
          return Color(argb);
        }

        s = s.toUpperCase().replaceAll('#', '');
        if (s.startsWith('0X')) s = s.substring(2);

        if (s.length == 6) {
          s = 'FF$s';
        } else if (s.length != 8) {
          return fallback;
        }

        final val = int.parse(s, radix: 16);
        final argb =
            ((val >>> 24) == 0x00) ? ((0xFF << 24) | (val & 0x00FFFFFF)) : val;
        return Color(argb);
      }

      return fallback;
    } catch (_) {
      return fallback;
    }
  }

  Color bestTextColorFor(Color bg) {
    final r = bg.red / 255.0;
    final g = bg.green / 255.0;
    final b = bg.blue / 255.0;
    final l = 0.2126 * r + 0.7152 * g + 0.0722 * b;
    return (l > 0.6) ? Colors.black : Colors.white;
  }

  Widget _buildJobRow(Map<String, String> job) {
  final Color bgColor = parseColorSafe(job['color']);
  final Color fg = bestTextColorFor(bgColor);

  return InkWell(
    onTap: () {
      final jid = int.tryParse(job['Id'] ?? '');
      if (jid != null) {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (context) => JobDetailsFullView(jobId: jid),
          ),
        );
      }
    },
    child: Container(
      color: bgColor,
      padding: const EdgeInsets.symmetric(vertical: 12, horizontal: 16),
      margin: const EdgeInsets.symmetric(vertical: 2),
      child: Row(
        children: [
          Expanded(
            flex: 3,
            child: Text(
              job['name'] ?? "",
              style: TextStyle(color: fg),
              overflow: TextOverflow.ellipsis,
            ),
          ),
          Expanded(
            flex: 2,
            child: Text(
              job['location'] ?? "",
              style: TextStyle(color: fg),
              overflow: TextOverflow.ellipsis,
            ),
          ),
          Expanded(
            flex: 1,
            child: Text(
              job['code'] ?? "",
              style: TextStyle(color: fg),
              overflow: TextOverflow.ellipsis,
            ),
          ),
        ],
      ),
    ),
  );
}

  Widget _buildJobSection(String date, List<Map<String, String>> jobs) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Container(
          color: const Color(0xFFEFEFEF),
          width: double.infinity,
          padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
          child: Text(
            date,
            style: const TextStyle(fontWeight: FontWeight.bold),
          ),
        ),
        ...jobs.map(_buildJobRow),
      ],
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        backgroundColor: const Color(0xFF418E1C),
        title: Text(
          widget.technicianName.isEmpty
              ? "Technician Jobs"
              : widget.technicianName,
          style: const TextStyle(
            fontSize: 20,
            fontWeight: FontWeight.bold,
            color: Colors.white,
          ),
          overflow: TextOverflow.ellipsis,
        ),
        iconTheme: const IconThemeData(color: Colors.white),
        actions: [
          IconButton(
            onPressed: _pickDate,
            icon: const Icon(Icons.calendar_today, color: Colors.white),
          ),
        ],
        bottom: PreferredSize(
          preferredSize: const Size.fromHeight(70),
          child: Padding(
            padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
            child: GestureDetector(
              onTap: _pickDate,
              child: AbsorbPointer(
                child: TextField(
                  controller: _dateController,
                  style: const TextStyle(color: Colors.white),
                  decoration: InputDecoration(
                    hintText: 'Tap to select date',
                    hintStyle: const TextStyle(color: Colors.white54),
                    filled: true,
                    fillColor: const Color(0xFF418E1C),
                    prefixIcon: const Icon(Icons.search, color: Colors.white),
                    contentPadding: const EdgeInsets.symmetric(
                      horizontal: 12,
                      vertical: 10,
                    ),
                    border: OutlineInputBorder(
                      borderRadius: BorderRadius.circular(8),
                    ),
                  ),
                ),
              ),
            ),
          ),
        ),
      ),
      body: _isLoading
          ? const Center(child: CircularProgressIndicator())
          : groupedJobs.isEmpty
              ? const Center(child: Text("No jobs found."))
              : ListView(
                  children: groupedJobs.entries
                      .map((entry) => _buildJobSection(entry.key, entry.value))
                      .toList(),
                ),
    );
  }
}

/// ======================
/// Technician model
/// ======================
class Technician {
  final int id;
  final String name;
  final String email;

  Technician({
    required this.id,
    required this.name,
    required this.email,
  });

  factory Technician.fromJson(dynamic json) {
    final m = (json is Map<String, dynamic>) ? json : <String, dynamic>{};

    final idVal = m["id"] ?? m["techid"] ?? m["userid"] ?? m["user_id"];
    final nameVal = m["name"] ??
        m["fullname"] ??
        m["techname"] ??
        m["technician_name"] ??
        "";
    final emailVal = m["email"] ?? "";

    return Technician(
      id: _toInt(idVal),
      name: nameVal.toString(),
      email: emailVal.toString(),
    );
  }
}

int _toInt(dynamic v) {
  if (v == null) return 0;
  if (v is int) return v;
  if (v is double) return v.toInt();
  final s = v.toString().trim();
  return int.tryParse(s) ?? 0;
}

/// ======================
/// Small error widget
/// ======================
class _ErrorView extends StatelessWidget {
  final String message;
  final Future<void> Function() onRetry;

  const _ErrorView({
    required this.message,
    required this.onRetry,
  });

  @override
  Widget build(BuildContext context) {
    return Center(
      child: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            Text(message, textAlign: TextAlign.center),
            const SizedBox(height: 12),
            ElevatedButton(
              onPressed: () async => await onRetry(),
              child: const Text("Retry"),
            ),
          ],
        ),
      ),
    );
  }
}
