// lib/pages/job_details_tabs_shell.dart
import 'package:flutter/material.dart';

// Reuse your existing pages:
import 'job_details_full_view.dart';
import 'job_notes_full_view.dart';
import 'job_photos_full_view.dart';
import 'job_docs_full_view.dart';

enum JobShellInitialTab { details, notes, photos, docs }

class JobDetailsTabsShell extends StatefulWidget {
  final int jobId;
  final int? customerId;
  final String customerName;

  /// Default opens Photos to match your “open from customer details” flow.
  final JobShellInitialTab initialTab;

  const JobDetailsTabsShell({
    super.key,
    required this.jobId,
    required this.customerName,
    this.customerId,
    this.initialTab = JobShellInitialTab.photos,
  });

  @override
  State<JobDetailsTabsShell> createState() => _JobDetailsTabsShellState();
}

class _JobDetailsTabsShellState extends State<JobDetailsTabsShell>
    with SingleTickerProviderStateMixin {
  late final TabController _tabController;

  int _indexFor(JobShellInitialTab t) {
    switch (t) {
      case JobShellInitialTab.details:
        return 0;
      case JobShellInitialTab.notes:
        return 1;
      case JobShellInitialTab.photos:
        return 2;
      case JobShellInitialTab.docs:
        return 3;
    }
  }

  @override
  void initState() {
    super.initState();
    _tabController = TabController(
      length: 4,
      vsync: this,
      initialIndex: _indexFor(widget.initialTab),
    );
  }

  @override
  void dispose() {
    _tabController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      // Green AppBar like your screenshot
      appBar: AppBar(
        backgroundColor: Colors.green,
        title: const Text('Job Details'),
        actions: [
          IconButton(
            icon: const Icon(Icons.help_outline),
            onPressed: () {},
          ),
        ],
      ),

      body: Column(
        children: [
          // White header with customer name (top band under the green AppBar)
          Container(
            color: Colors.white,
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
            child: Row(
              children: [
                const Icon(Icons.person, size: 36, color: Colors.black54),
                const SizedBox(width: 12),
                Expanded(
                  child: Text(
                    widget.customerName,
                    maxLines: 2,
                    overflow: TextOverflow.ellipsis,
                    style: const TextStyle(
                      fontSize: 18,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                ),
              ],
            ),
          ),

          // Grey TabBar strip (details, notes, photos, docs)
          Container(
            color: Colors.grey.shade200,
            child: TabBar(
              controller: _tabController,
              isScrollable: true,
              labelColor: Colors.green, // selected label = green
              unselectedLabelColor: Colors.black54,
              // thin underline like your screenshot; adjust color if you like
              indicator: const UnderlineTabIndicator(
                borderSide: BorderSide(
                    color: Color(0xFF7E57C2), width: 2), // purple-ish
              ),
              tabs: const [
                Tab(text: 'Details'),
                Tab(text: 'Notes'),
                Tab(text: 'Photos'),
                Tab(text: 'Docs'),
              ],
            ),
          ),

          // Tab contents
          Expanded(
            child: TabBarView(
              controller: _tabController,
              children: [
                // 1) Details tab (your existing page)
                JobDetailsFullView(jobId: widget.jobId),

                // 2) Notes tab (pass both IDs as your impl expects)
                JobNotesFullView(
                  customerId: widget.customerId ?? 0,
                  jobId: widget.jobId,
                ),

                // 3) Photos tab (your working file; it renders its own inner header with AE + icons)
                JobPhotosFullView(
                  jobId: widget.jobId,
                  customerName: 'Photos',
                ),

                // 4) Docs tab
                JobDocsFullView(jobId: widget.jobId),
              ],
            ),
          ),
        ],
      ),
    );
  }
}
