import 'dart:async';
import 'dart:convert';
import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:http/http.dart' as http;
import 'package:image_picker/image_picker.dart';
import 'package:provider/provider.dart';
import 'package:flutter_cache_manager/flutter_cache_manager.dart';

import '../models/photo_item.dart';
import '../services/upload_queue_service.dart';
import '../widgets/upload_status_widget.dart';
import 'full_screen_gallery.dart';

import '../token_provider.dart';

// ---- Simple model that keeps a stable cache key (server's fullpath) ----


// ---- App-specific cache manager ----
class AeiImageCache {
  static final instance = CacheManager(
    Config(
      'aei_image_cache',
      stalePeriod: const Duration(days: 30),
      maxNrOfCacheObjects: 1000,
    ),
  );
}

class JobPhotosFullView extends StatefulWidget {
  final int jobId; // required
  final String? customerName; // optional header label
  final String? tokenOverride; // optional if you want to pass a token

  const JobPhotosFullView({
    super.key,
    required this.jobId,
    this.customerName,
    this.tokenOverride,
  });

  @override
  State<JobPhotosFullView> createState() => _JobPhotosFullViewState();
}

class _JobPhotosFullViewState extends State<JobPhotosFullView> {
  // ------ State ------
  List<PhotoItem> _photos = [];
  bool isLoading = true;
  String? _error;

  // Upload UI state
  final ImagePicker _picker = ImagePicker();
  StreamSubscription<int>? _uploadCompleteSub;

  // Selection / delete state
  bool _selectionMode = false;
  final Set<String> _selectedKeys = <String>{};
  bool _isDeleting = false;

  // Memoize cached-file futures so Grid rebuilds don't spam cache manager
  final Map<String, Future<File>> _fileFutures = {};

  // ------ API endpoints ------
  static const String _baseEndpoint =
      'https://aeihawaii.com/photoappsch/loginapinew/getimagelisting';

  // Upload endpoint expects JSON: { auth_token, file_name, image_data(base64), job_id }
  static const String _uploadEndpoint = 'https://aeihawaii.com/photoapi/upload.php';

  // DELETE endpoint (adjust to your actual endpoint if different).
  // Expects JSON: { auth_token, job_id, paths: [<fullpath|cacheKey>...] }
  static const String _deleteEndpoint = 'https://aeihawaii.com/photoapi/delete.php';

  @override
  void initState() {
    super.initState();
    _fetchPhotos();
    // Auto-refresh grid when background queue finishes an upload for this job
    _uploadCompleteSub = UploadQueueService.instance.onUploadComplete.stream.listen((jobId) {
      if (jobId == widget.jobId && mounted) _fetchPhotos();
    });
  }

  @override
  void dispose() {
    _uploadCompleteSub?.cancel();
    super.dispose();
  }

  String _getToken() {
    if (widget.tokenOverride != null && widget.tokenOverride!.isNotEmpty) {
      return widget.tokenOverride!;
    }
    try {
      final prov = context.read<TokenProvider>();
      return prov.token ?? '';
    } catch (_) {
      return '';
    }
  }

  Future<void> _fetchPhotos({int retries = 3}) async {
    if (!mounted) return;
    setState(() {
      isLoading = true;
      _error = null;
    });

    final token = _getToken();
    final uri = Uri.parse('$_baseEndpoint/${widget.jobId}');

    for (int attempt = 0; attempt < retries; attempt++) {
      try {
        final resp = await http.get(
          uri,
          headers: <String, String>{
            'Content-Type': 'application/json',
            if (token.isNotEmpty) 'authorization': token, // raw token
          },
        ).timeout(const Duration(seconds: 25));

        if (resp.statusCode != 200) {
          throw HttpException('HTTP ${resp.statusCode}');
        }

        final decoded = json.decode(resp.body);

        // Accept either a bare array or an object { images: [...] }
        final List images = decoded is List
            ? decoded
            : (decoded is Map && decoded['images'] is List
                ? decoded['images']
                : const []);

        final items = <PhotoItem>[];
        for (final item in images) {
          if (item is Map) {
            final link = (item['link'] ?? '').toString();
            final thumbLink = (item['thumb_link'] ?? '').toString();
            final fullLink = (item['full_link'] ?? '').toString();
            final fullpath = (item['fullpath'] ?? '').toString();
            if (link.isNotEmpty) {
              items.add(PhotoItem(
                url: link,
                thumbUrl: thumbLink,
                fullUrl: fullLink.isNotEmpty ? fullLink : link,
                cacheKey: fullpath.isNotEmpty ? fullpath : link,
              ));
            }
          } else if (item is String) {
            items.add(PhotoItem(url: item, cacheKey: item));
          }
        }

        if (!mounted) return;
        setState(() {
          _photos = items;
          _fileFutures.clear();
          _thumbFutures.clear();
          isLoading = false;
          _error = null;
        });
        return; // success
      } catch (e) {
        if (attempt == retries - 1) {
          if (!mounted) return;
          setState(() {
            _error = 'Failed to load photos: $e';
            isLoading = false;
          });
        } else {
          await Future.delayed(Duration(milliseconds: 350 * (attempt + 1)));
        }
      }
    }
  }

  // ---------- Image Picking ----------
  Future<void> _pickFromGallery() async {
    final XFile? x =
        await _picker.pickImage(source: ImageSource.gallery, imageQuality: 85);
    if (x == null) return;
    await _uploadToServerMany([File(x.path)]);
  }

  Future<void> _pickMultipleFromGallery() async {
    final List<XFile> files = await _picker.pickMultiImage(imageQuality: 85);
    if (files.isEmpty) return;
    await _uploadToServerMany(files.map((x) => File(x.path)).toList());
  }

  Future<void> _capturePhoto() async {
    final XFile? x =
        await _picker.pickImage(source: ImageSource.camera, imageQuality: 85);
    if (x == null) return;
    await _uploadToServerMany([File(x.path)]);
  }

  // ---------- Upload (Non-blocking queue) ----------
  Future<void> _uploadToServerMany(List<File> files) async {
    if (!mounted || files.isEmpty) return;

    int enqueued = 0;
    for (final file in files) {
      final added = await UploadQueueService.instance.enqueue(
        filePath: file.path,
        jobId: widget.jobId,
        fileName: file.path.split('/').last,
      );
      if (added) enqueued++;
    }

    if (!mounted) return;

    if (enqueued > 0) {
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('$enqueued photo(s) queued for upload')),
      );
    } else {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Photos already queued')),
      );
    }
  }

  // ---------- Delete ----------
  Map<String, dynamic> _buildDeletePayload(List<String> keysOrUrls) {
    return {
      'auth_token': 'aei@89806849',
      'job_id': widget.jobId.toString(),
      'paths': keysOrUrls, // should be fullpath (cacheKey)
    };
  }

  Future<bool> _deleteOnServer(List<String> keysOrUrls) async {
    final token = _getToken();
    final url = Uri.parse(_deleteEndpoint);

    final resp = await http
        .post(
          url,
          headers: {
            'Content-Type': 'application/json',
            if (token.isNotEmpty) 'authorization': token,
          },
          body: jsonEncode(_buildDeletePayload(keysOrUrls)),
        )
        .timeout(const Duration(seconds: 45));

    if (resp.statusCode != 200) {
      throw HttpException('Delete failed: HTTP ${resp.statusCode}');
    }

    final decoded = jsonDecode(resp.body);
    final bool ok = decoded is Map &&
        (decoded['success'] == true ||
            decoded['status'] == 'ok' ||
            decoded['deleted'] == true);
    return ok;
  }

  Future<void> _deleteSelected() async {
    if (_selectedKeys.isEmpty || _isDeleting) return;

    final List<String> toDelete = _selectedKeys.toList();

    setState(() => _isDeleting = true);
    try {
      final ok = await _deleteOnServer(toDelete);
      if (!ok) throw StateError('Server returned failure');

      // remove cached files for deleted keys
      for (final key in toDelete) {
        await AeiImageCache.instance.removeFile(key);
        _fileFutures.remove(key);
      }

      // remove locally
      setState(() {
        _photos.removeWhere((p) => _selectedKeys.contains(p.cacheKey));
        _selectedKeys.clear();
        _selectionMode = false;
      });

      if (mounted) {
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Deleted ${toDelete.length} photo(s)')),
        );
      }
    } catch (e) {
      if (!mounted) return;
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Delete failed: $e')),
      );
    } finally {
      if (mounted) setState(() => _isDeleting = false);
    }
  }

  void _toggleSelectionMode([bool? value]) {
    setState(() {
      _selectionMode = value ?? !_selectionMode;
      if (!_selectionMode) _selectedKeys.clear();
    });
  }

  void _toggleSelect(PhotoItem p) {
    setState(() {
      if (_selectedKeys.contains(p.cacheKey)) {
        _selectedKeys.remove(p.cacheKey);
      } else {
        _selectedKeys.add(p.cacheKey);
      }
      if (_selectedKeys.isEmpty) {
        _selectionMode = false;
      }
    });
  }

  // ---------- Caching ----------
  // Separate future maps for thumb vs full-size to avoid conflicts
  final Map<String, Future<File>> _thumbFutures = {};

  Future<File> _getCachedThumb(PhotoItem p) {
    final thumbKey = 'thumb_${p.cacheKey}';
    final thumbSrc = p.thumbUrl.isNotEmpty ? p.thumbUrl : p.url;
    return _thumbFutures.putIfAbsent(thumbKey, () {
      final token = _getToken();
      return AeiImageCache.instance.getSingleFile(
        thumbSrc,
        key: thumbKey,
        headers: token.isNotEmpty ? {'authorization': token} : null,
      );
    });
  }

  Future<File> _getCachedFile(PhotoItem p) {
    // For full-screen: use full_link (hi-res) if available, else standard
    final fullSrc = p.fullUrl.isNotEmpty ? p.fullUrl : p.url;
    return _fileFutures.putIfAbsent(p.cacheKey, () {
      final token = _getToken();
      return AeiImageCache.instance.getSingleFile(
        fullSrc,
        key: p.cacheKey,
        headers: token.isNotEmpty ? {'authorization': token} : null,
      );
    });
  }

  // ✅ Full-screen viewer (OPEN ON TAP)
  void _openGallery(int initialIndex) {
    if (!mounted) return;
    Navigator.of(context).push(MaterialPageRoute(
      fullscreenDialog: false,
      builder: (_) => FullScreenGallery(
        photos: _photos,
        initialIndex: initialIndex,
        getCachedFile: _getCachedFile,
        title: widget.customerName ?? 'Photos',
      ),
    ));
  }

  PreferredSizeWidget _buildHeader(String name) {
    final hasSelection = _selectionMode && _selectedKeys.isNotEmpty;

    return AppBar(
      backgroundColor: Colors.white,
      elevation: 0,
      foregroundColor: Colors.black87,
      titleSpacing: 0,
      title: Row(
        children: [
          const SizedBox(width: 8),
          const CircleAvatar(
            backgroundColor: Colors.green,
            child: Text("AE", style: TextStyle(color: Colors.white)),
          ),
          const SizedBox(width: 12),
          Expanded(
            child: Text(
              hasSelection
                  ? '${_selectedKeys.length} selected'
                  : (name.isEmpty ? 'Photos' : name),
              style: const TextStyle(fontSize: 16, fontWeight: FontWeight.bold),
            ),
          ),
          if (!hasSelection) ...[
            IconButton(
              icon: const Icon(Icons.photo_library_outlined),
              tooltip: 'Pick Multiple from Gallery',
              onPressed: _pickMultipleFromGallery,
            ),
            IconButton(
              icon: const Icon(Icons.photo_outlined),
              tooltip: 'Pick Single from Gallery',
              onPressed: _pickFromGallery,
            ),
            IconButton(
              icon: const Icon(Icons.camera_alt_outlined),
              tooltip: 'Capture Photo',
              onPressed: _capturePhoto,
            ),
          ] else ...[
            IconButton(
              icon: _isDeleting
                  ? const SizedBox(
                      width: 20,
                      height: 20,
                      child: CircularProgressIndicator(strokeWidth: 2),
                    )
                  : const Icon(Icons.delete_outline),
              tooltip: 'Delete selected',
              onPressed: _isDeleting ? null : _deleteSelected,
            ),
            IconButton(
              icon: const Icon(Icons.close),
              tooltip: 'Cancel selection',
              onPressed: () => _toggleSelectionMode(false),
            ),
          ],
          const SizedBox(width: 8),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final name = widget.customerName ?? '';

    return Scaffold(
      backgroundColor: Colors.white,
      appBar: _buildHeader(name),
      body: Column(
        children: [
          Container(
            color: Colors.grey.shade200,
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                Text(
                  isLoading
                      ? "Loading photos..."
                      : "${_photos.length} Photos Available",
                  style: const TextStyle(fontWeight: FontWeight.bold),
                ),
                const UploadStatusWidget(),
              ],
            ),
          ),

          Expanded(
            child: RefreshIndicator(
              onRefresh: _fetchPhotos,
              child: Builder(
                builder: (_) {
                  if (isLoading) {
                    return const Center(child: CircularProgressIndicator());
                  }
                  if (_error != null) {
                    return ListView(
                      physics: const AlwaysScrollableScrollPhysics(),
                      children: [
                        Padding(
                          padding: const EdgeInsets.all(16),
                          child: Text(
                            _error!,
                            style: const TextStyle(color: Colors.red),
                          ),
                        ),
                      ],
                    );
                  }
                  if (_photos.isEmpty) {
                    return ListView(
                      physics: const AlwaysScrollableScrollPhysics(),
                      children: const [
                        Padding(
                          padding: EdgeInsets.all(16),
                          child: Text('No photos found.'),
                        ),
                      ],
                    );
                  }

                  return Padding(
                    padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 8),
                    child: GridView.builder(
                      physics: const AlwaysScrollableScrollPhysics(),
                      itemCount: _photos.length,
                      gridDelegate: const SliverGridDelegateWithFixedCrossAxisCount(
                        crossAxisCount: 3,
                        mainAxisSpacing: 8,
                        crossAxisSpacing: 8,
                      ),
                      itemBuilder: (context, index) {
                        final p = _photos[index];

                        return FutureBuilder<File>(
                          future: _getCachedThumb(p),
                          builder: (context, snap) {
                            if (snap.hasError) {
                              return GestureDetector(
                                onTap: () {
                                  setState(() {
                                    _thumbFutures.remove('thumb_${p.cacheKey}');
                                  });
                                },
                                child: const Center(
                                  child: Column(
                                    mainAxisSize: MainAxisSize.min,
                                    children: [
                                      Icon(Icons.broken_image, color: Colors.grey, size: 32),
                                      SizedBox(height: 4),
                                      Text('Tap to retry', style: TextStyle(fontSize: 10, color: Colors.grey)),
                                    ],
                                  ),
                                ),
                              );
                            }
                            if (!snap.hasData) {
                              return const Center(
                                child: SizedBox(
                                  width: 24,
                                  height: 24,
                                  child: CircularProgressIndicator(strokeWidth: 2),
                                ),
                              );
                            }

                            final file = snap.data!;
                            final isSelected =
                                _selectionMode && _selectedKeys.contains(p.cacheKey);

                            return GestureDetector(
                              // Long press -> selection mode
                              onLongPress: () {
                                if (!_selectionMode) _toggleSelectionMode(true);
                                _toggleSelect(p);
                              },

                              // ✅ Tap -> ALWAYS open full screen (if not selecting)
                              onTap: () {
                                if (_selectionMode) {
                                  _toggleSelect(p);
                                } else {
                                  _openGallery(index); // FULL SCREEN page
                                }
                              },

                              child: Stack(
                                children: [
                                  ClipRRect(
                                    borderRadius: BorderRadius.circular(6),
                                    child: Hero(
                                      tag: p.cacheKey,
                                      child: Image.file(
                                        file,
                                        fit: BoxFit.cover,
                                        width: double.infinity,
                                        height: double.infinity,
                                      ),
                                    ),
                                  ),

                                  // Selection overlay
                                  if (_selectionMode)
                                    Positioned.fill(
                                      child: AnimatedContainer(
                                        duration: const Duration(milliseconds: 120),
                                        decoration: BoxDecoration(
                                          color: isSelected
                                              ? Colors.black26
                                              : Colors.black12.withOpacity(0.08),
                                          borderRadius: BorderRadius.circular(6),
                                          border: Border.all(
                                            color: isSelected
                                                ? Colors.blueAccent
                                                : Colors.transparent,
                                            width: 2,
                                          ),
                                        ),
                                      ),
                                    ),
                                  if (_selectionMode)
                                    Positioned(
                                      top: 6,
                                      right: 6,
                                      child: Container(
                                        decoration: BoxDecoration(
                                          color: isSelected
                                              ? Colors.blueAccent
                                              : Colors.white,
                                          shape: BoxShape.circle,
                                          boxShadow: [
                                            BoxShadow(
                                              color: Colors.black.withOpacity(0.2),
                                              blurRadius: 4,
                                            )
                                          ],
                                        ),
                                        padding: const EdgeInsets.all(2),
                                        child: Icon(
                                          isSelected
                                              ? Icons.check_circle
                                              : Icons.radio_button_unchecked,
                                          size: 20,
                                          color: isSelected
                                              ? Colors.white
                                              : Colors.black54,
                                        ),
                                      ),
                                    ),
                                ],
                              ),
                            );
                          },
                        );
                      },
                    ),
                  );
                },
              ),
            ),
          ),
        ],
      ),
    );
  }
}

// ---------------- FullScreenGallery ----------------
//
