import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:intl/intl.dart';
import 'package:shared_preferences/shared_preferences.dart';
import '../token_provider.dart';
import 'package:provider/provider.dart';
import 'my_job_page.dart';
import 'job_details_full_view.dart';

class JobPage extends StatefulWidget {
  final DateTime selectedDate;

  const JobPage({super.key, required this.selectedDate});

  @override
  State<JobPage> createState() => _JobPageState();
}

class _JobPageState extends State<JobPage> {
  late DateTime _currentDate;
  Map<String, List<Map<String, String>>> groupedJobs = {};
  final TextEditingController _searchController = TextEditingController();
  bool _isLoading = false;
  String? userId;

  @override
  void initState() {
    super.initState();
    _currentDate = widget.selectedDate;
    _searchController.text = _formatDateForInput(_currentDate);
    _loadUserAndJobs();
  }

  Future<void> _loadUserAndJobs() async {
    final prefs = await SharedPreferences.getInstance();
    userId = prefs.getString('user_id');

    if (userId == null) {
      print('No user ID found.');
      return;
    }

    await _loadJobs();
  }

  Future<void> _loadJobs() async {
    setState(() {
      _isLoading = true;
    });

    final formattedDate = DateFormat('yyyy-MM-dd').format(_currentDate);
    final token = Provider.of<TokenProvider>(context, listen: false).token;

    final url = Uri.parse(
      'https://aeihawaii.com/photoappsch/loginapinew/getmysearchjob/$userId/$formattedDate',
    );

    try {
      final response = await http.get(
        url,
        headers: {
          'authorization': token,
          'Content-Type': 'application/json',
        },
      );

      if (response.statusCode == 200) {
        final List<dynamic> data = json.decode(response.body);
        Map<String, List<Map<String, String>>> grouped = {};

        for (var job in data) {
          final jobDate = job['job_date'] ?? 'Unknown Date';
          final formatted = _formatDateForHeader(jobDate);

          grouped.putIfAbsent(formatted, () => []);
          grouped[formatted]!.add({
            'name': job['customername']?.toString() ?? '',
            'location': job['joblocation']?.toString() ?? '',
            'code': job['jobcode']?.toString() ?? '',
            'customerId': job['customerid']?.toString() ?? '',
            'Id': job['id']?.toString() ?? '',
            'color': job['color']?.toString() ??
                '', // may be int-like or hex as string
          });
        }

        setState(() {
          groupedJobs = grouped;
        });
      } else {
        print('API error: ${response.statusCode}');
      }
    } catch (e) {
      print('Error fetching jobs: $e');
    } finally {
      setState(() {
        _isLoading = false;
      });
    }
  }

  String _formatDateForInput(DateTime date) {
    return DateFormat('yyyy-MM-dd').format(date);
  }

  String _formatDateForHeader(String rawDate) {
    try {
      final date = DateTime.parse(rawDate);
      return DateFormat.yMMMMd().format(date);
    } catch (_) {
      return rawDate;
    }
  }

  Future<void> _showDatePicker() async {
    final picked = await showDatePicker(
      context: context,
      initialDate: _currentDate,
      firstDate: DateTime(2020),
      lastDate: DateTime(2100),
    );

    if (picked != null && picked != _currentDate) {
      setState(() {
        _currentDate = picked;
        _searchController.text = _formatDateForInput(_currentDate);
        _loadJobs();
      });
    }
  }

  // -------- Color helpers (FIX) --------

  /// Parse any of:
  /// - ARGB int (e.g., 4283794388)
  /// - decimal string int ("4283794388")
  /// - hex strings: "#RRGGBB", "RRGGBB", "#AARRGGBB", "AARRGGBB", "0xFFRRGGBB"
  /// Ensures alpha is opaque (FF) when missing or 00.
  Color parseColorSafe(dynamic v, {Color fallback = const Color(0xFFE0E0E0)}) {
    try {
      if (v == null) return fallback;

      if (v is int) {
        int argb = v;
        if ((argb >>> 24) == 0x00) {
          argb = (0xFF << 24) | (argb & 0x00FFFFFF);
        }
        return Color(argb);
      }

      if (v is num) {
        int argb = v.toInt();
        if ((argb >>> 24) == 0x00) {
          argb = (0xFF << 24) | (argb & 0x00FFFFFF);
        }
        return Color(argb);
      }

      if (v is String) {
        var s = v.trim();

        // Decimal integer in string
        if (RegExp(r'^\d+$').hasMatch(s)) {
          int argb = int.parse(s);
          if ((argb >>> 24) == 0x00) {
            argb = (0xFF << 24) | (argb & 0x00FFFFFF);
          }
          return Color(argb);
        }

        // Hex variants
        s = s.toUpperCase().replaceAll('#', '');
        if (s.startsWith('0X')) s = s.substring(2);

        if (s.length == 6) {
          s = 'FF$s'; // add alpha if missing
        } else if (s.length != 8) {
          return fallback;
        }

        final val = int.parse(s, radix: 16);
        final argb =
            ((val >>> 24) == 0x00) ? ((0xFF << 24) | (val & 0x00FFFFFF)) : val;
        return Color(argb);
      }

      return fallback;
    } catch (_) {
      return fallback;
    }
  }

  /// Choose black/white based on perceived luminance for readability.
  Color bestTextColorFor(Color bg) {
    final r = bg.red / 255.0;
    final g = bg.green / 255.0;
    final b = bg.blue / 255.0;
    final l = 0.2126 * r + 0.7152 * g + 0.0722 * b;
    return (l > 0.6) ? Colors.black : Colors.white;
  }

Widget _buildJobRow(Map<String, String> job) {
  final Color bgColor = parseColorSafe(job['color']);
  final Color fg = bestTextColorFor(bgColor);

  return InkWell(
    onTap: () {
      final jid = int.tryParse(job['Id'] ?? '');
      if (jid != null) {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (context) => JobDetailsFullView(jobId: jid),
          ),
        );
      }
    },
    child: Container(
      color: bgColor,
      padding: const EdgeInsets.symmetric(vertical: 12, horizontal: 16),
      margin: const EdgeInsets.symmetric(vertical: 2),
      child: Row(
        children: [
          Expanded(
            flex: 3,
            child: Text(
              job['name'] ?? "",
              style: TextStyle(color: fg),
              overflow: TextOverflow.ellipsis,
            ),
          ),
          Expanded(
            flex: 2,
            child: Text(
              job['location'] ?? "",
              style: TextStyle(color: fg),
              overflow: TextOverflow.ellipsis,
            ),
          ),
          Expanded(
            flex: 1,
            child: Text(
              job['code'] ?? "",
              style: TextStyle(color: fg),
              overflow: TextOverflow.ellipsis,
            ),
          ),
        ],
      ),
    ),
  );
}
  Widget _buildJobSection(String date, List<Map<String, String>> jobs) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Container(
          color: const Color(0xFFEFEFEF),
          width: double.infinity,
          padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
          child: Text(
            date,
            style: const TextStyle(fontWeight: FontWeight.bold),
          ),
        ),
        ...jobs.map(_buildJobRow),
      ],
    );
  }

  @override
  Widget build(BuildContext context) {
    return DefaultTabController(
      length: 2,
      child: Scaffold(
        appBar: AppBar(
          backgroundColor: const Color(0xFF418E1C),
          title: const Text(
            'Job Schedule',
            style: TextStyle(
              fontSize: 20,
              fontWeight: FontWeight.bold,
              color: Colors.white,
            ),
            overflow: TextOverflow.ellipsis,
          ),
          leading: IconButton(
            icon: const Icon(Icons.arrow_back),
            onPressed: () => Navigator.pop(context),
          ),
          actions: const [
            Icon(Icons.help_outline),
            SizedBox(width: 12),
            Icon(Icons.menu),
            SizedBox(width: 8),
          ],
          bottom: PreferredSize(
            preferredSize: const Size.fromHeight(110),
            child: Column(
              children: [
                Padding(
                  padding:
                      const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
                  child: Row(
                    children: [
                      Expanded(
                        child: GestureDetector(
                          onTap: _showDatePicker,
                          child: AbsorbPointer(
                            child: TextField(
                              controller: _searchController,
                              style: const TextStyle(color: Colors.white),
                              decoration: InputDecoration(
                                hintText: 'Tap to select date',
                                hintStyle:
                                    const TextStyle(color: Colors.white54),
                                filled: true,
                                fillColor: const Color(0xFF418E1C),
                                prefixIcon: const Icon(Icons.search,
                                    color: Colors.white),
                                contentPadding: const EdgeInsets.symmetric(
                                  horizontal: 12,
                                  vertical: 10,
                                ),
                                border: OutlineInputBorder(
                                  borderRadius: BorderRadius.circular(8),
                                ),
                              ),
                            ),
                          ),
                        ),
                      ),
                      const SizedBox(width: 8),
                      IconButton(
                        onPressed: _showDatePicker,
                        icon: const Icon(Icons.calendar_today,
                            color: Colors.white),
                      ),
                    ],
                  ),
                ),
                const TabBar(
                  indicatorColor: Colors.black,
                  labelColor: Color.fromARGB(255, 245, 245, 243),
                  unselectedLabelColor: Colors.grey,
                  tabs: [
                    Tab(text: "Schedule"),
                    Tab(text: "My Jobs"),
                  ],
                ),
              ],
            ),
          ),
        ),
        body: _isLoading
            ? const Center(child: CircularProgressIndicator())
            : TabBarView(
                children: [
                  groupedJobs.isEmpty
                      ? const Center(child: Text("No jobs found."))
                      : ListView(
                          children: groupedJobs.entries
                              .map((entry) =>
                                  _buildJobSection(entry.key, entry.value))
                              .toList(),
                        ),
                  MyJobPage(),
                ],
              ),
      ),
    );
  }
}
