import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:flutter_html/flutter_html.dart';

import 'edit_job_note_page.dart';
import '../token_provider.dart';

class JobNotesFullView extends StatefulWidget {
  final int customerId; // non-nullable
  final int jobId; // non-nullable

  const JobNotesFullView({
    super.key,
    required this.customerId,
    required this.jobId,
  });

  @override
  State<JobNotesFullView> createState() => _JobNotesFullViewState();
}

class _JobNotesFullViewState extends State<JobNotesFullView> {
  // ---- Local state ----
  final List<Map<String, dynamic>> notes = [];

  // Job summary (for header + info rows)
  Map<String, dynamic> jobSummary = {};
  bool _isSummaryLoading = true;

  bool _isNotesLoading = true;
  String _token = '';
  String? _userId;

  @override
  void initState() {
    super.initState();
    _loadTokenAndData();
  }

  Future<void> _loadTokenAndData() async {
    try {
      final token = context.read<TokenProvider>().token ?? '';
      _token = token;

      final prefs = await SharedPreferences.getInstance();
      _userId = prefs.getString('user_id');

      if (_userId == null || _token.isEmpty) {
        debugPrint('Missing user ID or token');
        setState(() {
          _isNotesLoading = false;
          _isSummaryLoading = false;
        });
        return;
      }

      // fetch notes + job summary in parallel
      await Future.wait([
        _loadCustomerNotes(_userId!, _token),
        _loadJobSummary(_token),
      ]);
    } catch (e) {
      debugPrint('Bootstrap error: $e');
      if (mounted) {
        setState(() {
          _isNotesLoading = false;
          _isSummaryLoading = false;
        });
      }
    }
  }

  Future<void> _loadCustomerNotes(String userId, String token) async {
    setState(() => _isNotesLoading = true);
    try {
      final uri = Uri.parse(
        'https://aeihawaii.com/photoappsch/loginapinew/getcustomernotes/${widget.jobId}',
      );
      final response = await http.get(uri, headers: {
        'Content-Type': 'application/json',
        'authorization': token,
      });

      if (response.statusCode == 200) {
        final List<dynamic> data = json.decode(response.body);
        setState(() {
          notes
            ..clear()
            ..addAll(
              data.map<Map<String, dynamic>>((note) => {
                    'postedBy': note['postedby'] ?? '',
                    'date': note['createdat'] ?? '',
                    'rank': note['rank'] ?? 'Note',
                    'category': note['category'] ?? 'General Comments',
                    'message': note['message'] ?? '',
                    'replies': (note['replies'] as List<dynamic>? ?? [])
                        .map<Map<String, dynamic>>((reply) => {
                              'postedBy': reply['postedby'] ?? '',
                              'date': reply['createdat'] ?? '',
                              'message': reply['message'] ?? '',
                            })
                        .toList(),
                  }),
            );
        });
      } else {
        debugPrint('Note API error: ${response.statusCode}');
      }
    } catch (e) {
      debugPrint('Failed to load notes: $e');
    } finally {
      if (mounted) setState(() => _isNotesLoading = false);
    }
  }

  // fetch summary for header + info rows
  Future<void> _loadJobSummary(String token) async {
    setState(() => _isSummaryLoading = true);
    try {
      final uri = Uri.parse(
        'https://aeihawaii.com/photoappsch/loginapinew/jobdetailsfullviewstate/${widget.jobId}',
      );

      final response = await http.get(uri, headers: {
        'Content-Type': 'application/json',
        'authorization': token,
      });

      if (response.statusCode == 200) {
        final Map<String, dynamic> data = json.decode(response.body);

        // Normalize expected keys so the UI is consistent
        jobSummary = {
          'customerName':
              (data['customerName'] ?? data['customer_name'] ?? '').toString(),
          'jobType': (data['jobType'] ?? data['job_type'] ?? '').toString(),
          'status': (data['status'] ?? '').toString(),
          'date': (data['date'] ?? '').toString(),
          'time': (data['time'] ?? '').toString(),
          'jobId': (data['jobId'] ?? data['id'] ?? widget.jobId).toString(),
          'technicians':
              (data['technicians'] ?? data['staff'] ?? '').toString(),
        };

        if (mounted) setState(() => _isSummaryLoading = false);
      } else {
        debugPrint(
            'Summary API error: ${response.statusCode} -> ${response.body}');
        if (mounted) setState(() => _isSummaryLoading = false);
      }
    } catch (e) {
      debugPrint('Failed to load summary: $e');
      if (mounted) setState(() => _isSummaryLoading = false);
    }
  }

  // ---- API: Add note ----
  Future<bool> _apiAddNote({
    required int jobId,
    required String note,
    required String category, // ctype
    required String rating, // rtype
  }) async {
    if (_token.isEmpty) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Missing auth token')),
      );
      return false;
    }

    final uri = Uri.parse(
      'https://aeihawaii.com/photoappsch/loginapinew/addnote/$jobId',
    );

    try {
      final resp = await http.post(
        uri,
        headers: {
          // CI $this->input->post works best with form-encoded
          'Content-Type': 'application/x-www-form-urlencoded',
          'authorization': _token, // raw token (no "Bearer ")
        },
        body: {
          'note': note,
          'ctype': category,
          'rtype': rating,
        },
      ).timeout(const Duration(seconds: 30));

      if (resp.statusCode != 200) {
        debugPrint('addnote HTTP ${resp.statusCode}: ${resp.body}');
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Add note failed: HTTP ${resp.statusCode}')),
        );
        return false;
      }

      // Try parse JSON; if it's plain text, treat HTTP 200 as success
      try {
        final decoded = jsonDecode(resp.body);
        if (decoded is Map && decoded['status'] == 4011) {
          ScaffoldMessenger.of(context).showSnackBar(
            SnackBar(
                content: Text(decoded['message']?.toString() ?? 'Auth error')),
          );
          return false;
        }
      } catch (_) {}

      return true;
    } catch (e) {
      debugPrint('addnote error: $e');
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Add note error: $e')),
      );
      return false;
    }
  }

  Future<void> _addNote() async {
    final result = await Navigator.push(
      context,
      MaterialPageRoute(
        builder: (_) => const EditJobNotePage(
          author: '',
          datetime: '',
          initialNote: '',
        ),
      ),
    );

    if (!mounted) return;

    if (result is Map) {
      final body = (result['body'] ?? '').toString().trim();
      final category = (result['category'] ?? '').toString();
      final rating = (result['rank'] ?? '').toString();

      if (body.isEmpty) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Note cannot be empty')),
        );
        return;
      }

      setState(() => _isNotesLoading = true);

      final ok = await _apiAddNote(
        jobId: widget.jobId,
        note: body,
        category: category,
        rating: rating,
      );

      if (ok) {
        await _loadCustomerNotes(_userId!, _token);
        if (mounted) {
          ScaffoldMessenger.of(context).showSnackBar(
            const SnackBar(content: Text('Note added')),
          );
        }
      }

      if (mounted) setState(() => _isNotesLoading = false);
    }
  }

  Future<void> _editNote(int index) async {
    final note = notes[index];
    final result = await Navigator.push(
      context,
      MaterialPageRoute(
        builder: (_) => EditJobNotePage(
          author: note['postedBy'] ?? '',
          datetime: note['date'] ?? '',
          initialNote: note['message'] ?? '',
          initialCategory: note['category']?.toString(),
          initialRating: note['rank']?.toString(),
        ),
      ),
    );

    if (result != null) {
      if (result == 'delete') {
        // No delete API provided; remove locally for now
        setState(() => notes.removeAt(index));
      } else if (result is Map) {
        // No update API provided; update locally for now
        setState(() {
          notes[index] = {
            'postedBy': result['author'] ?? (note['postedBy'] ?? ''),
            'date': result['datetime'] ?? (note['date'] ?? ''),
            'rank': result['rank'] ?? (note['rank'] ?? 'Note'),
            'category':
                result['category'] ?? (note['category'] ?? 'General Comments'),
            'message': result['body'] ?? (note['message'] ?? ''),
            'replies': note['replies'] ?? <Map<String, dynamic>>[],
          };
        });
      }
    }
  }

  Widget _buildCommentCard(Map<String, dynamic> comment) {
    final List<Map<String, dynamic>> replies =
        (comment['replies'] as List?)?.cast<Map<String, dynamic>>() ?? [];

    return Container(
      margin: const EdgeInsets.only(bottom: 12),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: const Color(0xFFF9F9F9),
        border: Border.all(color: const Color(0xFFDDDDDD)),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text('Posted: ${comment['date']} by ${comment['postedBy']}',
              style: const TextStyle(fontWeight: FontWeight.bold)),
          Text('Rank: ${comment['rank'] ?? 'Note'}',
              style: const TextStyle(color: Colors.green)),
          const SizedBox(height: 8),
          Html(data: (comment['message'] ?? '').toString()),
          const SizedBox(height: 8),
          if (replies.isNotEmpty)
            Padding(
              padding: const EdgeInsets.only(left: 12, top: 12),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: replies
                    .map((reply) => Container(
                          margin: const EdgeInsets.only(bottom: 8),
                          padding: const EdgeInsets.all(10),
                          decoration: const BoxDecoration(
                            color: Color(0xFFF0F0F0),
                            border: Border(
                              left: BorderSide(color: Colors.green, width: 3),
                            ),
                          ),
                          child: Column(
                            crossAxisAlignment: CrossAxisAlignment.start,
                            children: [
                              Text(
                                  'Date Posted: ${reply['date']} by: ${reply['postedBy']}',
                                  style: const TextStyle(
                                      fontWeight: FontWeight.w600)),
                              const SizedBox(height: 4),
                              Html(data: (reply['message'] ?? '').toString()),
                            ],
                          ),
                        ))
                    .toList(),
              ),
            ),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final name = (jobSummary['customerName'] ?? '—').toString();
    final jobType = (jobSummary['jobType'] ?? '').toString();
    final status = (jobSummary['status'] ?? '').toString();
    final date = (jobSummary['date'] ?? '').toString();
    final time = (jobSummary['time'] ?? '').toString();
    final jobId = (jobSummary['jobId'] ?? widget.jobId).toString();
    final techs = (jobSummary['technicians'] ?? '').toString();

    return Scaffold(
      backgroundColor: Colors.white,
      body: Column(
        children: [
          // Job Info (dynamic)
          Container(
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
            child: _isSummaryLoading
                ? const _ShimmerInfoRows()
                : Column(
                    children: [
                      _InfoRow(
                          Icons.flag,
                          [jobType, status]
                              .where((s) => s.isNotEmpty)
                              .join(' | ')),
                      _InfoRow(Icons.calendar_today, date.isEmpty ? '—' : date),
                      _InfoRow(Icons.access_time, time.isEmpty ? '—' : time),
                      _InfoRow(Icons.tag, jobId.isEmpty ? '—' : jobId),
                      _InfoRow(Icons.group, techs.isEmpty ? '—' : techs),
                    ],
                  ),
          ),

          // Job Notes Header
          Container(
            color: Colors.grey.shade200,
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                const Text('Job Notes',
                    style: TextStyle(fontWeight: FontWeight.bold)),
                IconButton(
                  icon: const Icon(Icons.add_circle_outline),
                  onPressed: _addNote,
                  tooltip: 'Add Note',
                ),
              ],
            ),
          ),

          // Notes List
          Expanded(
            child: _isNotesLoading
                ? const Center(child: CircularProgressIndicator())
                : (notes.isEmpty
                    ? const Center(
                        child: Text('No notes found.',
                            style: TextStyle(color: Colors.black54)))
                    : ListView.builder(
                        padding: const EdgeInsets.symmetric(
                            horizontal: 16, vertical: 8),
                        itemCount: notes.length,
                        itemBuilder: (context, index) {
                          final note = notes[index];
                          final category = (note['category'] ?? '').toString();
                          return GestureDetector(
                            onTap: () => _editNote(index),
                            child: Column(
                              crossAxisAlignment: CrossAxisAlignment.start,
                              children: [
                                if (category.isNotEmpty)
                                  Text(
                                    category,
                                    style: const TextStyle(
                                      fontSize: 18,
                                      fontWeight: FontWeight.bold,
                                      color: Colors.black87,
                                    ),
                                  ),
                                const SizedBox(height: 8),
                                _buildCommentCard(note),
                                const Divider(height: 32),
                              ],
                            ),
                          );
                        },
                      )),
          ),

          // Activity Log Button
          Container(
            padding: const EdgeInsets.all(12),
            child: Center(
              child: TextButton(
                onPressed: () {
                  // TODO: show activity log
                },
                child: const Text(
                  'Show Activity Log',
                  style: TextStyle(
                    fontWeight: FontWeight.bold,
                    color: Colors.black87,
                  ),
                ),
              ),
            ),
          ),
        ],
      ),
    );
  }
}

class _InfoRow extends StatelessWidget {
  final IconData icon;
  final String text;
  const _InfoRow(this.icon, this.text);

  @override
  Widget build(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 4),
      child: Row(
        children: [
          Icon(icon, color: Colors.black54, size: 18),
          const SizedBox(width: 12),
          Expanded(
            child: Text(text, style: const TextStyle(fontSize: 14)),
          ),
        ],
      ),
    );
  }
}

class _ShimmerInfoRows extends StatelessWidget {
  const _ShimmerInfoRows();

  @override
  Widget build(BuildContext context) {
    return Column(
      children: List.generate(
        5,
        (i) => Padding(
          padding: const EdgeInsets.symmetric(vertical: 6),
          child: Row(
            children: [
              const Icon(Icons.more_horiz, color: Colors.black26, size: 18),
              const SizedBox(width: 12),
              Expanded(
                child: Container(
                  height: 12,
                  color: Colors.black12,
                ),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
