// lib/pages/job_docs_full_view.dart
import 'dart:convert';
import 'dart:io';

import 'package:file_picker/file_picker.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart'; // MissingPluginException
import 'package:http/http.dart' as http;
import 'package:image_picker/image_picker.dart'; // optional: for camera/photos
import 'package:open_filex/open_filex.dart';
import 'package:path_provider/path_provider.dart';
import 'package:provider/provider.dart';

import '../token_provider.dart';
import 'in_app_doc_viewer.dart';

class JobDocsFullView extends StatefulWidget {
  final int jobId;

  const JobDocsFullView({
    super.key,
    required this.jobId,
  });

  @override
  State<JobDocsFullView> createState() => _JobDocsFullViewState();
}

class _JobDocsFullViewState extends State<JobDocsFullView> {
  bool _isLoading = true;
  bool _isUploading = false;
  String? _error;
  final List<_DocItem> _documents = [];
  final Set<String> _downloading = {}; // pfids currently downloading
  final ImagePicker _picker = ImagePicker();

  String get _listApi =>
      'https://aeihawaii.com/photoappsch/loginapinew/customerdocstabdetails/${widget.jobId}';
  String _fileApi(String pfid) =>
      'https://aeihawaii.com/photoappsch/loginapinew/getcustomerdoc/$pfid';
  String get _uploadApi =>
      'https://aeihawaii.com/photoappsch/loginapinew/uploaddoc/${widget.jobId}';

  @override
  void initState() {
    super.initState();
    _fetchDocs();
  }

  // ----------------------- UTILITIES -----------------------

  Future<Directory> _safeTempDir() async {
    try {
      return await getTemporaryDirectory();
    } on MissingPluginException {
      // Fallback to a process temp directory so the app doesn't crash.
      return Directory.systemTemp;
    }
  }

  String _readToken() =>
      Provider.of<TokenProvider>(context, listen: false).token ?? '';

  void _toast(String msg) {
    if (!mounted) return;
    ScaffoldMessenger.of(context).showSnackBar(SnackBar(content: Text(msg)));
  }

  // ----------------------- FETCH LIST -----------------------

  Future<void> _fetchDocs() async {
    setState(() {
      _isLoading = true;
      _error = null;
    });

    try {
      final token = _readToken();
      if (token.isEmpty) {
        throw Exception('Missing token');
      }

      final resp = await http.get(
        Uri.parse(_listApi),
        headers: {
          'Content-Type': 'application/json',
          'authorization': token, // raw token (no "Bearer ")
        },
      );

      if (resp.statusCode == 401 || resp.statusCode == 403) {
        throw Exception('Session expired or invalid token.');
      }
      if (resp.statusCode != 200) {
        throw Exception('HTTP ${resp.statusCode}');
      }

      final decoded = json.decode(resp.body);
      if (decoded is! List) {
        throw Exception('Unexpected response shape');
      }

      final List<_DocItem> items = [];
      for (final item in decoded) {
        final type = (item['type'] ?? '').toString();
        final jobDate = (item['jobDate'] ?? '').toString();
        final jobId = (item['jobId'] ?? '').toString();
        final uploaded = (item['uploadedDate'] ?? '').toString();

        // pfid can be null/"null"/empty
        final pfidRaw = item['pfid'];
        final pfid = (pfidRaw == null)
            ? null
            : pfidRaw.toString().trim().toLowerCase() == 'null'
                ? null
                : (pfidRaw.toString().trim().isEmpty
                    ? null
                    : pfidRaw.toString().trim());

        final title = [
          if (type.isNotEmpty) type,
          if (jobDate.isNotEmpty) jobDate,
          if (jobId.isNotEmpty) 'JobID $jobId',
        ].join(' – ');

        items.add(_DocItem(
          title: title.isEmpty ? 'Document' : title,
          uploadedDisplay: uploaded,
          pfid: pfid, // nullable
          kind: _guessKindFromMeta(type),
        ));
      }

      setState(() {
        _documents
          ..clear()
          ..addAll(items);
        _isLoading = false;
      });
    } catch (e) {
      setState(() {
        _error = 'Failed to load documents: $e';
        _isLoading = false;
      });
    }
  }

  // ----------------------- DOWNLOAD / PREVIEW -----------------------

  Future<void> _downloadAndOpen(String pfid,
      {required String suggestedName}) async {
    if (_downloading.contains(pfid)) return;
    setState(() => _downloading.add(pfid));
    try {
      final token = _readToken();
      if (token.isEmpty) {
        _toast('Missing token. Please log in again.');
        return;
      }

      // 1) First attempt: direct binary download
      final primary = await http.get(
        Uri.parse(_fileApi(pfid)),
        headers: {'authorization': token},
      );

      if (primary.statusCode == 401 || primary.statusCode == 403) {
        _toast('Session expired. Please log in.');
        return;
      }
      if (primary.statusCode != 200) {
        _toast('Download failed (HTTP ${primary.statusCode}).');
        return;
      }

      List<int> bytes;
      String filename;
      String? contentType = primary.headers['content-type'];

      // Some APIs return JSON { url: "..."} or { data: "<base64>" }
      final isJson =
          (contentType ?? '').toLowerCase().contains('application/json') ||
              _looksLikeJson(primary.bodyBytes);

      if (isJson) {
        final jsonMap = jsonDecode(utf8.decode(primary.bodyBytes));
        if (jsonMap is Map && jsonMap['url'] is String) {
          // 2) Follow URL (may be presigned)
          final url = jsonMap['url'] as String;
          final follow = await http.get(Uri.parse(url));
          if (follow.statusCode != 200) {
            _toast('Download failed (follow-up ${follow.statusCode}).');
            return;
          }
          bytes = follow.bodyBytes;
          contentType = follow.headers['content-type'] ?? contentType;
          filename = _filenameFromContentDisposition(
                  follow.headers['content-disposition']) ??
              _sanitizeFilename(suggestedName);
        } else if (jsonMap is Map && jsonMap['data'] is String) {
          // base64 payload
          final b64 = jsonMap['data'] as String;
          bytes = base64Decode(b64);
          contentType = jsonMap['contentType']?.toString() ?? contentType;
          filename = _sanitizeFilename(suggestedName);
        } else {
          // Unknown JSON
          _toast('Download failed (unexpected JSON).');
          return;
        }
      } else {
        // Binary path
        bytes = primary.bodyBytes;
        filename = _filenameFromContentDisposition(
                primary.headers['content-disposition']) ??
            _sanitizeFilename(suggestedName);
      }

      // Ensure extension from content-type if possible
      final ext = _extFromContentType(contentType ?? '');
      if (ext != null && !filename.toLowerCase().endsWith('.$ext')) {
        filename = '$filename.$ext';
      }

      // Save
      final dir = await _safeTempDir();
      final file = File('${dir.path}/$filename');
      await file.writeAsBytes(bytes);

      // Open in-app viewer (PDF gets native pinch-to-zoom; others use system viewer)
      if (mounted) {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (_) => InAppDocViewer(
              filePath: file.path,
              title: suggestedName,
            ),
          ),
        );
      }
    } catch (e) {
      _toast('Could not open file: $e');
    } finally {
      if (mounted) {
        setState(() => _downloading.remove(pfid));
      }
    }
  }

  bool _looksLikeJson(List<int> bodyBytes) {
    // Quick check: body starts with "{" or "[" in UTF-8
    if (bodyBytes.isEmpty) return false;
    final ch = bodyBytes.first;
    return ch == 0x7B /*{*/ || ch == 0x5B /*[*/;
  }

  String? _filenameFromContentDisposition(String? cd) {
    if (cd == null) return null;
    // e.g., attachment; filename="name.pdf" OR RFC5987: filename*=UTF-8''name.pdf
    final match =
        RegExp(r'filename\*?=([^;]+)', caseSensitive: false).firstMatch(cd);
    if (match == null) return null;
    var raw = match.group(1)!.trim();
    // Handle RFC5987 flag safely (fixed quotes)
    raw = raw.replaceAll(RegExp(r"UTF-8'\w*'", caseSensitive: false), '');
    raw = raw.replaceAll('"', '');
    return _sanitizeFilename(Uri.decodeFull(raw));
  }

  String _sanitizeFilename(String name) {
    final cleaned = name.replaceAll(RegExp(r'[\\/:*?"<>|]'), '_').trim();
    return cleaned.isEmpty ? 'document' : cleaned;
  }

  String? _extFromContentType(String ct) {
    final l = ct.toLowerCase();
    if (l.contains('pdf')) return 'pdf';
    if (l.contains('png')) return 'png';
    if (l.contains('jpeg') || l.contains('jpg')) return 'jpg';
    if (l.contains('msword')) return 'doc';
    if (l.contains('officedocument.wordprocessingml.document')) return 'docx';
    if (l.contains('plain')) return 'txt';
    return null;
  }

  // ----------------------- UPLOAD -----------------------

  Future<void> _showUploadOptions() async {
    await showModalBottomSheet(
      context: context,
      builder: (context) => SafeArea(
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            ListTile(
              leading: const Icon(Icons.folder),
              title: const Text('Choose File'),
              onTap: () async {
                Navigator.pop(context);
                await _pickAndUploadFile();
              },
            ),
            ListTile(
              leading: const Icon(Icons.photo_library),
              title: const Text('Photo Library'),
              onTap: () async {
                Navigator.pop(context);
                await _pickAndUploadFromGallery();
              },
            ),
            ListTile(
              leading: const Icon(Icons.camera_alt),
              title: const Text('Take Photo'),
              onTap: () async {
                Navigator.pop(context);
                await _pickAndUploadFromCamera();
              },
            ),
          ],
        ),
      ),
    );
  }

  Future<void> _pickAndUploadFile() async {
    final res = await FilePicker.platform.pickFiles(
      allowMultiple: false,
      withData: true, // so we have bytes on all platforms
    );
    if (res == null || res.files.isEmpty) return;

    final file = res.files.first;
    final bytes = file.bytes ??
        (file.path != null ? await File(file.path!).readAsBytes() : null);
    if (bytes == null) {
      _toast('Could not read file bytes.');
      return;
    }

    final fileName = file.name.isNotEmpty
        ? file.name
        : (file.path != null ? file.path!.split('/').last : 'upload.bin');

    await _uploadMultipart(bytes: bytes, fileName: fileName);
  }

  Future<void> _pickAndUploadFromGallery() async {
    final XFile? x =
        await _picker.pickImage(source: ImageSource.gallery, imageQuality: 90);
    if (x == null) return;
    final bytes = await File(x.path).readAsBytes();
    final fileName = x.name.isNotEmpty ? x.name : x.path.split('/').last;
    await _uploadMultipart(bytes: bytes, fileName: fileName);
  }

  Future<void> _pickAndUploadFromCamera() async {
    final XFile? x =
        await _picker.pickImage(source: ImageSource.camera, imageQuality: 90);
    if (x == null) return;
    final bytes = await File(x.path).readAsBytes();
    final fileName = x.name.isNotEmpty ? x.name : x.path.split('/').last;
    await _uploadMultipart(bytes: bytes, fileName: fileName);
  }

  // *** UPDATED: single multipart request with field name 'file' ***
  Future<void> _uploadMultipart({
    required List<int> bytes,
    required String fileName,
  }) async {
    try {
      setState(() => _isUploading = true);
      final token = _readToken();
      if (token.isEmpty) {
        _toast('Missing token. Please log in.');
        return;
      }

      final uri = Uri.parse(_uploadApi);
      final req = http.MultipartRequest('POST', uri);
      req.headers['authorization'] = token; // raw token, no "Bearer"
      // DO NOT set Content-Type manually; MultipartRequest sets it.

      // Some backends still expect job_id in body even if it's in the URL:
      req.fields['job_id'] = widget.jobId.toString();

      // Use ONE field name that server expects: 'file'
      req.files.add(http.MultipartFile.fromBytes(
        'file',
        bytes,
        filename: fileName,
      ));

      final streamed = await req.send();
      final resp = await http.Response.fromStream(streamed);
      final bodyText = resp.body.trim();

      bool ok = resp.statusCode == 200;
      try {
        final decoded = json.decode(bodyText);
        if (decoded is Map) {
          final status = (decoded['status'] ?? '').toString().toLowerCase();
          final success = decoded['success'] == true;
          if (success || status == 'ok') ok = true;
        }
      } catch (_) {
        // Non-JSON body; leave ok as statusCode==200
      }

      if (ok) {
        _toast('Uploaded "$fileName".');
        await _fetchDocs();
      } else {
        _toast(
            'Upload failed: HTTP ${resp.statusCode}${bodyText.isNotEmpty ? " – $bodyText" : ""}');
      }
    } catch (e) {
      _toast('Upload error: $e');
    } finally {
      if (mounted) setState(() => _isUploading = false);
    }
  }

  // ----------------------- UI HELPERS -----------------------

  static _DocKind _guessKindFromMeta(String meta) {
    final m = meta.toLowerCase();
    if (m.contains('permit') || m.contains('report') || m.contains('pdf')) {
      return _DocKind.pdf;
    }
    if (m.contains('doc')) return _DocKind.doc;
    if (m.contains('img') || m.contains('photo') || m.contains('image')) {
      return _DocKind.img;
    }
    return _DocKind.other;
  }

  Icon _docTypeIcon(_DocKind kind) {
    switch (kind) {
      case _DocKind.pdf:
        return const Icon(Icons.picture_as_pdf);
      case _DocKind.doc:
        return const Icon(Icons.description);
      case _DocKind.img:
        return const Icon(Icons.image);
      case _DocKind.other:
      default:
        return const Icon(Icons.insert_drive_file);
    }
  }

  Widget _sectionHeader(String title, {VoidCallback? onAdd}) {
    return Container(
      color: Colors.grey.shade200,
      padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 10),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
          if (onAdd != null)
            Row(
              children: [
                if (_isUploading)
                  const Padding(
                    padding: EdgeInsets.only(right: 8),
                    child: SizedBox(
                      width: 16,
                      height: 16,
                      child: CircularProgressIndicator(strokeWidth: 2),
                    ),
                  ),
                IconButton(
                  icon: const Icon(Icons.add_circle_outline),
                  onPressed: onAdd,
                  tooltip: 'Upload',
                ),
              ],
            ),
        ],
      ),
    );
  }

  Widget _header() {
    return Container(
      color: Colors.white,
      padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
      child: const Row(
        children: [
          CircleAvatar(
            backgroundColor: Colors.green,
            child: Text("AE", style: TextStyle(color: Colors.white)),
          ),
          SizedBox(width: 12),
          Expanded(
            child: Text(
              "Documents",
              style: TextStyle(fontSize: 16, fontWeight: FontWeight.bold),
            ),
          ),
        ],
      ),
    );
  }

  Widget _docRow(_DocItem doc) {
    final hasFile = doc.pfid != null && doc.pfid!.isNotEmpty;
    final isBusy = hasFile && _downloading.contains(doc.pfid);

    return Container(
      padding: const EdgeInsets.all(12),
      margin: const EdgeInsets.symmetric(vertical: 6),
      decoration: BoxDecoration(
        border: Border.all(color: Colors.grey.shade300),
        borderRadius: BorderRadius.circular(6),
        color: Colors.white,
      ),
      child: Row(
        children: [
          Padding(
            padding: const EdgeInsets.only(right: 12),
            child: _docTypeIcon(doc.kind),
          ),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(doc.title,
                    style: const TextStyle(
                        fontWeight: FontWeight.w600, fontSize: 14)),
                const SizedBox(height: 4),
                Text("Date Uploaded: ${doc.uploadedDisplay}",
                    style:
                        const TextStyle(fontSize: 12, color: Colors.black54)),
                if (!hasFile)
                  const Padding(
                    padding: EdgeInsets.only(top: 4),
                    child: Text(
                      'No file attached yet',
                      style: TextStyle(fontSize: 11, color: Colors.orange),
                    ),
                  ),
              ],
            ),
          ),
          if (isBusy)
            const SizedBox(
              width: 20,
              height: 20,
              child: CircularProgressIndicator(strokeWidth: 2),
            )
          else ...[
            IconButton(
              icon: const Icon(Icons.remove_red_eye_outlined),
              tooltip: hasFile ? 'Preview' : 'No file attached',
              onPressed: hasFile
                  ? () => _downloadAndOpen(doc.pfid!, suggestedName: doc.title)
                  : null,
            ),
            IconButton(
              icon: const Icon(Icons.download),
              color: hasFile ? Colors.deepPurple : Colors.grey,
              tooltip: hasFile ? 'Download' : 'No file attached',
              onPressed: hasFile
                  ? () => _downloadAndOpen(doc.pfid!, suggestedName: doc.title)
                  : null,
            ),
            PopupMenuButton<String>(
              onSelected: (value) async {
                switch (value) {
                  case 'preview':
                    if (hasFile) {
                      await _downloadAndOpen(doc.pfid!,
                          suggestedName: doc.title);
                    }
                    break;
                  case 'rename':
                    // TODO: implement rename dialog + API (send `authorization` header)
                    break;
                  case 'delete':
                    // TODO: confirm + delete API (send `authorization` header), then refresh
                    break;
                }
              },
              itemBuilder: (context) => const [
                PopupMenuItem(value: 'preview', child: Text('Preview')),
                PopupMenuItem(value: 'rename', child: Text('Rename')),
                PopupMenuItem(value: 'delete', child: Text('Delete')),
              ],
            ),
          ],
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final content = _isLoading
        ? const Center(child: CircularProgressIndicator())
        : (_error != null
            ? Center(
                child: Padding(
                  padding: const EdgeInsets.all(16),
                  child: Text(
                    _error!,
                    textAlign: TextAlign.center,
                    style: const TextStyle(color: Colors.red),
                  ),
                ),
              )
            : ListView(
                physics: const AlwaysScrollableScrollPhysics(),
                padding: const EdgeInsets.all(16),
                children: [
                  _header(),
                  const SizedBox(height: 8),
                  _sectionHeader("Documents", onAdd: _showUploadOptions),
                  const SizedBox(height: 8),
                  if (_documents.isEmpty)
                    const Padding(
                      padding: EdgeInsets.all(12),
                      child: Text('No documents found.'),
                    )
                  else
                    ..._documents.map(_docRow),
                ],
              ));

    return Scaffold(
      body: RefreshIndicator(
        onRefresh: _fetchDocs,
        child: content,
      ),
    );
  }
}

enum _DocKind { pdf, doc, img, other }

class _DocItem {
  final String title;
  final String uploadedDisplay;
  final String? pfid; // nullable
  final _DocKind kind;

  _DocItem({
    required this.title,
    required this.uploadedDisplay,
    required this.pfid,
    required this.kind,
  });
}
