import 'dart:io';
import 'package:flutter/material.dart';
import 'package:dio/dio.dart';
import 'package:image_picker/image_picker.dart';
import 'package:cached_network_image/cached_network_image.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:photo_view/photo_view.dart';
import 'package:photo_view/photo_view_gallery.dart';
import 'dart:convert';
import 'package:http/http.dart' as http;
import '../services/upload_queue_service.dart';

class JobDetailsPage extends StatefulWidget {
  final Map<String, dynamic> job;
  final String token;

  const JobDetailsPage({super.key, required this.job, required this.token});

  @override
  _JobDetailsPageState createState() => _JobDetailsPageState();
}

class _JobDetailsPageState extends State<JobDetailsPage> {
  bool _isLoading = false;
  final List<Map<String, dynamic>> _files = [];
  final Set<String> _uniqueUrls = {};
  int _currentPage = 1;
  bool _hasMore = true;

  @override
  void initState() {
    super.initState();
    _loadInitialFiles();
  }

  Future<void> _loadInitialFiles() async {
    if (_isLoading) return;
    setState(() => _isLoading = true);

    try {
      final localFiles = await _fetchPendingUploads();
      final s3Files = await _fetchS3Files(page: 1);

      _addUniqueFiles(localFiles, isLocal: true);
      _addUniqueFiles(s3Files);

      setState(() {
        _currentPage = 2;
        _hasMore = s3Files.isNotEmpty;
      });
    } catch (e) {
      _showError('Error loading initial files: $e');
    } finally {
      setState(() => _isLoading = false);
    }
  }

  void _addUniqueFiles(List<Map<String, dynamic>> newFiles,
      {bool isLocal = false}) {
    for (var file in newFiles) {
      final identifier = isLocal ? file['path'] : file['url'];
      if (!_uniqueUrls.contains(identifier)) {
        _files.add(file);
        _uniqueUrls.add(identifier);
      }
    }
  }

  Future<List<Map<String, dynamic>>> _fetchPendingUploads() async {
    final prefs = await SharedPreferences.getInstance();
    final pendingUploads = prefs.getStringList('pending_uploads') ?? [];
    return pendingUploads
        .map((path) => {'type': 'local', 'path': path})
        .toList();
  }

  Future<List<Map<String, dynamic>>> _fetchS3Files({required int page}) async {
    final dio = Dio();
    final jobId = widget.job['id'].toString();
    final url =
        'https://aeihawaii.com/photoappsch/loginapi/getimagelisting/$jobId?page=$page';

    try {
      final response = await dio.get(
        url,
        options: Options(headers: {'authorization': widget.token}),
      );

      if (response.statusCode == 200) {
        final List<dynamic> data =
            response.data is String ? jsonDecode(response.data) : response.data;
        return data.map((file) => {
          'type': 's3',
          'url': file['link'],
          'thumbUrl': file['thumb_link'] ?? file['link'],
        }).toList();
      }
      throw Exception('Please upload photo');
    } catch (e) {
      _showError('Please upload photo');
      return [];
    }
  }

  Future<void> _loadMoreFiles() async {
    if (_isLoading || !_hasMore) return;

    setState(() => _isLoading = true);
    try {
      final newFiles = await _fetchS3Files(page: _currentPage);

      if (newFiles.isNotEmpty) {
        _addUniqueFiles(newFiles);
        setState(() {
          _currentPage++;
        });
      } else {
        setState(() {
          _hasMore = false;
        });
      }
    } catch (e) {
      _showError('Error loading more files: $e');
    } finally {
      setState(() => _isLoading = false);
    }
  }

  Future<void> _pickImageFromCamera() async {
    await _pickAndUploadImage(ImageSource.camera);
  }

  Future<void> _pickImageFromGallery() async {
    await _pickAndUploadImage(ImageSource.gallery);
  }

  Future<void> _pickAndUploadImage(ImageSource source) async {
    try {
      final picker = ImagePicker();
      final pickedFile = await picker.pickImage(source: source);

      if (pickedFile == null) return;

      final jobId = int.tryParse(widget.job['id'].toString()) ?? 0;
      final fileName = pickedFile.path.split('/').last;

      final added = await UploadQueueService.instance.enqueue(
        filePath: pickedFile.path,
        jobId: jobId,
        fileName: fileName,
      );

      if (mounted) {
        // Show local preview immediately
        setState(() {
          final localFile = {'type': 'local', 'path': pickedFile.path};
          _files.insert(0, localFile);
          _uniqueUrls.add(pickedFile.path);
        });

        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text(added ? 'Photo queued for upload' : 'Photo already queued')),
        );
      }
    } catch (e) {
      _showError('Error picking image: $e');
    }
  }

  void _showError(String message) {
    print(message);
    ScaffoldMessenger.of(context)
        .showSnackBar(SnackBar(content: Text(message)));
  }

  Widget _buildGrid() {
    return NotificationListener<ScrollNotification>(
      onNotification: (scrollInfo) {
        if (scrollInfo.metrics.pixels == scrollInfo.metrics.maxScrollExtent &&
            !_isLoading) {
          _loadMoreFiles();
        }
        return false;
      },
      child: GridView.builder(
        padding: const EdgeInsets.all(8.0),
        gridDelegate: SliverGridDelegateWithFixedCrossAxisCount(
          crossAxisCount: 4,
          crossAxisSpacing: 8,
          mainAxisSpacing: 8,
        ),
        itemCount: _files.length,
        itemBuilder: (context, index) {
          final file = _files[index];
          return GestureDetector(
            onTap: () => _showImageViewer(index),
            child: file['type'] == 'local'
                ? Image.file(
                    File(file['path']),
                    fit: BoxFit.cover,
                  )
                : CachedNetworkImage(
                    imageUrl: file['thumbUrl'] ?? file['url'],
                    placeholder: (context, url) => Container(
                      color: Colors.grey[200],
                      child: Center(child: CircularProgressIndicator()),
                    ),
                    errorWidget: (context, url, error) => Icon(Icons.error),
                    fit: BoxFit.cover,
                  ),
          );
        },
      ),
    );
  }

  void _showImageViewer(int index) {
    Navigator.push(
      context,
      MaterialPageRoute(
        builder: (_) => Scaffold(
          appBar: AppBar(),
          body: PhotoViewGallery.builder(
            itemCount: _files.length,
            builder: (context, index) {
              final file = _files[index];
              return PhotoViewGalleryPageOptions(
                imageProvider: file['type'] == 'local'
                    ? FileImage(File(file['path']))
                    : CachedNetworkImageProvider(file['url']),
                minScale: PhotoViewComputedScale.contained,
                maxScale: PhotoViewComputedScale.covered * 2.0,
              );
            },
            pageController: PageController(initialPage: index),
          ),
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(title: Text(widget.job['title'] ?? 'Job Details')),
      body: _isLoading && _files.isEmpty
          ? Center(child: CircularProgressIndicator())
          : _files.isEmpty
              ? Center(child: Text('No files available'))
              : _buildGrid(),
      floatingActionButton: Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                FloatingActionButton(
                  heroTag: 'camera',
                  onPressed: _pickImageFromCamera,
                  tooltip: 'Capture from Camera',
                  child: Icon(Icons.camera_alt),
                ),
                SizedBox(height: 12),
                FloatingActionButton(
                  heroTag: 'gallery',
                  onPressed: _pickImageFromGallery,
                  tooltip: 'Pick from Gallery',
                  child: Icon(Icons.photo_library),
                ),
              ],
            ),
    );
  }
}
