import 'package:flutter/material.dart';
import 'package:url_launcher/url_launcher.dart';
import 'job_details_edit_view.dart';

import 'job_notes_full_view.dart';
import 'job_photos_full_view.dart';
import 'job_docs_full_view.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:provider/provider.dart';
import '../token_provider.dart';
import 'customer_details_full_view.dart';
import 'package:flutter/gestures.dart';
import '../widgets/clock_in_button.dart';

class JobDetailsFullView extends StatefulWidget {
  final int? jobId;
  const JobDetailsFullView({super.key, this.jobId});

  @override
  State<JobDetailsFullView> createState() => _JobDetailsFullViewState();
}

class _JobDetailsFullViewState extends State<JobDetailsFullView>
    with SingleTickerProviderStateMixin {
  late TabController _tabController;

  Map<String, dynamic> jobDetails = {};
  bool isLoading = true;
  String _token = '';
  String? _userId;

  // NEW: role flag
  bool _isAdmin = false;

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 4, vsync: this);
    _loadTokenAndData();
  }

  @override
  void dispose() {
    _tabController.dispose();
    super.dispose();
  }

  Future<void> _loadTokenAndData() async {
    try {
      final token =
          Provider.of<TokenProvider>(context, listen: false).token ?? '';
      _token = token;

      final prefs = await SharedPreferences.getInstance();
      _userId = prefs.getString('user_id');

      // Try to get role quickly from prefs
      final role = prefs.getString('role');
      if (role != null && role != "none") {
        _isAdmin = role.toLowerCase() == 'admin';
      } else {
        //await _fetchAndCacheRoleFromApi();
      }
      await _fetchAndCacheRoleFromApi();
      if (_userId == null || _token.isEmpty) {
        debugPrint('Missing user ID or token');
        if (mounted) setState(() => isLoading = false);
        return;
      }

      await _loadJobDetails(_token);
    } catch (e) {
      debugPrint('Init error: $e');
      if (mounted) setState(() => isLoading = false);
    }
  }

  Future<void> _fetchAndCacheRoleFromApi() async {
    final idt = widget.jobId;
    if (idt == null) {
      debugPrint('jobId is null');
      if (mounted) setState(() => isLoading = false);
      return;
    }

    try {
      final uri =
          Uri.parse('https://aeihawaii.com/photoappsch/loginapinew/me/$idt');
      final res = await http.get(
        uri,
        headers: {
          'authorization': _token, // raw token (as your API expects)
          'Content-Type': 'application/json',
        },
      );

      if (!mounted) return;

      if (res.statusCode == 200) {
        final Map<String, dynamic> m = json.decode(res.body);
        final role = (m['role'] ?? '').toString().toLowerCase();

        setState(() {
          _isAdmin = role == 'admin';
        });

        final prefs = await SharedPreferences.getInstance();
        await prefs.setString('role', role);

        debugPrint('✅ Role fetched: $role');
      } else {
        debugPrint('❌ Role fetch failed: ${res.statusCode} ${res.body}');
      }
    } catch (e) {
      debugPrint('🔥 Role fetch error: $e');
    }
  }

  Future<void> _loadJobDetails(String token) async {
    if (!mounted) return;
    setState(() => isLoading = true);
    try {
      final id = widget.jobId;
      if (id == null) {
        debugPrint('jobId is null');
        if (mounted) setState(() => isLoading = false);
        return;
      }

      final uri = Uri.parse(
        'https://aeihawaii.com/photoappsch/loginapinew/jobdetailsfullviewstate/$id',
      );

      final response = await http.get(
        uri,
        headers: {
          'Content-Type': 'application/json',
          'authorization': token, // raw token per your API
        },
      );

      if (!mounted) return;

      if (response.statusCode == 200) {
        final Map<String, dynamic> data = json.decode(response.body);
        setState(() {
          jobDetails = data;
          isLoading = false;
        });
      } else {
        debugPrint(
            'Job detail API error: ${response.statusCode} -> ${response.body}');
        setState(() => isLoading = false);
      }
    } catch (e) {
      debugPrint('Error loading job details: $e');
      if (mounted) setState(() => isLoading = false);
    }
  }

  // ======== NEW: call Update API (admin-only) ========
  Future<bool> _submitJobUpdate({
    required int jobId,
    required Map<String, dynamic> payload,
  }) async {
    if (!_isAdmin) {
      if (!mounted) return false;
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Only admins can edit jobs.')),
      );
      return false;
    }

    try {
      final uri = Uri.parse(
          'https://aeihawaii.com/photoappsch/loginapinew/updatejob/$jobId');

      final res = await http.post(
        uri,
        headers: {
          'authorization': _token, // raw token per your API
          'Content-Type': 'application/json',
        },
        body: json.encode(payload),
      );

      if (res.statusCode == 200) {
        // Success – refresh details
        await _loadJobDetails(_token);
        if (!mounted) return true;
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Job updated successfully.')),
        );
        return true;
      } else {
        debugPrint('Update failed: ${res.statusCode} ${res.body}');
        if (!mounted) return false;
        final err = _safeApiError(res.body);
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Update failed: $err')),
        );
        return false;
      }
    } catch (e) {
      debugPrint('Update error: $e');
      if (!mounted) return false;
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Network error: $e')),
      );
      return false;
    }
  }

  String _safeApiError(String body) {
    try {
      final m = json.decode(body);
      if (m is Map && m['error'] is String) return m['error'];
    } catch (_) {}
    return 'Unexpected error';
  }

  bool get _canEdit => _isAdmin;

  @override
  Widget build(BuildContext context) {
    if (isLoading) {
      return Scaffold(
        appBar: AppBar(
          backgroundColor: Colors.green,
          title: const Text("Job Details"),
        ),
        body: const Center(child: CircularProgressIndicator()),
      );
    }

    // Compute local values BEFORE returning the widget tree
    final customerName = (jobDetails["customerName"] ?? '—').toString();
    final jobIdInt = _safeInt(jobDetails["jobId"]) ?? (widget.jobId ?? 0);
    final customerIdInt = _safeInt(jobDetails["customerId"]) ?? 0;

    return Scaffold(
      appBar: AppBar(
        backgroundColor: Colors.green,
        title: const Text("Job Details"),
        actions: [
          IconButton(icon: const Icon(Icons.help_outline), onPressed: () {}),
          PopupMenuButton<String>(
            icon: const Icon(Icons.menu),
            onSelected: (value) {
              // TODO: handle actions
            },
            itemBuilder: (BuildContext context) {
              return ['Edit Job', 'Cancel Job']
                  .map((choice) => PopupMenuItem<String>(
                        value: choice,
                        child: Text(choice),
                      ))
                  .toList();
            },
          ),
        ],
      ),
      body: Column(
        children: [
          // Header with Customer Name
          Container(
            color: Colors.white,
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
            child: Row(
              children: [
                const Icon(Icons.person, size: 36, color: Colors.black54),
                const SizedBox(width: 12),
                Expanded(
                  child: Text(
                    customerName,
                    maxLines: 2,
                    overflow: TextOverflow.ellipsis,
                    style: const TextStyle(
                        fontSize: 18, fontWeight: FontWeight.bold),
                  ),
                ),
                if (!_canEdit)
                  const Padding(
                    padding: EdgeInsets.only(left: 8.0),
                    child: Text(
                      '',
                      style: TextStyle(color: Colors.black45, fontSize: 12),
                    ),
                  ),
              ],
            ),
          ),

          // Clock In/Out Banner (ENH-034)
          Padding(
            padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 4),
            child: ClockInButton(
              jobId: jobIdInt,
              customerId: customerIdInt,
              customerName: customerName,
            ),
          ),

          // Tabs
          Container(
            color: Colors.grey.shade200,
            child: TabBar(
              controller: _tabController,
              labelColor: Colors.green,
              unselectedLabelColor: Colors.black54,
              tabs: const [
                Tab(text: 'Details'),
                Tab(text: 'Notes'),
                Tab(text: 'Photos'),
                Tab(text: 'Docs'),
              ],
            ),
          ),

          // Tab Views
          Expanded(
            child: TabBarView(
              controller: _tabController,
              children: [
                _buildDetailsTab(jobIdInt, customerName, customerIdInt),
                JobNotesFullView(
                  customerId: customerIdInt,
                  jobId: jobIdInt,
                ),
                JobPhotosFullView(jobId: jobIdInt),
                JobDocsFullView(jobId: jobIdInt),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildDetailsTab(
      int jobIdInt, String customerName, int customerIdInt) {
    final jobType = (jobDetails["jobType"] ?? '').toString();
    final status = (jobDetails["status"] ?? '').toString();
    final date = (jobDetails["date"] ?? '').toString();
    final time = (jobDetails["time"] ?? '').toString();
    final jobIdStr = (jobDetails["jobId"] ?? '').toString();
    final technicians = (jobDetails["technicians"] ?? '').toString();
    final address = (jobDetails["serviceLocation"] ?? '').toString();

    // ---- description: support multiple JSON shapes ----
    final descriptionObj = jobDetails["description"];
    List<dynamic> descriptionFields = [];
    String fallbackDesc = '';

    if (descriptionObj is Map && descriptionObj["fields"] is List) {
      // shape: { "description": { "fields": [ {label, value}, ... ] } }
      descriptionFields = descriptionObj["fields"] as List<dynamic>;
    } else if (descriptionObj is List) {
      // shape: "description": [ {label, value}, ... ]
      descriptionFields = descriptionObj;
    } else if (descriptionObj != null) {
      // plain string, last fallback
      fallbackDesc = descriptionObj.toString();
    }

    final customerid = int.tryParse('${jobDetails["customer_id"]}') ??
        double.tryParse('${jobDetails["customer_id"]}')?.toInt() ??
        customerIdInt;
    final jid = widget.jobId ?? jobIdInt;

    return RefreshIndicator(
      onRefresh: () => _loadJobDetails(_token),
      child: ListView(
        physics: const AlwaysScrollableScrollPhysics(),
        padding: const EdgeInsets.all(16),
        children: [
          // Scheduled Service Date Section
          Container(
            padding: const EdgeInsets.all(12),
            color: Colors.grey.shade200,
            child: Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                const Text(
                  "Scheduled Service Date",
                  style: TextStyle(fontWeight: FontWeight.bold, fontSize: 16),
                ),

                // Show EDIT only if admin
                if (_canEdit)
                  ElevatedButton(
                    onPressed: () async {
                      final result = await Navigator.push(
                        context,
                        MaterialPageRoute(
                          builder: (context) => JobDetailsEditView(
                            customerName: customerName,
                            initialDate: DateTime.now(),
                            initialStartTime: "9:00 AM",
                            initialEndTime: "10:00 AM",
                            staffList: const [
                              "Radford Nakamura",
                              "Oliver Galvan",
                              "Wesley Bahilot",
                            ],
                          ),
                        ),
                      );

                      if (!mounted) return;

                      if (result is Map<String, dynamic>) {
                        await _submitJobUpdate(
                            jobId: jobIdInt, payload: result);
                        return;
                      }

                      if (result == "updated") {
                        await _loadJobDetails(_token);
                      }
                    },
                    style: ElevatedButton.styleFrom(
                      backgroundColor: Colors.black,
                      padding: const EdgeInsets.symmetric(
                        horizontal: 16,
                        vertical: 4,
                      ),
                    ),
                    child: const Text(
                      "Edit",
                      style: TextStyle(color: Colors.white, fontSize: 14),
                    ),
                  ),
              ],
            ),
          ),
          const SizedBox(height: 12),
          _buildDetailRow(Icons.flag,
              [jobType, status].where((s) => s.isNotEmpty).join(' | ')),
          _buildDetailRow(Icons.calendar_today, date),
          _buildDetailRow(Icons.access_time, time),
          _buildDetailRow(Icons.tag, jobIdStr),
          _buildDetailRow(Icons.group, technicians),
          _buildDetailRow(
            Icons.person,
            RichText(
              text: TextSpan(
                children: [
                  const TextSpan(
                    text: 'Customer: ',
                    style: TextStyle(color: Colors.black),
                  ),
                  TextSpan(
                    text: (jobDetails["customerName"] ?? '—').toString(),
                    style: const TextStyle(
                      color: Colors.blue,
                      decoration: TextDecoration.underline,
                    ),
                    recognizer: TapGestureRecognizer()
                      ..onTap = () {
                        Navigator.push(
                          context,
                          MaterialPageRoute(
                            builder: (_) => CustomerDetailsFullView(
                              jobId: jid,
                              customerId: customerid,
                            ),
                          ),
                        );
                      },
                  ),
                ],
              ),
            ),
          ),

          const SizedBox(height: 16),

          // Service Location
          _buildSectionTitle("Service Location"),
        InkWell(
          onTap: () async {
            final addr = address.trim();
            if (addr.isEmpty) return;

            // Proper Google Maps URL (safe encoding)
            final googleMaps = Uri.https(
              'www.google.com',
              '/maps/search/',
              {
                'api': '1',
                'query': addr,
              },
            );

            try {
              // Best mode for Android 13/14 + iOS + Web
              final opened =
                  await launchUrl(googleMaps, mode: LaunchMode.platformDefault);

              // Android fallback (fixes Galaxy S23 issue)
              if (!opened) {
                final geo =
                    Uri.parse('geo:0,0?q=${Uri.encodeComponent(addr)}');
                await launchUrl(geo, mode: LaunchMode.externalApplication);
              }
            } catch (e) {
              debugPrint('Map launch error: $e');
              if (!mounted) return;
              ScaffoldMessenger.of(context).showSnackBar(
                const SnackBar(content: Text('Unable to open Maps')),
              );
            }
          },
          child: Padding(
            padding: const EdgeInsets.symmetric(vertical: 6),
            child: Text(
              address.trim().isEmpty ? '—' : address, // ✅ use address here
              style: TextStyle(
                color: address.trim().isEmpty ? Colors.black87 : Colors.green,
                decoration: address.trim().isEmpty
                    ? TextDecoration.none
                    : TextDecoration.underline,
                fontSize: 16,
              ),
            ),

          ),
        ),

          const SizedBox(height: 20),

          const Text(
            "Job Description",
            style: TextStyle(
              fontSize: 16,
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 10),

// ---- SIMPLE MULTILINE "label value" LINES ----
          if (descriptionFields.isNotEmpty) ...[
            for (final f in descriptionFields) ...[
              Builder(
                builder: (_) {
                  final label = (f['label'] ?? '').toString().trim();
                  final value = (f['value'] ?? '').toString().trim();

                  return Padding(
                    padding: const EdgeInsets.only(bottom: 4),
                    child: RichText(
                      text: TextSpan(
                        children: [
                          // BOLD LABEL (same line)
                          TextSpan(
                            text: "$label ",
                            style: const TextStyle(
                              fontSize: 14,
                              fontWeight: FontWeight.bold,
                              color: Colors.black,
                            ),
                          ),

                          // NORMAL VALUE
                          TextSpan(
                            text: value,
                            style: const TextStyle(
                              fontSize: 14,
                              color: Colors.black87,
                            ),
                          ),
                        ],
                      ),
                    ),
                  );
                },
              ),
            ],
          ] else if (fallbackDesc.isNotEmpty) ...[
            Text(
              fallbackDesc,
              style: const TextStyle(
                fontSize: 14,
                height: 1.3,
              ),
              softWrap: true,
            ),
          ] else ...[
            const Text(
              '—',
              style: TextStyle(fontSize: 14),
            ),
          ],
        ],
      ),
    );
  }

  Widget _buildDetailRow(IconData icon, Object? content) {
    final Widget child;
    if (content is Widget) {
      child = content;
    } else {
      final s = (content?.toString() ?? '').trim();
      child = Text(s.isEmpty ? '—' : s);
    }

    return Row(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Icon(icon, color: Colors.blue),
        const SizedBox(width: 10),
        Expanded(child: child),
      ],
    );
  }

  Widget _buildSectionTitle(String title) {
    return Container(
      color: Colors.grey.shade200,
      padding: const EdgeInsets.all(8),
      margin: const EdgeInsets.symmetric(vertical: 4),
      child: Text(title,
          style: const TextStyle(fontSize: 16, fontWeight: FontWeight.bold)),
    );
  }

  /// Safely parse dynamic -> int?
  int? _safeInt(dynamic v) {
    if (v == null) return null;
    if (v is int) return v;
    return int.tryParse(v.toString());
  }
}
