import 'package:flutter/material.dart';
import 'package:intl/intl.dart';

class JobDetailsEditView extends StatefulWidget {
  final String customerName;
  final DateTime initialDate;
  final String initialStartTime; // e.g., "9:00 AM"
  final String initialEndTime; // e.g., "10:00 AM"
  final List<String> staffList;

  // Optional prefill fields (backward compatible)
  final String? initialStatus; // e.g., "Scheduled"
  final List<String>? initialTechnicians; // subset of staffList
  final String? initialAddress; // service location
  final String? initialDescription; // job description

  // Optional admin guard (defaults to true; parent can pass false)
  final bool isAdmin;

  const JobDetailsEditView({
    super.key,
    required this.customerName,
    required this.initialDate,
    required this.initialStartTime,
    required this.initialEndTime,
    required this.staffList,
    this.initialStatus,
    this.initialTechnicians,
    this.initialAddress,
    this.initialDescription,
    this.isAdmin = true,
  });

  @override
  _JobDetailsEditViewState createState() => _JobDetailsEditViewState();
}

class _JobDetailsEditViewState extends State<JobDetailsEditView> {
  late DateTime selectedDate;
  late String selectedStartTime;
  late String selectedEndTime;

  // NEW: status + techs + text fields
  final List<String> statusOptions = const [
    'Scheduled',
    'In Progress',
    'Completed',
    'Canceled',
  ];
  late String selectedStatus;
  late Set<String> selectedTechs;

  final TextEditingController addressCtrl = TextEditingController();
  final TextEditingController descCtrl = TextEditingController();

  final List<String> timeSlots = const [
    '8:00 AM',
    '8:30 AM',
    '9:00 AM',
    '9:30 AM',
    '10:00 AM',
    '10:30 AM',
    '11:00 AM',
    '11:30 AM',
    '12:00 PM',
    '1:00 PM',
    '1:30 PM',
    '2:00 PM',
    '2:30 PM',
    '3:00 PM',
    '3:30 PM',
    '4:00 PM',
  ];

  @override
  void initState() {
    super.initState();
    selectedDate = widget.initialDate;
    selectedStartTime = widget.initialStartTime;
    selectedEndTime = widget.initialEndTime;

    selectedStatus = widget.initialStatus ?? 'Scheduled';
    selectedTechs = {...(widget.initialTechnicians ?? const <String>[])};

    addressCtrl.text = widget.initialAddress ?? '';
    descCtrl.text = widget.initialDescription ?? '';
  }

  @override
  void dispose() {
    addressCtrl.dispose();
    descCtrl.dispose();
    super.dispose();
  }

  Future<void> _selectDate(BuildContext context) async {
    final DateTime? picked = await showDatePicker(
      context: context,
      initialDate: selectedDate,
      firstDate: DateTime(2024),
      lastDate: DateTime(2030),
    );
    if (picked != null && picked != selectedDate) {
      setState(() => selectedDate = picked);
    }
  }

  // --- Helpers ---
  String _formatDateYMD(DateTime d) => DateFormat('yyyy-MM-dd').format(d);

  /// Convert "9:00 AM" → "09:00", "12:30 PM" → "12:30", "12:00 AM" → "00:00"
  String _to24h(String ampm) {
    try {
      final dt = DateFormat.jm().parse(ampm); // uses device locale for AM/PM
      return DateFormat('HH:mm').format(dt);
    } catch (_) {
      // Fallback: if already HH:mm, return as-is
      final ok = RegExp(r'^\d{2}:\d{2}$');
      if (ok.hasMatch(ampm)) return ampm;
      return '09:00';
    }
  }

  void _toggleTech(String name, bool? checked) {
    setState(() {
      if (checked == true) {
        selectedTechs.add(name);
      } else {
        selectedTechs.remove(name);
      }
    });
  }

  Future<void> _onSave() async {
    if (!widget.isAdmin) {
      if (!mounted) return;
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Only admins can edit jobs.')),
      );
      return;
    }

    final startMins = _parseAmPmToMinutes(selectedStartTime);
    final endMins = _parseAmPmToMinutes(selectedEndTime);

    if (startMins == null || endMins == null) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Invalid time format.')),
      );
      return;
    }

    if (startMins >= endMins) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Start time must be before end time.')),
      );
      return;
    }

// If you need 24h strings for the API:
    final start24 = _minutesTo24h(startMins); // or _to24h(selectedStartTime)
    final end24 = _minutesTo24h(endMins);

    // Build payload for PHP update endpoint
    final payload = <String, dynamic>{
      "service_date": _formatDateYMD(selectedDate), // YYYY-MM-DD
      "start_time": start24, // HH:mm (24h)
      "end_time": end24, // HH:mm (24h)
      "technicians": selectedTechs.isEmpty
          ? widget.staffList // default: send full list if nothing selected
          : selectedTechs.toList(),
      "status": selectedStatus,
    };

    // Optional fields (only include if provided)
    if (addressCtrl.text.trim().isNotEmpty) {
      payload["service_location"] = addressCtrl.text.trim();
    }
    if (descCtrl.text.trim().isNotEmpty) {
      payload["description"] = descCtrl.text.trim();
    }

    // Return payload to the parent (JobDetailsFullView), which will call the API
    if (!mounted) return;
    Navigator.pop(context, payload);
  }

  @override
  Widget build(BuildContext context) {
    final datePretty = DateFormat('MM/dd/yyyy').format(selectedDate);

    return Scaffold(
      appBar: AppBar(
        backgroundColor: Colors.green,
        title: const Text("Edit Job Details"),
      ),
      body: ListView(
        padding: const EdgeInsets.all(16),
        children: [
          // Customer Header
          Container(
            color: Colors.white,
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
            child: Row(
              children: [
                const Icon(Icons.person, size: 36, color: Colors.black54),
                const SizedBox(width: 12),
                Expanded(
                  child: Text(
                    widget.customerName,
                    style: const TextStyle(
                        fontSize: 18, fontWeight: FontWeight.bold),
                  ),
                ),
                if (!widget.isAdmin)
                  const Text('(admin only)',
                      style: TextStyle(color: Colors.black45)),
              ],
            ),
          ),
          const SizedBox(height: 16),

          // Date & Time
          _sectionHeader("Date & Arrival Time"),
          const SizedBox(height: 12),

          // Date Picker
          GestureDetector(
            onTap: () => _selectDate(context),
            child: Container(
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 14),
              decoration: BoxDecoration(
                border: Border.all(color: Colors.black54),
                borderRadius: BorderRadius.circular(8),
              ),
              child: Row(
                mainAxisAlignment: MainAxisAlignment.spaceBetween,
                children: [
                  Text(datePretty),
                  const Icon(Icons.calendar_today, color: Colors.black54),
                ],
              ),
            ),
          ),
          const SizedBox(height: 16),

          // Time Start & End
          Row(
            children: [
              Expanded(
                child: _buildDropdown(
                  "Time Start",
                  selectedStartTime,
                  (value) => setState(() => selectedStartTime = value!),
                ),
              ),
              const SizedBox(width: 12),
              Expanded(
                child: _buildDropdown(
                  "Time End",
                  selectedEndTime,
                  (value) => setState(() => selectedEndTime = value!),
                ),
              ),
            ],
          ),
          const SizedBox(height: 20),

          // Status
          _sectionHeader("Status"),
          const SizedBox(height: 8),
          Container(
            padding: const EdgeInsets.symmetric(horizontal: 12),
            decoration: BoxDecoration(
              border: Border.all(color: Colors.black54),
              borderRadius: BorderRadius.circular(8),
            ),
            child: DropdownButton<String>(
              value: selectedStatus,
              isExpanded: true,
              underline: const SizedBox(),
              items: statusOptions
                  .map((s) => DropdownMenuItem(value: s, child: Text(s)))
                  .toList(),
              onChanged: (v) =>
                  setState(() => selectedStatus = v ?? selectedStatus),
            ),
          ),
          const SizedBox(height: 20),

          // Staff (multi-select)
          _sectionHeader("Technicians"),
          const SizedBox(height: 8),
          ...widget.staffList.map((name) {
            final checked = selectedTechs.contains(name);
            return CheckboxListTile(
              dense: true,
              contentPadding: EdgeInsets.zero,
              value: checked,
              onChanged: (v) => _toggleTech(name, v),
              title: Text(name),
              controlAffinity: ListTileControlAffinity.leading,
            );
          }),

          const SizedBox(height: 20),

          // Address
          _sectionHeader("Service Location"),
          const SizedBox(height: 8),
          TextField(
            controller: addressCtrl,
            decoration: _inputDecoration("Enter address (optional)"),
            minLines: 1,
            maxLines: 2,
          ),
          const SizedBox(height: 16),

          // Description
          _sectionHeader("Job Description"),
          const SizedBox(height: 8),
          TextField(
            controller: descCtrl,
            decoration: _inputDecoration("Enter description (optional)"),
            minLines: 2,
            maxLines: 5,
          ),

          const SizedBox(height: 24),

          // Save Button
          Center(
            child: ElevatedButton(
              style: ElevatedButton.styleFrom(
                backgroundColor: Colors.black,
                padding:
                    const EdgeInsets.symmetric(horizontal: 40, vertical: 14),
              ),
              onPressed: _onSave,
              child: const Text("Save", style: TextStyle(fontSize: 16)),
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildDropdown(
      String label, String value, ValueChanged<String?> onChanged) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(label, style: const TextStyle(fontWeight: FontWeight.bold)),
        const SizedBox(height: 6),
        Container(
          padding: const EdgeInsets.symmetric(horizontal: 12),
          decoration: BoxDecoration(
            border: Border.all(color: Colors.black54),
            borderRadius: BorderRadius.circular(8),
          ),
          child: DropdownButton<String>(
            value: value,
            isExpanded: true,
            underline: const SizedBox(),
            items: timeSlots
                .map((slot) => DropdownMenuItem<String>(
                      value: slot,
                      child: Text(slot),
                    ))
                .toList(),
            onChanged: onChanged,
          ),
        ),
      ],
    );
  }

  Widget _sectionHeader(String title) {
    return Container(
      color: Colors.grey.shade200,
      padding: const EdgeInsets.all(12),
      child: Text(
        title,
        style: const TextStyle(fontWeight: FontWeight.bold, fontSize: 16),
      ),
    );
  }

  InputDecoration _inputDecoration(String hint) => InputDecoration(
        hintText: hint,
        border: OutlineInputBorder(borderRadius: BorderRadius.circular(8)),
        isDense: true,
        contentPadding:
            const EdgeInsets.symmetric(horizontal: 12, vertical: 12),
      );
}

/// Parse "h:mm AM/PM" safely → total minutes since midnight.
/// Returns null if the input doesn't match.
int? _parseAmPmToMinutes(String s) {
  final re =
      RegExp(r'^\s*(\d{1,2}):(\d{2})\s*([AP]M)\s*$', caseSensitive: false);
  final m = re.firstMatch(s);
  if (m == null) return null;

  var h = int.parse(m.group(1)!);
  final mins = int.parse(m.group(2)!);
  final mer = m.group(3)!.toUpperCase();

  if (h == 12) h = 0; // 12:xx AM → 0:xx ; 12 PM handled by +12 below
  if (mer == 'PM') h += 12;

  return h * 60 + mins; // 0..1439
}

/// Minutes → "HH:mm" (24h)
String _minutesTo24h(int minutes) {
  final h = (minutes ~/ 60).toString().padLeft(2, '0');
  final m = (minutes % 60).toString().padLeft(2, '0');
  return '$h:$m';
}

/// Convert "h:mm AM/PM" → "HH:mm" using the same robust parser.
String _to24h(String ampm) {
  final mins = _parseAmPmToMinutes(ampm);
  if (mins == null) return '09:00'; // safe fallback
  return _minutesTo24h(mins);
}
