// lib/pages/job_costs_full_view.dart
import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';

import '../token_provider.dart';
import 'edit_job_cost_page.dart';

class JobCostsFullView extends StatefulWidget {
  final int jobId;

  /// Optional: if you already computed permission in parent,
  /// pass it here to override local role check.
  final bool? canEditOverride;

  const JobCostsFullView({
    super.key,
    required this.jobId,
    this.canEditOverride,
  });

  @override
  State<JobCostsFullView> createState() => _JobCostsFullViewState();
}

class _JobCostsFullViewState extends State<JobCostsFullView> {
  bool _isLoading = true;
  String? _error;

  // Display fields
  String price = '—';
  String tax = '—';
  String subtotal = '—';
  String method = '—';
  String comments = '—';

  bool _canEdit = false; // UI gate (server still enforces perms)
  String _token = '';

  String get _fetchApi =>
      'https://aeihawaii.com/photoappsch/loginapinew/jobcostsfullviewstate/${widget.jobId}';

  /// 🔧 Adjust if your update route differs
  String get _updateApi =>
      'https://aeihawaii.com/photoappsch/loginapinew//updatecost/${widget.jobId}';

  @override
  void initState() {
    super.initState();
    _bootstrap();
  }

  Future<void> _bootstrap() async {
    // token + canEdit (role=admin) check
    final token =
        Provider.of<TokenProvider>(context, listen: false).token ?? '';
    _token = token;

    await _resolveEditPermission();
    await _fetchCosts();
  }

  Future<void> _resolveEditPermission() async {
    if (widget.canEditOverride != null) {
      _canEdit = widget.canEditOverride!;
      return;
    }
    try {
      final prefs = await SharedPreferences.getInstance();
      final role = (prefs.getString('role') ?? '').toLowerCase();
      _canEdit = role == 'admin';
    } catch (_) {
      _canEdit = false;
    }
    if (mounted) setState(() {});
  }

  Future<void> _fetchCosts() async {
    setState(() {
      _isLoading = true;
      _error = null;
    });

    try {
      if (_token.isEmpty) throw Exception('Missing token');

      final resp = await http.get(
        Uri.parse(_fetchApi),
        headers: {
          'Content-Type': 'application/json',
          'authorization': _token, // raw token (no "Bearer ")
        },
      );

      if (resp.statusCode == 401 || resp.statusCode == 403) {
        throw Exception('Session expired or invalid token.');
      }
      if (resp.statusCode != 200) {
        throw Exception('HTTP ${resp.statusCode}');
      }

      final decoded = json.decode(resp.body);
      Map<String, dynamic>? obj;
      if (decoded is Map<String, dynamic>) {
        obj = decoded;
      } else if (decoded is List &&
          decoded.isNotEmpty &&
          decoded.first is Map) {
        obj = (decoded.first as Map).cast<String, dynamic>();
      }
      if (obj == null) throw Exception('Unexpected response shape');

      // Map flexible API field names
      final newPrice = _asString(obj['price']);
      final newMethod =
          _asString(obj['method']) ?? _asString(obj['mop']); // alias
      final newDesc = _asString(obj['description']) ??
          _asString(obj['desc']) ??
          _asString(obj['addwork_desc']); // aliases
      final newTax = _asString(obj['tax']);
      final newSubtotal = _asString(obj['subtotal']);

      setState(() {
        price = newPrice ?? '—';
        method = newMethod ?? '—';
        comments = (newDesc?.isNotEmpty ?? false) ? newDesc! : '—';
        tax = newTax ?? '—';
        subtotal = newSubtotal ?? '—';
        _isLoading = false;
      });
    } catch (e) {
      setState(() {
        _error = 'Failed to load job costs: $e';
        _isLoading = false;
      });
    }
  }

  String? _asString(dynamic v) {
    if (v == null) return null;
    final s = v.toString().trim();
    if (s.isEmpty || s.toLowerCase() == 'null') return null;
    return s;
  }

  String? _numOnlyOrNull(String? s) {
    if (s == null) return null;
    final cleaned = s.replaceAll(RegExp(r'[^0-9.\-]'), '');
    return cleaned.isEmpty ? null : cleaned;
  }

  Future<void> _openEditPage() async {
    // Pass current values to the editor (if your page supports it)
    final result = await Navigator.push(
      context,
      MaterialPageRoute(
        builder: (_) => EditJobCostPage(
          initialPrice: price == '—' ? '' : price,
          initialTax: tax == '—' ? '' : tax,
          initialSubtotal: subtotal == '—' ? '' : subtotal,
          initialPaymentMethod: method == '—' ? '' : method,
          initialComment: comments == '—' ? '' : comments,
        ),
      ),
    );

    if (!mounted) return;
    if (result is Map) {
      // Build payload for API and send
      final payload = _buildUpdatePayload(result);
      final ok = await _submitCostUpdate(payload);
      if (ok) await _fetchCosts(); // refresh from server truth
    }
  }

  Map<String, dynamic> _buildUpdatePayload(Map result) {
    // Accept fields from editor:
    // 'price', 'tax', 'subtotal', 'payment', 'comment'
    final p = <String, dynamic>{};

    final priceStr = _numOnlyOrNull(_asString(result['price']));
    final taxStr = _numOnlyOrNull(_asString(result['tax']));
    final subStr = _numOnlyOrNull(_asString(result['subtotal']));
    final payStr = _asString(result['payment']);
    final comStr = _asString(result['comment']);

    if (priceStr != null) p['price'] = priceStr;
    if (taxStr != null) p['tax'] = taxStr;
    if (subStr != null) p['subtotal'] = subStr;

    // Provide both new and legacy keys to be safe
    if (payStr != null) {
      p['method'] = payStr;
      p['mop'] = payStr; // alias for older backend
    }
    if (comStr != null) {
      p['description'] = comStr;
      p['addwork_desc'] = comStr; // alias for older backend
      p['desc'] = comStr; // another alias just in case
    }

    return p;
  }

  Future<bool> _submitCostUpdate(Map<String, dynamic> payload) async {
    if (!_canEdit) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Only admins can edit costs.')),
      );
      return false;
    }
    if (_token.isEmpty) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Missing token.')),
      );
      return false;
    }

    try {
      final res = await http.post(
        Uri.parse(_updateApi),
        headers: {
          'authorization': _token, // raw token per your API
          'Content-Type': 'application/json',
        },
        body: json.encode(payload),
      );

      if (res.statusCode == 200) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Job cost updated.')),
        );
        return true;
      } else {
        final err = _safeApiError(res.body);
        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text('Update failed: $err')),
        );
        return false;
      }
    } catch (e) {
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Network error: $e')),
      );
      return false;
    }
  }

  String _safeApiError(String body) {
    try {
      final m = json.decode(body);
      if (m is Map && m['error'] is String) return m['error'];
    } catch (_) {}
    return 'Unexpected error';
  }

  @override
  Widget build(BuildContext context) {
    final body = _isLoading
        ? const Center(child: CircularProgressIndicator())
        : (_error != null
            ? Center(
                child: Padding(
                  padding: const EdgeInsets.all(16),
                  child: Text(
                    _error!,
                    textAlign: TextAlign.center,
                    style: const TextStyle(color: Colors.red),
                  ),
                ),
              )
            : ListView(
                physics: const AlwaysScrollableScrollPhysics(),
                padding: const EdgeInsets.all(16),
                children: [
                  _sectionHeader("Job Costs",
                      onEdit: _canEdit ? _openEditPage : null),
                  const SizedBox(height: 12),
                  _costRow("List Price of System (\$)", price),
                  _costRow("State Excise Tax", tax),
                  _costRow("Sub Total", subtotal),
                  const SizedBox(height: 12),
                  _costRow("Method of Payment", method),
                  const SizedBox(height: 20),
                  const Text(
                    "Additional Comments/Requirements",
                    style: TextStyle(fontWeight: FontWeight.bold),
                  ),
                  const SizedBox(height: 8),
                  Text(comments, style: const TextStyle(fontSize: 14)),
                ],
              ));

    return Scaffold(
      body: RefreshIndicator(
        onRefresh: _fetchCosts,
        child: body,
      ),
    );
  }

  Widget _sectionHeader(String title, {VoidCallback? onEdit}) {
    return Container(
      color: Colors.grey.shade200,
      padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 10),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
          if (onEdit != null)
            ElevatedButton(
              onPressed: onEdit,
              style: ElevatedButton.styleFrom(
                backgroundColor: Colors.black,
                minimumSize: const Size(50, 26),
              ),
              child: const Text("Edit", style: TextStyle(fontSize: 12)),
            ),
        ],
      ),
    );
  }

  Widget _costRow(String label, String value) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 6),
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          Expanded(
              child: Text(label,
                  style: const TextStyle(
                      fontSize: 14, fontWeight: FontWeight.w500))),
          const SizedBox(width: 8),
          Text(value,
              textAlign: TextAlign.right,
              style:
                  const TextStyle(fontSize: 14, fontWeight: FontWeight.w500)),
        ],
      ),
    );
  }
}
