// lib/pages/in_app_doc_viewer.dart
import 'dart:io';
import 'package:flutter/material.dart';
import 'package:flutter_pdfview/flutter_pdfview.dart';
import 'package:open_filex/open_filex.dart';

/// In-app document viewer.
///
/// * **PDF** files are rendered natively with pinch-to-zoom via flutter_pdfview.
/// * **Word / other** files fall back to the system viewer (open_filex).
class InAppDocViewer extends StatefulWidget {
  final String filePath;
  final String title;

  const InAppDocViewer({
    super.key,
    required this.filePath,
    required this.title,
  });

  @override
  State<InAppDocViewer> createState() => _InAppDocViewerState();
}

class _InAppDocViewerState extends State<InAppDocViewer> {
  int _totalPages = 0;
  int _currentPage = 0;
  PDFViewController? _pdfController;

  bool get _isPdf =>
      widget.filePath.toLowerCase().endsWith('.pdf');

  @override
  void initState() {
    super.initState();
    if (!_isPdf) {
      // For non-PDF files, open via system viewer and pop back
      WidgetsBinding.instance.addPostFrameCallback((_) async {
        final result = await OpenFilex.open(widget.filePath);
        if (mounted) {
          if (result.type != ResultType.done) {
            ScaffoldMessenger.of(context).showSnackBar(
              SnackBar(content: Text('Could not open file: ${result.message}')),
            );
          }
          Navigator.pop(context);
        }
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    // Non-PDF: show brief loading while system viewer opens
    if (!_isPdf) {
      return Scaffold(
        appBar: AppBar(
          backgroundColor: Colors.green,
          title: Text(widget.title),
        ),
        body: const Center(child: CircularProgressIndicator()),
      );
    }

    // PDF viewer with pinch-to-zoom
    return Scaffold(
      appBar: AppBar(
        backgroundColor: Colors.green,
        title: Text(
          widget.title,
          overflow: TextOverflow.ellipsis,
        ),
        actions: [
          if (_totalPages > 0)
            Center(
              child: Padding(
                padding: const EdgeInsets.only(right: 16),
                child: Text(
                  '${_currentPage + 1} / $_totalPages',
                  style: const TextStyle(fontSize: 14),
                ),
              ),
            ),
        ],
      ),
      body: PDFView(
        filePath: widget.filePath,
        enableSwipe: true,
        swipeHorizontal: false,
        autoSpacing: true,
        pageFling: true,
        pageSnap: true,
        fitPolicy: FitPolicy.BOTH,
        onRender: (pages) {
          if (mounted) {
            setState(() => _totalPages = pages ?? 0);
          }
        },
        onViewCreated: (controller) {
          _pdfController = controller;
        },
        onPageChanged: (page, total) {
          if (mounted) {
            setState(() {
              _currentPage = page ?? 0;
              _totalPages = total ?? 0;
            });
          }
        },
        onError: (error) {
          debugPrint('PDF render error: $error');
          if (mounted) {
            ScaffoldMessenger.of(context).showSnackBar(
              SnackBar(content: Text('Error loading PDF: $error')),
            );
          }
        },
      ),
    );
  }
}
