import 'package:flutter/material.dart';
import 'package:flutter_pdfview/flutter_pdfview.dart';
import 'package:path_provider/path_provider.dart';
import 'package:http/http.dart' as http;
import 'dart:io';

class HelpPage extends StatefulWidget {
  const HelpPage({super.key});

  @override
  _HelpPageState createState() => _HelpPageState();
}

class _HelpPageState extends State<HelpPage>
    with SingleTickerProviderStateMixin {
  late TabController _tabController;
  final Map<String, String> _pdfPaths = {
    'pv': '',
    'swh': '',
    'ac': '',
  };
  bool _isLoading = false;
  String _currentType = 'pv';

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 3, vsync: this);
    _tabController.addListener(() {
      if (_tabController.indexIsChanging) {
        _handleTabSelection(_tabController.index);
      }
    });
    _loadPDFFromAPI('pv'); // Default tab PDF
  }

  Future<void> _loadPDFFromAPI(String type) async {
    setState(() {
      _isLoading = true;
      _currentType = type;
    });
    try {
      final url = 'https://aeihawaii.com/photoappsch/loginapi/get_pdf/$type';
      final response = await http.get(Uri.parse(url));

      if (response.statusCode == 200) {
        // Save the PDF locally
        final dir = await getApplicationDocumentsDirectory();
        final pdfFilePath = '${dir.path}/$type.pdf';
        final file = File(pdfFilePath);
        await file.writeAsBytes(response.bodyBytes);

        setState(() {
          _pdfPaths[type] = pdfFilePath;
          _isLoading = false;
        });
      } else {
        throw Exception('Failed to load PDF from API');
      }
    } catch (e) {
      setState(() {
        _isLoading = false;
      });
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Error fetching PDF')),
      );
    }
  }

  void _handleTabSelection(int index) {
    switch (index) {
      case 0:
        _loadPDFFromAPI('pv');
        break;
      case 1:
        _loadPDFFromAPI('swh');
        break;
      case 2:
        _loadPDFFromAPI('ac');
        break;
    }
  }

  Widget _buildPDFView(String type) {
    if (_isLoading && _currentType == type) {
      return Center(child: CircularProgressIndicator());
    }
    final path = _pdfPaths[type];
    if (path != null && path.isNotEmpty) {
      return PDFView(
        filePath: path,
        enableSwipe: true,
        swipeHorizontal: false,
        autoSpacing: true,
        pageSnap: true,
        fitPolicy: FitPolicy.BOTH,
        onError: (error) {
          ScaffoldMessenger.of(context).showSnackBar(
            SnackBar(content: Text('Error displaying PDF: $error')),
          );
        },
      );
    }
    return Center(child: Text('No PDF available for $type'));
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Row(
          children: [
            Icon(Icons.info_outline, size: 24),
            SizedBox(width: 8),
            Text('Help Page'),
          ],
        ),
        bottom: TabBar(
          controller: _tabController,
          tabs: [
            Tab(text: 'PV'),
            Tab(text: 'SWH'),
            Tab(text: 'AC'),
          ],
        ),
      ),
      body: TabBarView(
        controller: _tabController,
        physics: NeverScrollableScrollPhysics(), // Disable horizontal scrolling
        children: [
          _buildPDFView('pv'),
          _buildPDFView('swh'),
          _buildPDFView('ac'),
        ],
      ),
    );
  }

  @override
  void dispose() {
    _tabController.dispose();
    super.dispose();
  }
}
