import 'dart:io';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';

import '../models/photo_item.dart';

class FullScreenGallery extends StatefulWidget {
  final List<PhotoItem> photos;
  final int initialIndex;
  final Future<File> Function(PhotoItem) getCachedFile;
  final String title;

  const FullScreenGallery({
    Key? key,
    required this.photos,
    required this.initialIndex,
    required this.getCachedFile,
    required this.title,
  }) : super(key: key);

  @override
  State<FullScreenGallery> createState() => _FullScreenGalleryState();
}

class _FullScreenGalleryState extends State<FullScreenGallery> {
  late final PageController _controller;
  late int _currentIndex;

  // ✅ iPhone style: show top bar initially
  bool _showUi = true;

  // ✅ One controller used for current page (we reset on page change)
  final TransformationController _tc = TransformationController();
  TapDownDetails? _doubleTapDetails;

  bool get _isZoomed => _tc.value != Matrix4.identity();

  @override
  void initState() {
    super.initState();
    _currentIndex = widget.initialIndex;
    _controller = PageController(initialPage: widget.initialIndex);

    // ✅ true fullscreen (like iPhone viewer)
    SystemChrome.setEnabledSystemUIMode(SystemUiMode.immersiveSticky);
    SystemChrome.setSystemUIOverlayStyle(const SystemUiOverlayStyle(
      statusBarColor: Colors.transparent,
      systemNavigationBarColor: Colors.transparent,
    ));
  }

  @override
  void dispose() {
    SystemChrome.setEnabledSystemUIMode(SystemUiMode.edgeToEdge);
    _controller.dispose();
    _tc.dispose();
    super.dispose();
  }

  void _toggleUi() => setState(() => _showUi = !_showUi);

  void _resetZoom() {
    _tc.value = Matrix4.identity();
  }

  void _zoomInAt(Offset pos, BoxConstraints c) {
    // iPhone-like zoom step
    const scale = 3.0;

    final x = -pos.dx * (scale - 1);
    final y = -pos.dy * (scale - 1);

    _tc.value = Matrix4.identity()
      ..translate(x, y)
      ..scale(scale);
  }

  void _handleDoubleTap(BoxConstraints c) {
    // ✅ double tap toggles zoom
    if (_isZoomed) {
      _resetZoom();
      return;
    }

    final pos = _doubleTapDetails?.localPosition ??
        Offset(c.maxWidth / 2, c.maxHeight / 2);

    // ✅ on double tap: zoom + hide UI (as you asked)
    setState(() => _showUi = false);
    _zoomInAt(pos, c);
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: Colors.black,
      body: Stack(
        children: [
          PageView.builder(
            controller: _controller,
            itemCount: widget.photos.length,

            // ✅ iPhone behavior: can't swipe pages when zoomed
            physics: _isZoomed
                ? const NeverScrollableScrollPhysics()
                : const BouncingScrollPhysics(),

            onPageChanged: (i) {
              setState(() => _currentIndex = i);
              _resetZoom();
              // keep UI hidden/shown as user left it; or show UI on new photo:
              // setState(() => _showUi = true);
            },

            itemBuilder: (context, index) {
              final item = widget.photos[index];

              return FutureBuilder<File>(
                future: widget.getCachedFile(item),
                builder: (context, snap) {
                  if (snap.hasError) {
                    return Center(
                      child: Column(
                        mainAxisSize: MainAxisSize.min,
                        children: [
                          const Icon(Icons.broken_image, color: Colors.white54, size: 48),
                          const SizedBox(height: 8),
                          Text(
                            'Failed to load image',
                            style: const TextStyle(color: Colors.white54),
                          ),
                        ],
                      ),
                    );
                  }
                  if (!snap.hasData) {
                    return const Center(child: CircularProgressIndicator());
                  }

                  final file = snap.data!;

                  return LayoutBuilder(
                    builder: (context, constraints) {
                      return GestureDetector(
                        behavior: HitTestBehavior.opaque,

                        // single tap toggles UI
                        onTap: _toggleUi,

                        // swipe down dismiss ONLY when not zoomed (iPhone style)
                        onVerticalDragEnd: (details) {
                          if (!_isZoomed && (details.primaryVelocity ?? 0) > 300) {
                            Navigator.of(context).pop();
                          }
                        },

                        // double tap zoom
                        onDoubleTapDown: (d) => _doubleTapDetails = d,
                        onDoubleTap: () => _handleDoubleTap(constraints),

                        child: Hero(
                          tag: item.cacheKey,
                          child: InteractiveViewer(
                            transformationController: _tc,
                            panEnabled: true,
                            scaleEnabled: true,
                            minScale: 1.0,
                            maxScale: 6.0,

                            // ✅ CRITICAL: without this, image “sticks” and can’t move after zoom
                            constrained: false,

                            // ✅ iPhone-ish elastic panning
                            boundaryMargin: const EdgeInsets.all(double.infinity),
                            clipBehavior: Clip.none,

                            // ✅ give viewer a real sized child so it can pan freely
                            child: SizedBox(
                              width: constraints.maxWidth,
                              height: constraints.maxHeight,
                              child: Image.file(
                                file,
                                fit: BoxFit.cover, // full screen
                              ),
                            ),
                          ),
                        ),
                      );
                    },
                  );
                },
              );
            },
          ),

          // Top overlay (AppBar-like)
          AnimatedOpacity(
            opacity: _showUi ? 1 : 0,
            duration: const Duration(milliseconds: 150),
            child: IgnorePointer(
              ignoring: !_showUi,
              child: SafeArea(
                child: Container(
                  padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 6),
                  decoration: BoxDecoration(
                    gradient: LinearGradient(
                      begin: Alignment.topCenter,
                      end: Alignment.bottomCenter,
                      colors: [
                        Colors.black.withOpacity(0.75),
                        Colors.black.withOpacity(0.0),
                      ],
                    ),
                  ),
                  child: Row(
                    children: [
                      IconButton(
                        icon: const Icon(Icons.arrow_back, color: Colors.white),
                        onPressed: () => Navigator.of(context).pop(),
                      ),
                      Expanded(
                        child: Text(
                          widget.title,
                          style: const TextStyle(
                            color: Colors.white,
                            fontSize: 16,
                            fontWeight: FontWeight.bold,
                          ),
                          overflow: TextOverflow.ellipsis,
                        ),
                      ),
                      Text(
                        '${_currentIndex + 1}/${widget.photos.length}',
                        style: const TextStyle(color: Colors.white),
                      ),
                    ],
                  ),
                ),
              ),
            ),
          ),
        ],
      ),
    );
  }
}
