import 'package:flutter/material.dart';

class EditJobNotePage extends StatefulWidget {
  final String author;
  final String datetime;
  final String initialNote;

  /// (Optional) for pre-filling the dropdowns when editing
  final String? initialCategory; // e.g., from server
  final String? initialRating; // e.g., "Urgent" / "Semi-Urgent" / "Note"

  const EditJobNotePage({
    super.key,
    required this.author,
    required this.datetime,
    required this.initialNote,
    this.initialCategory,
    this.initialRating,
  });

  @override
  State<EditJobNotePage> createState() => _EditJobNotePageState();
}

class _EditJobNotePageState extends State<EditJobNotePage> {
  final _formKey = GlobalKey<FormState>();
  late TextEditingController _noteController;

  // Canonical options — keep in sync with backend allowed values
  static const List<String> _categoryOptions = <String>[
    'Site Survey',
    'Installation(Roof & Electrical)',
    'Administrative & Accounting',
    'Rescheduling Activity Comment',
    'Technician  Notes',
    'Design & Engineering',
  ];

  static const List<String> _ratingOptions = <String>[
    'Urgent',
    'Semi-Urgent',
    'Note',
  ];

  late String _selectedCategory;
  late String _selectedRating;

  @override
  void initState() {
    super.initState();
    _noteController = TextEditingController(text: widget.initialNote);

    // Safe defaults
    _selectedCategory = (widget.initialCategory != null &&
            _categoryOptions.contains(widget.initialCategory))
        ? widget.initialCategory!
        : _categoryOptions.first; // fallback to first option

    _selectedRating = (widget.initialRating != null &&
            _ratingOptions.contains(widget.initialRating))
        ? widget.initialRating!
        : (_ratingOptions.contains('Note') ? 'Note' : _ratingOptions.first);
  }

  @override
  void dispose() {
    _noteController.dispose();
    super.dispose();
  }

  void _save() {
    if (!_formKey.currentState!.validate()) return;
    Navigator.pop(context, {
      'author': widget.author,
      'datetime': widget.datetime,
      'body': _noteController.text.trim(),
      'category': _selectedCategory,
      'rank': _selectedRating,
    });
  }

  void _delete() {
    Navigator.pop(context, 'delete');
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: Colors.white,
      appBar: AppBar(
        backgroundColor: Colors.green,
        title: const Text('Job Details'),
        actions: const [
          SizedBox(width: 8),
          Icon(Icons.menu),
          SizedBox(width: 12),
        ],
      ),
      resizeToAvoidBottomInset: true,
      body: Padding(
        padding: const EdgeInsets.all(16),
        child: Form(
          key: _formKey,
          child: SingleChildScrollView(
            keyboardDismissBehavior: ScrollViewKeyboardDismissBehavior.onDrag,
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
              // Header row
              Row(
                children: [
                  const CircleAvatar(
                    backgroundColor: Colors.green,
                    child: Text('AE', style: TextStyle(color: Colors.white)),
                  ),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Text(
                      'John Smith & Carolyn Walker',
                      style: const TextStyle(
                        fontSize: 16,
                        fontWeight: FontWeight.bold,
                      ),
                    ),
                  ),
                ],
              ),
              const SizedBox(height: 20),

              const Text('Job Notes',
                  style: TextStyle(fontWeight: FontWeight.bold)),
              const SizedBox(height: 12),

              Text('${widget.datetime} by ${widget.author}',
                  style: const TextStyle(fontWeight: FontWeight.bold)),
              const SizedBox(height: 12),

              // Comment Type
              DropdownButtonFormField<String>(
                value: _selectedCategory,
                items: _categoryOptions
                    .map((c) =>
                        DropdownMenuItem<String>(value: c, child: Text(c)))
                    .toList(),
                onChanged: (v) => setState(() => _selectedCategory = v!),
                validator: (v) =>
                    (v == null || v.isEmpty) ? 'Select a comment type' : null,
                decoration: InputDecoration(
                  labelText: 'Comment Type',
                  contentPadding:
                      const EdgeInsets.symmetric(horizontal: 8, vertical: 6),
                  border: OutlineInputBorder(
                      borderRadius: BorderRadius.circular(4)),
                ),
              ),
              const SizedBox(height: 12),

              // Rating
              DropdownButtonFormField<String>(
                value: _selectedRating,
                items: _ratingOptions
                    .map((r) =>
                        DropdownMenuItem<String>(value: r, child: Text(r)))
                    .toList(),
                onChanged: (v) => setState(() => _selectedRating = v!),
                validator: (v) =>
                    (v == null || v.isEmpty) ? 'Select a rating' : null,
                decoration: InputDecoration(
                  labelText: 'Rating',
                  contentPadding:
                      const EdgeInsets.symmetric(horizontal: 8, vertical: 6),
                  border: OutlineInputBorder(
                      borderRadius: BorderRadius.circular(4)),
                ),
              ),
              const SizedBox(height: 12),

              // Note body
              TextFormField(
                controller: _noteController,
                maxLines: 6,
                validator: (v) => (v == null || v.trim().isEmpty)
                    ? 'Note cannot be empty'
                    : null,
                decoration: InputDecoration(
                  labelText: 'Note',
                  alignLabelWithHint: true,
                  contentPadding:
                      const EdgeInsets.symmetric(horizontal: 8, vertical: 10),
                  border: OutlineInputBorder(
                    borderRadius: BorderRadius.circular(4),
                    borderSide: const BorderSide(color: Colors.black),
                  ),
                ),
              ),

              const SizedBox(height: 24),

              Row(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  _actionButton('Save', Colors.black, _save),
                  const SizedBox(width: 16),
                  _actionButton('Delete', Colors.black, _delete),
                ],
              ),
              const SizedBox(height: 24),
            ],
          ),
          ),
        ),
      ),
    );
  }

  Widget _actionButton(String label, Color color, VoidCallback onPressed) {
    return ElevatedButton(
      style: ElevatedButton.styleFrom(
        backgroundColor: color,
        minimumSize: const Size(100, 40),
      ),
      onPressed: onPressed,
      child: Text(label, style: const TextStyle(fontSize: 14)),
    );
  }
}
