import 'package:flutter/material.dart';
import 'package:flutter/services.dart';

class EditJobCostPage extends StatefulWidget {
  const EditJobCostPage({
    super.key,
    this.initialPrice = '',
    this.initialTax = '',
    this.initialSubtotal = '',
    this.initialPaymentMethod = '',
    this.initialComment = '',
  });

  final String initialPrice;
  final String initialTax;
  final String initialSubtotal;
  final String initialPaymentMethod;
  final String initialComment;

  @override
  State<EditJobCostPage> createState() => _EditJobCostPageState();
}

class _EditJobCostPageState extends State<EditJobCostPage> {
  final _formKey = GlobalKey<FormState>();
  final TextEditingController listPriceController = TextEditingController();
  final TextEditingController taxController = TextEditingController();
  final TextEditingController subTotalController = TextEditingController();
  final TextEditingController commentController = TextEditingController();

  String? selectedPaymentMethod;
  final List<String> paymentMethods = const ['Cash', 'Check', 'Card', 'Online'];

  @override
  void initState() {
    super.initState();
    listPriceController.text = widget.initialPrice;
    taxController.text = widget.initialTax;
    subTotalController.text = widget.initialSubtotal;
    commentController.text = widget.initialComment;
    selectedPaymentMethod = widget.initialPaymentMethod.isEmpty
        ? null
        : widget.initialPaymentMethod;

    // Recalculate subtotal whenever price or tax changes (you can still edit subtotal manually)
    listPriceController.addListener(_recomputeSubtotalIfPossible);
    taxController.addListener(_recomputeSubtotalIfPossible);
  }

  @override
  void dispose() {
    listPriceController.dispose();
    taxController.dispose();
    subTotalController.dispose();
    commentController.dispose();
    super.dispose();
  }

  // --- Helpers ---
  String _numOnly(String s) => s.replaceAll(RegExp(r'[^0-9.\-]'), '');
  double _toDouble(String s) => double.tryParse(_numOnly(s)) ?? 0.0;

  void _recomputeSubtotalIfPossible() {
    final p = _toDouble(listPriceController.text);
    final t = _toDouble(taxController.text);
    final sum = p + t;
    if (sum >= 0) {
      // Only update if user hasn't manually overridden recently (optional rule)
      // For simplicity, always update:
      subTotalController.text = sum.toStringAsFixed(2);
    }
  }

  void _save() {
    if (!_formKey.currentState!.validate()) return;

    // Build payload for API; include legacy aliases to be safe
    final price = _numOnly(listPriceController.text);
    final tax = _numOnly(taxController.text);
    final subtotal = _numOnly(subTotalController.text);
    final payment = selectedPaymentMethod ?? '';
    final comment = commentController.text.trim();

    final payload = <String, dynamic>{
      if (price.isNotEmpty) 'price': price,
      if (tax.isNotEmpty) 'tax': tax,
      if (subtotal.isNotEmpty) 'subtotal': subtotal,

      // new + legacy keys for method of payment
      if (payment.isNotEmpty) 'method': payment,
      if (payment.isNotEmpty) 'mop': payment,

      // new + legacy keys for comments/description
      if (comment.isNotEmpty) 'description': comment,
      if (comment.isNotEmpty) 'addwork_desc': comment,
      if (comment.isNotEmpty) 'desc': comment,
    };

    Navigator.pop(context, payload);
  }

  InputDecoration _dec(String label) => InputDecoration(
        labelText: label,
        border: OutlineInputBorder(borderRadius: BorderRadius.circular(6)),
        isDense: true,
      );

  @override
  Widget build(BuildContext context) {
    final onlyNumbers = <TextInputFormatter>[
      FilteringTextInputFormatter.allow(RegExp(r'[0-9.\-]')),
    ];

    return Scaffold(
      appBar: AppBar(
        title: const Text('Edit Job Cost'),
        backgroundColor: Colors.green,
      ),
      body: SingleChildScrollView(
        padding: const EdgeInsets.all(16),
        child: Form(
          key: _formKey,
          child: Column(
            children: [
              // Price
              TextFormField(
                controller: listPriceController,
                inputFormatters: onlyNumbers,
                keyboardType:
                    const TextInputType.numberWithOptions(decimal: true),
                decoration: _dec("List Price of System (\$)"),
                validator: (v) => (v == null || _numOnly(v).isEmpty)
                    ? 'Please enter price'
                    : null,
              ),
              const SizedBox(height: 12),

              // Tax
              TextFormField(
                controller: taxController,
                inputFormatters: onlyNumbers,
                keyboardType:
                    const TextInputType.numberWithOptions(decimal: true),
                decoration: _dec("State Excise Tax"),
                validator: (v) => (v == null || _numOnly(v).isEmpty)
                    ? 'Please enter tax'
                    : null,
              ),
              const SizedBox(height: 12),

              // Subtotal
              TextFormField(
                controller: subTotalController,
                inputFormatters: onlyNumbers,
                keyboardType:
                    const TextInputType.numberWithOptions(decimal: true),
                decoration: _dec("Sub Total"),
                validator: (v) => (v == null || _numOnly(v).isEmpty)
                    ? 'Please enter subtotal'
                    : null,
              ),
              const SizedBox(height: 12),

              // Payment method
              DropdownButtonFormField<String>(
                value: selectedPaymentMethod,
                decoration: _dec("Method of Payment"),
                items: paymentMethods
                    .map((m) => DropdownMenuItem(value: m, child: Text(m)))
                    .toList(),
                onChanged: (v) => setState(() => selectedPaymentMethod = v),
                validator: (v) =>
                    v == null ? 'Please select a payment method' : null,
              ),
              const SizedBox(height: 12),

              // Comments
              TextFormField(
                controller: commentController,
                maxLines: 3,
                decoration: _dec("Additional comments/requirements"),
                validator: (v) => (v == null || v.trim().isEmpty)
                    ? 'Please enter comments'
                    : null,
              ),
              const SizedBox(height: 20),

              ElevatedButton(
                onPressed: _save,
                style: ElevatedButton.styleFrom(
                  backgroundColor: Colors.black,
                  padding:
                      const EdgeInsets.symmetric(horizontal: 30, vertical: 12),
                ),
                child: const Text("Save", style: TextStyle(fontSize: 16)),
              ),
            ],
          ),
        ),
      ),
    );
  }
}
