import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import 'package:provider/provider.dart';
import '../token_provider.dart';
import 'job_details_full_view.dart';
import 'job_photos_full_view.dart';
import 'job_all_tabs_full_view.dart';
import 'job_tabs_full_screen.dart';

class CustomerPhotosTab extends StatefulWidget {
  final int customerId;

  const CustomerPhotosTab({
    super.key,
    required this.customerId,
  });

  @override
  _CustomerPhotosTabState createState() => _CustomerPhotosTabState();
}

class _CustomerPhotosTabState extends State<CustomerPhotosTab> {
  List<Map<String, String>> _photos = [];
  bool _loading = true;
  String? _error;

  static const String _baseUrl =
      'https://aeihawaii.com/photoappsch/loginapinew/customerphotostabstatenew';

  @override
  void initState() {
    super.initState();
    // Ensure Provider is ready
    WidgetsBinding.instance.addPostFrameCallback((_) => _fetchPhotos());
  }

  Future<void> _fetchPhotos() async {
    setState(() {
      _loading = true;
      _error = null;
    });

    try {
      final token = context.read<TokenProvider>().token ?? '';
      if (token.isEmpty) {
        throw Exception('Missing auth token');
      }

      final uri = Uri.parse('$_baseUrl/${widget.customerId}');
      final res = await http.get(
        uri,
        headers: {
          // ✅ Your API needs raw token (no "Bearer " prefix)
          'authorization': token,
          'accept': 'application/json',
        },
      );

      if (res.statusCode == 200) {
        final body = json.decode(res.body);

        // Expecting: either a List or an object with a list inside.
        final List dataList = body is List
            ? body
            : (body['data'] is List ? body['data'] : <dynamic>[]);

        final photos = dataList.map<Map<String, String>>((item) {
          return _parsePhoto(item);
        }).toList();

        setState(() {
          _photos = photos;
          _loading = false;
        });
      } else if (res.statusCode == 401) {
        throw Exception('Unauthorized (401). Please log in again.');
      } else {
        throw Exception('HTTP ${res.statusCode}: ${res.reasonPhrase}');
      }
    } catch (e) {
      setState(() {
        _error = e.toString();
        _loading = false;
      });
    }
  }

  Map<String, String> _parsePhoto(dynamic item) {
    // 🔧 Map server fields → UI fields (adjust if your keys differ)
    final type = (item['type'] ?? item['job_type'] ?? item['jobType'] ?? 'N/A')
        .toString();

    final jobDate =
        (item['jobDate'] ?? item['job_date'] ?? item['date'] ?? '').toString();

    final uploadDate = (item['uploadDate'] ??
            item['upload_date'] ??
            item['created_at'] ??
            item['uploadedOn'] ??
            '')
        .toString();

    final jobId =
        (item['jobId'] ?? item['job_id'] ?? item['jid'] ?? '').toString();

    // Common image keys your APIs tend to return
    final imageUrl = (item['imageUrl'] ??
            item['image_url'] ??
            item['thumb'] ??
            item['photo_url'] ??
            item['url'] ??
            '')
        .toString();

    // Thumbnail URL — prefer thumbUrl, fall back to imageUrl
    final thumbUrl = (item['thumbUrl'] ??
            item['thumb_url'] ??
            item['thumb_link'] ??
            '')
        .toString();

    return {
      "type": type,
      "jobDate": jobDate,
      "uploadDate": uploadDate,
      "jobId": jobId,
      "imageUrl": imageUrl,
      "thumbUrl": thumbUrl.isNotEmpty ? thumbUrl : imageUrl,
    };
  }

  // 🔑 Centralized redirect that guarantees full-page navigation (no popup)
  void _openJobPhotosFullView(String? jobIdStr, {String? label}) {
    final jobId = int.tryParse(jobIdStr ?? '');
    if (jobId == null) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text("Invalid job ID")),
      );
      return;
    }

    final rootNav = Navigator.of(context, rootNavigator: true);
    // final cid =  ${widget.customerId;}
    // Close only popup/sheet routes if any are open
    rootNav.popUntil((route) => route is! PopupRoute);

    // ✅ True redirect (replace current route) on ROOT navigator with an opaque page route
    WidgetsBinding.instance.addPostFrameCallback((_) {
      rootNav.pushReplacement(
        PageRouteBuilder(
          opaque: true,
          barrierDismissible: false,
          pageBuilder: (_, __, ___) => JobDetailsTabsShell(
            jobId: jobId,
            customerId: widget.customerId,
            customerName: '',
            initialTab: JobShellInitialTab.photos, // start on Photos
            // forceShowCostEdit: true,             // optional
          ),
          transitionsBuilder: (_, animation, __, child) {
            // Simple material fade/slide for nicer UX (optional)
            final offsetTween = Tween<Offset>(
              begin: const Offset(0.0, 0.05),
              end: Offset.zero,
            );
            return SlideTransition(
              position: animation.drive(offsetTween),
              child: FadeTransition(opacity: animation, child: child),
            );
          },
        ),
      );
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: RefreshIndicator(
        onRefresh: _fetchPhotos,
        child: _loading
            ? const Center(child: CircularProgressIndicator())
            : _error != null
                ? ListView(
                    padding: const EdgeInsets.all(16),
                    children: [
                      _ErrorCard(
                        message: _error!,
                        onRetry: _fetchPhotos,
                      )
                    ],
                  )
                : _photos.isEmpty
                    ? ListView(
                        padding: const EdgeInsets.all(16),
                        children: const [
                          _EmptyState(
                              message: 'No photos found for this customer.')
                        ],
                      )
                    : ListView(
                        padding: const EdgeInsets.all(16),
                        children: [
                          // Header with count and add button
                          Row(
                            mainAxisAlignment: MainAxisAlignment.spaceBetween,
                            children: [
                              Text(
                                '${_photos.length} Photos Available',
                                style: const TextStyle(
                                    fontSize: 18, fontWeight: FontWeight.bold),
                              ),
                              IconButton(
                                icon: const Icon(Icons.add_circle,
                                    color: Colors.green, size: 28),
                                onPressed: () {
                                  // TODO: Implement Add Photo flow
                                },
                              ),
                            ],
                          ),
                          const SizedBox(height: 8),

                          // Photo list
                          ..._photos.map((photo) {
                            final hasImage =
                                (photo["imageUrl"] ?? '').isNotEmpty;
                            return InkWell(
                              onTap: () => _openJobPhotosFullView(
                                photo["jobId"],
                                label: photo["type"],
                              ),
                              child: Container(
                                margin: const EdgeInsets.only(bottom: 12),
                                padding: const EdgeInsets.symmetric(
                                    vertical: 8, horizontal: 8),
                                decoration: BoxDecoration(
                                  color: Colors.grey.shade100,
                                  borderRadius: BorderRadius.circular(6),
                                ),
                                child: Row(
                                  crossAxisAlignment: CrossAxisAlignment.start,
                                  children: [
                                    // Text info
                                    Expanded(
                                      child: Column(
                                        crossAxisAlignment:
                                            CrossAxisAlignment.start,
                                        children: [
                                          Text(
                                            '${photo["type"] ?? "N/A"} - ${photo["jobDate"] ?? ""}',
                                            style: const TextStyle(
                                              fontWeight: FontWeight.bold,
                                              fontSize: 14,
                                            ),
                                          ),
                                          Text(photo["uploadDate"] ?? '',
                                              style: const TextStyle(
                                                  fontSize: 12,
                                                  color: Colors.black54)),
                                          const SizedBox(height: 4),
                                          Text(
                                            'JobID ${photo["jobId"] ?? ""}',
                                            style: const TextStyle(
                                              color: Colors.green,
                                              decoration:
                                                  TextDecoration.underline,
                                              fontSize: 14,
                                            ),
                                          ),
                                        ],
                                      ),
                                    ),

                                    // Thumbnail (if available)
                                    if (hasImage)
                                      ClipRRect(
                                        borderRadius: BorderRadius.circular(4),
                                        child: Image.network(
                                          photo["thumbUrl"] ?? photo["imageUrl"]!,
                                          width: 60,
                                          height: 60,
                                          fit: BoxFit.cover,
                                          errorBuilder: (c, e, s) => Container(
                                            width: 60,
                                            height: 60,
                                            alignment: Alignment.center,
                                            color: Colors.grey.shade300,
                                            child: const Icon(
                                                Icons.broken_image,
                                                size: 20,
                                                color: Colors.grey),
                                          ),
                                        ),
                                      ),
                                  ],
                                ),
                              ),
                            );
                          }),
                        ],
                      ),
      ),
    );
  }
}

class _ErrorCard extends StatelessWidget {
  final String message;
  final VoidCallback onRetry;
  const _ErrorCard({required this.message, required this.onRetry});

  @override
  Widget build(BuildContext context) {
    return Container(
      padding: const EdgeInsets.all(14),
      decoration: BoxDecoration(
        color: const Color(0xFFFFF2F2),
        borderRadius: BorderRadius.circular(8),
        border: Border.all(color: const Color(0xFFFFD0D0)),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          const Text('Couldn’t load photos',
              style: TextStyle(fontWeight: FontWeight.bold)),
          const SizedBox(height: 6),
          Text(message, style: const TextStyle(color: Colors.black87)),
          const SizedBox(height: 8),
          TextButton.icon(
            onPressed: onRetry,
            icon: const Icon(Icons.refresh),
            label: const Text('Retry'),
          ),
        ],
      ),
    );
  }
}

class _EmptyState extends StatelessWidget {
  final String message;
  const _EmptyState({required this.message});

  @override
  Widget build(BuildContext context) {
    return Container(
      padding: const EdgeInsets.all(14),
      decoration: BoxDecoration(
        color: const Color(0xFFF7F7F7),
        borderRadius: BorderRadius.circular(8),
        border: Border.all(color: const Color(0xFFE3E3E3)),
      ),
      child: Row(
        children: [
          const Icon(Icons.photo_library_outlined, color: Colors.grey),
          const SizedBox(width: 10),
          Expanded(child: Text(message)),
        ],
      ),
    );
  }
}
