import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:provider/provider.dart';
import '../token_provider.dart';
import 'job_details_full_view.dart';

class CustomerJobsTab extends StatefulWidget {
  final int? customerId;
  final int? jobId;

  const CustomerJobsTab({
    super.key,
    this.customerId,
    this.jobId,
  });

  @override
  State<CustomerJobsTab> createState() => _CustomerJobsTabState();
}

class _CustomerJobsTabState extends State<CustomerJobsTab> {
  List<Map<String, dynamic>> _jobs = [];
  String _token = '';
  String? _userId;

  // Loading states
  bool _isCustomerLoading = true; // initial page load
  bool _isPaging = false; // “Load More” in progress
  String? _error;

  // Pagination state
  int _page = 1;
  final int _pageSize = 20; // adjust as your API supports
  bool _hasMore = true;

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((_) => _loadTokenAndData());
  }

  Future<void> _loadTokenAndData() async {
    try {
      _token = context.read<TokenProvider>().token ?? '';

      final prefs = await SharedPreferences.getInstance();
      _userId = prefs.getString('user_id');

      if (widget.customerId == null) {
        setState(() {
          _isCustomerLoading = false;
          _error = 'Missing customerId.';
        });
        return;
      }

      if (_token.isEmpty) {
        setState(() {
          _isCustomerLoading = false;
          _error = 'Missing auth token.';
        });
        return;
      }

      // Initial fetch page 1
      _page = 1;
      _hasMore = true;
      await _loadCustomerJobs(page: _page, append: false);
    } catch (e) {
      debugPrint('Bootstrap error: $e');
      setState(() {
        _isCustomerLoading = false;
        _error = 'Something went wrong loading your data.';
      });
    }
  }

  Future<void> _loadCustomerJobs(
      {required int page, required bool append}) async {
    if (!append) {
      setState(() {
        _isCustomerLoading = true;
        _error = null;
      });
    } else {
      // paging call
      setState(() {
        _isPaging = true;
        _error = null;
      });
    }

    try {
      // NOTE: If your API expects offset/per_page instead, replace the query params accordingly.
      final uri = Uri.parse(
              'https://aeihawaii.com/photoappsch/loginapinew/customerjobstabstate/${widget.customerId}')
          .replace(queryParameters: {
        'page': page.toString(),
        'limit': _pageSize.toString(),
      });

      final response = await http.get(
        uri,
        headers: {
          'Content-Type': 'application/json',
          'authorization':
              _token, // change to 'Authorization' if your API needs it
        },
      );

      if (response.statusCode == 200) {
        final decoded = json.decode(response.body);

        // Extract list from different shapes
        List<dynamic> list;
        if (decoded is List) {
          list = decoded;
        } else if (decoded is Map<String, dynamic>) {
          if (decoded['jobs'] is List) {
            list = decoded['jobs'];
          } else if (decoded['data'] is List)
            list = decoded['data'];
          else if (decoded['result'] is List)
            list = decoded['result'];
          else
            list = const [];
        } else {
          list = const [];
        }

        final items = <Map<String, dynamic>>[];
        for (final item in list) {
          if (item is Map) items.add(Map<String, dynamic>.from(item));
        }

        setState(() {
          if (append) {
            _jobs.addAll(items);
          } else {
            _jobs = items;
          }

          // If fewer than pageSize came back, no more pages
          _hasMore = items.length >= _pageSize;
        });
      } else {
        debugPrint(
            'Customer jobs API error: ${response.statusCode} ${response.body}');
        setState(() {
          _error = 'Failed to load jobs (HTTP ${response.statusCode}).';
          _hasMore = false;
        });
      }
    } catch (e) {
      debugPrint('Error loading customer jobs: $e');
      setState(() {
        _error = 'Network error while loading jobs.';
        _hasMore = false;
      });
    } finally {
      if (mounted) {
        setState(() {
          _isCustomerLoading = false;
          _isPaging = false;
        });
      }
    }
  }

  // Helper to pick the first non-null/empty value from multiple possible keys
  String _pick(Map<String, dynamic> m, List<String> keys) {
    for (final k in keys) {
      final v = m[k];
      if (v != null && v.toString().trim().isNotEmpty) {
        return v.toString();
      }
    }
    return '';
  }

  Future<void> _loadMore() async {
    if (_isPaging || !_hasMore) return; // guard
    _page += 1;
    await _loadCustomerJobs(page: _page, append: true);
  }

  @override
  Widget build(BuildContext context) {
    if (_isCustomerLoading) {
      return const Center(child: CircularProgressIndicator());
    }

    if (_error != null) {
      return Center(
        child: Padding(
          padding: const EdgeInsets.all(16),
          child: Text(
            _error!,
            textAlign: TextAlign.center,
            style: const TextStyle(fontSize: 16, color: Colors.red),
          ),
        ),
      );
    }

    return Scaffold(
      body: _jobs.isEmpty
          ? const Center(child: Text('No jobs found for this customer.'))
          : ListView.separated(
              padding: const EdgeInsets.all(16),
              itemCount: _jobs.length + 2, // title + jobs + load more
              separatorBuilder: (_, __) => const SizedBox(height: 8),
              itemBuilder: (context, index) {
                // Section Title
                if (index == 0) {
                  return _buildSectionTitle('Completed Jobs');
                }

                // Load More row
                if (index == _jobs.length + 1) {
                  if (!_hasMore) {
                    return const SizedBox.shrink(); // hide row when no more
                  }
                  return Center(
                    child: TextButton(
                      onPressed: _isPaging ? null : _loadMore,
                      child: _isPaging
                          ? const Padding(
                              padding: EdgeInsets.symmetric(vertical: 6.0),
                              child: SizedBox(
                                height: 18,
                                width: 18,
                                child:
                                    CircularProgressIndicator(strokeWidth: 2),
                              ),
                            )
                          : const Text(
                              'Load More',
                              style: TextStyle(fontWeight: FontWeight.bold),
                            ),
                    ),
                  );
                }

                final job = _jobs[index - 1];
                final jobId = _pick(job, ['jobId', 'job_id', 'id']);
                final type = _pick(job, ['type', 'job_type', 'jobType']);
                final date = _pick(job, ['date', 'job_date', 'created_at']);
                final jid = int.tryParse(jobId ?? '');
                final description =
                    _pick(job, ['description', 'job_description', 'details']);

                return Container(
                  padding: const EdgeInsets.all(12),
                  decoration: BoxDecoration(
                    color: Colors.green.shade50,
                    borderRadius: BorderRadius.circular(8),
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      // Tappable headline (type - date - JobID)
                      GestureDetector(
                        onTap: () {
                          if (jid != null) {
                            Navigator.push(
                              context,
                              MaterialPageRoute(
                                builder: (_) => JobDetailsFullView(jobId: jid),
                              ),
                            );
                          }
                        },
                        child: Text(
                          '${type.isEmpty ? "Job" : type}'
                          '${date.isNotEmpty ? " - $date" : ""}'
                          '${jobId.isNotEmpty ? " - JobID $jobId" : ""}',
                          style: const TextStyle(
                            color: Colors.green,
                            fontWeight: FontWeight.w700,
                            fontSize: 16,
                            decoration: TextDecoration.underline,
                          ),
                        ),
                      ),
                      const SizedBox(height: 6),
                      if (description.isNotEmpty)
                        Text(
                          description,
                          style: const TextStyle(fontSize: 14),
                        ),
                    ],
                  ),
                );
              },
            ),
    );
  }

  Widget _buildSectionTitle(String title) {
    return Container(
      decoration: BoxDecoration(
        color: Colors.grey.shade200,
        borderRadius: BorderRadius.circular(8),
      ),
      padding: const EdgeInsets.all(10),
      child: Text(
        title,
        style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
      ),
    );
  }
}

/// Optional simple page; you’re already using JobDetailsFullView above.
class JobDetailsPage extends StatelessWidget {
  final String jobId;
  final String type;
  final String date;

  const JobDetailsPage({
    super.key,
    required this.jobId,
    required this.type,
    required this.date,
  });

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text("Job Details"),
        backgroundColor: Colors.green,
      ),
      body: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(
              "Job ID: ${jobId.isEmpty ? '-' : jobId}",
              style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
            ),
            Text("Type: ${type.isEmpty ? '-' : type}",
                style: const TextStyle(fontSize: 16)),
            Text("Date: ${date.isEmpty ? '-' : date}",
                style: const TextStyle(fontSize: 16)),
            const SizedBox(height: 20),
            const Text("Details coming soon...",
                style: TextStyle(fontSize: 16, color: Colors.grey)),
          ],
        ),
      ),
    );
  }
}
