import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import 'package:provider/provider.dart';
import '../token_provider.dart';
import 'job_details_full_view.dart'; // <- make sure this path is correct

class CustomerDocsTab extends StatefulWidget {
  final int customerId;
  const CustomerDocsTab({super.key, required this.customerId});

  @override
  _CustomerDocsTabState createState() => _CustomerDocsTabState();
}

class _CustomerDocsTabState extends State<CustomerDocsTab> {
  List<Map<String, String>> _docs = [];
  bool _loading = true;
  String? _error;

  static const String _baseUrl =
      'https://aeihawaii.com/photoappsch/loginapinew/customerdocstabstate';

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((_) => _fetchDocs());
  }

  Future<void> _fetchDocs() async {
    setState(() {
      _loading = true;
      _error = null;
    });

    try {
      final token = context.read<TokenProvider>().token ?? '';
      if (token.isEmpty) {
        throw Exception('Missing auth token');
      }

      final uri = Uri.parse('$_baseUrl/${widget.customerId}');
      final res = await http.get(
        uri,
        headers: {
          'authorization': token, // ✅ raw token (no "Bearer ")
          'accept': 'application/json',
        },
      );

      if (res.statusCode != 200) {
        throw Exception('HTTP ${res.statusCode}: ${res.reasonPhrase}');
      }

      final body = json.decode(res.body);
      final List dataList = body is List
          ? body
          : (body is Map && body['data'] is List ? body['data'] : const []);

      final docs = dataList.map<Map<String, String>>(_parseDoc).toList();

      setState(() {
        _docs = docs;
        _loading = false;
      });
    } catch (e) {
      setState(() {
        _error = e.toString();
        _loading = false;
      });
    }
  }

  // 🔧 Map server fields → UI (update keys to match your API exactly)
  Map<String, String> _parseDoc(dynamic item) {
    final map = (item is Map) ? item : <String, dynamic>{};

    final type = (map['type'] ?? map['job_type'] ?? 'N/A').toString();
    final jobDate =
        (map['jobDate'] ?? map['job_date'] ?? map['date'] ?? '').toString();
    final uploadedDate =
        (map['uploadedDate'] ?? map['upload_date'] ?? map['created_at'] ?? '')
            .toString();
    final jobId =
        (map['jobId'] ?? map['job_id'] ?? map['jid'] ?? '').toString();

    // Optional fields if you later add open/download
    final title = (map['title'] ?? map['name'] ?? '').toString();
    final url =
        (map['url'] ?? map['doc_url'] ?? map['download_url'] ?? '').toString();

    return {
      'type': type,
      'jobDate': jobDate,
      'uploadedDate': uploadedDate,
      'jobId': jobId,
      'title': title,
      'url': url,
    };
  }

  void _goToJobDetails(String? jobIdStr) {
    final jobId = int.tryParse(jobIdStr ?? '');
    if (jobId == null) {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text('Invalid job ID')),
      );
      return;
    }

    // Navigate within current navigator so your green AppBar/tabs remain
    Navigator.push(
      context,
      MaterialPageRoute(
        builder: (_) => JobDetailsFullView(jobId: jobId),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: RefreshIndicator(
        onRefresh: _fetchDocs,
        child: _loading
            ? const Center(child: CircularProgressIndicator())
            : _error != null
                ? ListView(
                    padding: const EdgeInsets.all(16),
                    children: [
                      _ErrorCard(message: _error!, onRetry: _fetchDocs),
                    ],
                  )
                : _docs.isEmpty
                    ? ListView(
                        padding: const EdgeInsets.all(16),
                        children: const [
                          _EmptyState(message: 'No documents found.'),
                        ],
                      )
                    : ListView(
                        padding: const EdgeInsets.all(16),
                        children: [
                          Row(
                            mainAxisAlignment: MainAxisAlignment.spaceBetween,
                            children: [
                              const Text(
                                'Documents',
                                style: TextStyle(
                                    fontSize: 18, fontWeight: FontWeight.bold),
                              ),
                              // You can keep or remove this menu
                              PopupMenuButton<String>(
                                icon: const Icon(Icons.add_circle,
                                    color: Colors.green, size: 28),
                                onSelected: (_) {},
                                itemBuilder: (context) => const [
                                  PopupMenuItem(
                                    value: 'Choose File',
                                    child: Row(
                                      children: [
                                        Icon(Icons.insert_drive_file,
                                            color: Colors.black54),
                                        SizedBox(width: 8),
                                        Text('Choose File'),
                                      ],
                                    ),
                                  ),
                                ],
                              ),
                            ],
                          ),
                          const SizedBox(height: 8),
                          ..._docs.map((doc) {
                            return InkWell(
                              onTap: () => _goToJobDetails(doc['jobId']),
                              child: Container(
                                margin: const EdgeInsets.only(bottom: 12),
                                padding: const EdgeInsets.symmetric(
                                    vertical: 8, horizontal: 8),
                                decoration: BoxDecoration(
                                  color: Colors.grey.shade100,
                                  borderRadius: BorderRadius.circular(6),
                                ),
                                child: Row(
                                  children: [
                                    // Text Info
                                    Expanded(
                                      child: Column(
                                        crossAxisAlignment:
                                            CrossAxisAlignment.start,
                                        children: [
                                          Text(
                                            '${doc["type"] ?? "N/A"} - ${doc["jobDate"] ?? ""} - JobID ${doc["jobId"] ?? ""}',
                                            style: const TextStyle(
                                                fontWeight: FontWeight.bold,
                                                fontSize: 14),
                                          ),
                                          Text(
                                            'Date Uploaded: ${doc["uploadedDate"] ?? ""}',
                                            style: const TextStyle(
                                              fontSize: 12,
                                              color: Colors.black54,
                                            ),
                                          ),
                                        ],
                                      ),
                                    ),
                                    IconButton(
                                      tooltip: 'Open job',
                                      icon: const Icon(Icons.open_in_new,
                                          color: Colors.black54),
                                      onPressed: () =>
                                          _goToJobDetails(doc['jobId']),
                                    ),
                                  ],
                                ),
                              ),
                            );
                          }),
                        ],
                      ),
      ),
    );
  }
}

class _ErrorCard extends StatelessWidget {
  final String message;
  final VoidCallback onRetry;
  const _ErrorCard({required this.message, required this.onRetry});

  @override
  Widget build(BuildContext context) {
    return Container(
      padding: const EdgeInsets.all(14),
      decoration: BoxDecoration(
        color: const Color(0xFFFFF2F2),
        borderRadius: BorderRadius.circular(8),
        border: Border.all(color: const Color(0xFFFFD0D0)),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          const Text('Couldn’t load documents',
              style: TextStyle(fontWeight: FontWeight.bold)),
          const SizedBox(height: 6),
          Text(message, style: const TextStyle(color: Colors.black87)),
          const SizedBox(height: 8),
          TextButton.icon(
            onPressed: onRetry,
            icon: const Icon(Icons.refresh),
            label: const Text('Retry'),
          ),
        ],
      ),
    );
  }
}

class _EmptyState extends StatelessWidget {
  final String message;
  const _EmptyState({required this.message});

  @override
  Widget build(BuildContext context) {
    return Container(
      padding: const EdgeInsets.all(14),
      decoration: BoxDecoration(
        color: const Color(0xFFF7F7F7),
        borderRadius: BorderRadius.circular(8),
        border: Border.all(color: const Color(0xFFE3E3E3)),
      ),
      child: Row(
        children: [
          const Icon(Icons.description_outlined, color: Colors.grey),
          const SizedBox(width: 10),
          Expanded(child: Text(message)),
        ],
      ),
    );
  }
}
