import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:shared_preferences/shared_preferences.dart';
import 'job_details_page.dart';
import 'package:provider/provider.dart';
import '../token_provider.dart';
import 'package:flutter_html/flutter_html.dart';

class CustomerDetailsPage extends StatefulWidget {
  final int customerId;
  final int? jobId;
  const CustomerDetailsPage({
    super.key,
    required this.customerId,
    this.jobId,
  });

  @override
  State<CustomerDetailsPage> createState() => _CustomerDetailsPageState();
}

class _CustomerDetailsPageState extends State<CustomerDetailsPage>
    with SingleTickerProviderStateMixin {
  late TabController _tabController;
  String _token = '';
  String? _userId;

  List<Map<String, dynamic>> notes = [];
  Map<String, dynamic>? customerDetails;

  bool _isCustomerLoading = true;
  bool _isNotesLoading = true;

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 3, vsync: this, initialIndex: 0);
    _loadTokenAndData();
  }

  Future<void> _loadTokenAndData() async {
    final token = Provider.of<TokenProvider>(context, listen: false).token;
    _token = token;

    SharedPreferences prefs = await SharedPreferences.getInstance();
    String? userId = prefs.getString('user_id');

    if (userId == null || token.isEmpty) {
      print('Missing user ID or token');
      return;
    }

    await _loadCustomerDetails(token);
    await _loadCustomerNotes(userId, token);
  }

  Future<void> _loadCustomerDetails(String token) async {
    setState(() => _isCustomerLoading = true);
    try {
      final response = await http.get(
        Uri.parse(
            'https://aeihawaii.com/photoappsch/loginapi/getcustomerdetail/${widget.customerId}'),
        headers: {
          'Content-Type': 'application/json',
          'authorization': token,
        },
      );

      if (response.statusCode == 200) {
        final Map<String, dynamic> data = json.decode(response.body);
        setState(() {
          customerDetails = data;
        });
      } else {
        print('Customer detail API error: ${response.statusCode}');
      }
    } catch (e) {
      print('Error loading customer details: $e');
    } finally {
      setState(() => _isCustomerLoading = false);
    }
  }

  Future<void> _loadCustomerNotes(String userId, String token) async {
    setState(() => _isNotesLoading = true);
    try {
      final response = await http.get(
        Uri.parse(
            'https://aeihawaii.com/photoappsch/loginapi/getcustomernotes/${widget.jobId}'),
        headers: {
          'Content-Type': 'application/json',
          'authorization': token,
        },
      );

      if (response.statusCode == 200) {
        final List<dynamic> data = json.decode(response.body);
        setState(() {
          notes = data
              .map<Map<String, dynamic>>((note) => {
                    'postedBy': note['postedby'] ?? '',
                    'date': note['createdat'] ?? '',
                    'rank': note['rank'] ?? '',
                    'category': note['category'] ?? 'General Comments',
                    'message': note['message'] ?? '',
                    'replies': (note['replies'] as List<dynamic>? ?? [])
                        .map<Map<String, dynamic>>((reply) => {
                              'postedBy': reply['postedby'],
                              'date': reply['createdat'],
                              'message': reply['message'],
                            })
                        .toList(),
                  })
              .toList();
        });
      } else {
        print('Note API error: ${response.statusCode}');
      }
    } catch (e) {
      print('Failed to load notes: $e');
    } finally {
      setState(() => _isNotesLoading = false);
    }
  }

  Widget _buildDetailRow(String label, String value) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 4, horizontal: 8),
      child: Row(
        children: [
          Expanded(
              flex: 3,
              child: Text(label,
                  style: const TextStyle(fontWeight: FontWeight.bold))),
          Expanded(flex: 5, child: Text(value)),
        ],
      ),
    );
  }

  Widget _buildSectionTitle(String title) {
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 8, horizontal: 8),
      child: Text('[$title]',
          style: const TextStyle(
              color: Colors.green, fontWeight: FontWeight.bold)),
    );
  }

  Widget _buildContactDetailsTab() {
    if (_isCustomerLoading) {
      return const Center(child: CircularProgressIndicator());
    }

    final c = customerDetails ?? {};

    return ListView(
      padding: const EdgeInsets.all(16),
      children: [
        _buildSectionTitle('Customer Contact Details'),
        _buildDetailRow('First Name', c['first_name'] ?? ''),
        _buildDetailRow('Last Name', c['last_name'] ?? ''),
        _buildDetailRow('Email', c['email'] ?? ''),
        _buildDetailRow('Select Phone', c['preferred_phone'] ?? ''),
        _buildDetailRow('Phone No.', c['phone_number'] ?? ''),
        _buildSectionTitle('Customer Address Details'),
        _buildDetailRow('Address', c['address'] ?? ''),
        _buildDetailRow('City', c['city'] ?? ''),
        _buildDetailRow('State', c['state'] ?? ''),
        _buildDetailRow('Zip Code', c['zip_code'] ?? ''),
      ],
    );
  }

  Widget _buildCommentCard(Map<String, dynamic> comment) {
    final List<Map<String, dynamic>> replies =
        (comment['replies'] as List?)?.cast<Map<String, dynamic>>() ?? [];

    return Container(
      margin: const EdgeInsets.only(bottom: 12),
      padding: const EdgeInsets.all(12),
      decoration: BoxDecoration(
        color: const Color(0xFFF9F9F9),
        border: Border.all(color: const Color(0xFFDDDDDD)),
        borderRadius: BorderRadius.circular(8),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text("Posted: ${comment['date']} by ${comment['postedBy']}",
              style: const TextStyle(fontWeight: FontWeight.bold)),
          Text("Rank: ${comment['rank'] ?? 'Note'}",
              style: const TextStyle(color: Colors.green)),
          const SizedBox(height: 8),
          Html(data: comment['message'] ?? ''),
          const SizedBox(height: 8),
          if (replies.isNotEmpty)
            Padding(
              padding: const EdgeInsets.only(left: 12, top: 12),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: replies
                    .map((reply) => Container(
                          margin: const EdgeInsets.only(bottom: 8),
                          padding: const EdgeInsets.all(10),
                          decoration: BoxDecoration(
                            color: const Color(0xFFF0F0F0),
                            border: Border(
                                left:
                                    BorderSide(color: Colors.green, width: 3)),
                          ),
                          child: Column(
                            crossAxisAlignment: CrossAxisAlignment.start,
                            children: [
                              Text(
                                'Date Posted: ${reply['date']} by: ${reply['postedBy']}',
                                style: const TextStyle(
                                    fontWeight: FontWeight.w600),
                              ),
                              const SizedBox(height: 4),
                              Html(data: reply['message'] ?? ''),
                            ],
                          ),
                        ))
                    .toList(),
              ),
            ),
        ],
      ),
    );
  }

  Widget _buildNotesTab() {
    if (_isNotesLoading) {
      return const Center(child: CircularProgressIndicator());
    }

    if (notes.isEmpty) {
      return const Center(child: Text('No comments yet.'));
    }

    Map<String, List<Map<String, dynamic>>> grouped = {};
    for (var note in notes) {
      final category = note['category'] ?? 'General Comments';
      grouped.putIfAbsent(category, () => []).add(note);
    }

    return ListView(
      padding: const EdgeInsets.all(16),
      children: grouped.entries.map((entry) {
        return Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(entry.key,
                style: const TextStyle(
                    fontSize: 18,
                    fontWeight: FontWeight.bold,
                    color: Colors.black87)),
            const SizedBox(height: 8),
            ...entry.value.map(_buildCommentCard),
            const Divider(height: 32),
          ],
        );
      }).toList(),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text('Customer Details'),
        backgroundColor: const Color(0xFF418E1C),
        leading: IconButton(
          icon: const Icon(Icons.arrow_back),
          onPressed: () => Navigator.pop(context),
        ),
        bottom: TabBar(
          controller: _tabController,
          indicatorColor: Colors.white,
          labelColor: Colors.white,
          unselectedLabelColor: Colors.grey[300],
          tabs: const [
            Tab(text: 'Details'),
            Tab(text: 'Notes'),
            Tab(text: 'Photos'),
          ],
        ),
      ),
      body: TabBarView(
        controller: _tabController,
        children: [
          _buildContactDetailsTab(),
          _buildNotesTab(),
          JobDetailsPage(
            job: {
              'id': widget.jobId?.toString() ?? '',
              'title': 'Customer Photos'
            },
            token: _token ?? '',
          ),
        ],
      ),
    );
  }
}
