import 'package:flutter/material.dart';
import 'package:url_launcher/url_launcher.dart';
import 'edit_customer_page.dart';
import 'customer_jobs_tab.dart';
import 'customer_photos_tab.dart';
import 'customer_docs_tab.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:provider/provider.dart';
import '../token_provider.dart';

class CustomerDetailsFullView extends StatefulWidget {
  final int? customerId;
  final int? jobId;

  const CustomerDetailsFullView({super.key, this.customerId, this.jobId});

  @override
  _CustomerDetailsFullViewState createState() =>
      _CustomerDetailsFullViewState();
}

class _CustomerDetailsFullViewState extends State<CustomerDetailsFullView>
    with TickerProviderStateMixin {
  late TabController _tabController;
  Map<String, dynamic> customerDetails = {};
  bool isLoading = true;
  String _token = '';
  String? _userId;

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 4, vsync: this);
    _loadTokenAndData();
  }

  Future<void> _loadTokenAndData() async {
    try {
      final token = context.read<TokenProvider>().token ?? '';
      _token = token;

      final prefs = await SharedPreferences.getInstance();
      _userId = prefs.getString('user_id');

      if (_userId == null || _token.isEmpty) {
        debugPrint('Missing user ID or token');
        setState(() => isLoading = false);
        return;
      }

      if (widget.customerId != null) {
        await _loadCustomerDetails(_userId!, _token);
      } else {
        setState(() => isLoading = false);
      }
    } catch (e) {
      debugPrint('Bootstrap error: $e');
      setState(() => isLoading = false);
    }
  }

  Future<void> _loadCustomerDetails(String userId, String token) async {
    try {
      final uri = Uri.parse(
        'https://aeihawaii.com/photoappsch/loginapinew/customerdetailsfullviewstate/${widget.customerId}',
      );

      final response = await http.get(
        uri,
        headers: {
          'Content-Type': 'application/json',
          'authorization': token,
        },
      );

      if (response.statusCode == 200) {
        final body = jsonDecode(response.body);
        if (body is Map) {
          setState(() {
            customerDetails = Map<String, dynamic>.from(body);
            isLoading = false;
          });
        } else if (body is List && body.isNotEmpty && body.first is Map) {
          setState(() {
            customerDetails = Map<String, dynamic>.from(body.first as Map);
            isLoading = false;
          });
        }
      } else {
        debugPrint('Customer API error: ${response.statusCode}');
        setState(() => isLoading = false);
      }
    } catch (e) {
      debugPrint('Failed to load customer details: $e');
      setState(() => isLoading = false);
    }
  }

  String s(dynamic v, [String fallback = '']) =>
      v == null ? fallback : v.toString();

  List<Map<String, dynamic>> lm(dynamic v) {
    if (v is List) {
      return v
          .whereType<Map>()
          .map((e) => Map<String, dynamic>.from(e))
          .toList();
    }
    return <Map<String, dynamic>>[];
  }

  @override
  Widget build(BuildContext context) {
    final contactsForName = lm(customerDetails['contacts']);
    final displayName = s(
      customerDetails['name'] ??
          (contactsForName.isNotEmpty ? contactsForName.first['name'] : null),
    );

    return Scaffold(
      appBar: AppBar(
        backgroundColor: Colors.green,
        title: const Text("Customer Details"),
        actions: [
          IconButton(icon: const Icon(Icons.help_outline), onPressed: () {}),
          PopupMenuButton<String>(
            icon: const Icon(Icons.menu),
            onSelected: (value) async {
              if (value == 'Edit Customer') {
                final result = await Navigator.push(
                  context,
                  MaterialPageRoute(
                    builder: (context) => EditCustomerPage(
                      customerId: widget.customerId,
                    ),
                  ),
                );
                if (result == 'updated') {
                  _fetchCustomerDetails();
                }
              }
            },
            itemBuilder: (BuildContext context) {
              return ['Schedule Job', 'Edit Customer']
                  .map((choice) => PopupMenuItem<String>(
                        value: choice,
                        child: Text(choice),
                      ))
                  .toList();
            },
          ),
        ],
      ),
      body: isLoading
          ? const Center(child: CircularProgressIndicator())
          : Column(
              children: [
                Container(
                  color: Colors.green,
                  padding:
                      const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
                  child: Row(
                    children: [
                      const Icon(Icons.person, color: Colors.white, size: 36),
                      const SizedBox(width: 12),
                      Expanded(
                        child: Text(
                          displayName,
                          style: const TextStyle(
                            color: Colors.white,
                            fontSize: 18,
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ),
                      IconButton(
                        icon: const Icon(Icons.add_circle, color: Colors.white),
                        onPressed: () {},
                      ),
                    ],
                  ),
                ),
                Container(
                  color: Colors.grey.shade200,
                  child: TabBar(
                    controller: _tabController,
                    labelColor: Colors.green,
                    unselectedLabelColor: Colors.black54,
                    tabs: const [
                      Tab(text: 'Details'),
                      Tab(text: 'Jobs'),
                      Tab(text: 'Photos'),
                      Tab(text: 'Docs'),
                    ],
                  ),
                ),
                Expanded(
                  child: TabBarView(
                    controller: _tabController,
                    children: [
                      KeyedSubtree(
                        key: ValueKey(customerDetails.hashCode),
                        child: _buildDetailsTab(),
                      ),
                      CustomerJobsTab(customerId: widget.customerId, jobId: 3),
                      CustomerPhotosTab(customerId: widget.customerId!),
                      CustomerDocsTab(customerId: widget.customerId!),
                    ],
                  ),
                ),
              ],
            ),
    );
  }

  Widget _buildDetailsTab() {
    final agreements = lm(customerDetails['service_agreement']);
    final contacts = lm(customerDetails['contacts']);
    final locations = lm(customerDetails['service_locations']);

    return ListView(
      padding: const EdgeInsets.all(16),
      children: [
        _buildSectionTitle('Service Agreements'),
        ...agreements.map<Widget>((a) {
          final status = s(a['status']).toLowerCase();
          final isExpired = status == 'expired';
          return Container(
            margin: const EdgeInsets.symmetric(vertical: 4),
            padding: const EdgeInsets.all(12),
            decoration: BoxDecoration(
              color: isExpired ? Colors.red.shade100 : Colors.green.shade100,
              borderRadius: BorderRadius.circular(8),
            ),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(s(a['title']),
                    style: const TextStyle(
                        fontSize: 16, fontWeight: FontWeight.bold)),
                const SizedBox(height: 4),
                Text(
                  "Through ${s(a['through_date'])} (${s(a['status'])})",
                  style: TextStyle(
                    color: isExpired ? Colors.red : Colors.green,
                    fontSize: 14,
                  ),
                ),
              ],
            ),
          );
        }),
        const SizedBox(height: 16),
        _buildSectionTitle('Customer Contacts'),
        ...contacts.map<Widget>((c) {
          final name = s(c['name']);
          final phone = s(c['phone']);
          final email = s(c['email']);
          return Container(
            padding: const EdgeInsets.symmetric(vertical: 12),
            decoration: const BoxDecoration(
              border:
                  Border(bottom: BorderSide(color: Colors.grey, width: 0.3)),
            ),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(name,
                    style: const TextStyle(
                        fontSize: 16, fontWeight: FontWeight.bold)),
                const SizedBox(height: 4),
                Text(phone, style: TextStyle(color: Colors.grey[700])),
                const SizedBox(height: 8),
                Row(
                  children: [
                    _buildActionButton(
                        Icons.call, 'Call', 'tel:$phone', Colors.green),
                    const SizedBox(width: 12),
                    _buildActionButton(
                        Icons.message, 'Text', 'sms:$phone', Colors.blue),
                    if (email.isNotEmpty) ...[
                      const SizedBox(width: 12),
                      _buildActionButton(
                          Icons.email, 'Email', 'mailto:$email', Colors.orange),
                    ],
                  ],
                ),
              ],
            ),
          );
        }),
        const SizedBox(height: 16),
        _buildSectionTitle('Service Locations'),
        ...locations.map<Widget>((loc) {
          final address = s(loc['address']);
          final city = s(loc['city']);
          final state = s(loc['state']);
          final zip = s(loc['zip']);

          return Padding(
            padding: const EdgeInsets.symmetric(vertical: 6),
            child: InkWell(
              onTap: () async {
                final query = Uri.encodeComponent('$address $city $state $zip');
                final url =
                    'https://www.google.com/maps/search/?api=1&query=$query';
                if (await canLaunchUrl(Uri.parse(url))) {
                  await launchUrl(Uri.parse(url),
                      mode: LaunchMode.externalApplication);
                }
              },
              child: Text(
                '$address\n$city, $state $zip',
                style: const TextStyle(
                  color: Colors.green,
                  fontSize: 16,
                  decoration: TextDecoration.underline,
                ),
              ),
            ),
          );
        }),
      ],
    );
  }

  Widget _buildSectionTitle(String title) {
    return Container(
      color: Colors.grey.shade200,
      padding: const EdgeInsets.all(8),
      margin: const EdgeInsets.symmetric(vertical: 4),
      child: Text(title,
          style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold)),
    );
  }

  Widget _buildActionButton(
      IconData icon, String label, String url, Color color) {
    return InkWell(
      onTap: () async {
        if (await canLaunchUrl(Uri.parse(url))) {
          await launchUrl(Uri.parse(url), mode: LaunchMode.externalApplication);
        }
      },
      child: Container(
        padding: const EdgeInsets.symmetric(horizontal: 10, vertical: 6),
        decoration: BoxDecoration(
          color: color.withOpacity(0.1),
          borderRadius: BorderRadius.circular(6),
        ),
        child: Row(
          children: [
            Icon(icon, color: color, size: 18),
            const SizedBox(width: 4),
            Text(label,
                style: TextStyle(color: color, fontWeight: FontWeight.bold)),
          ],
        ),
      ),
    );
  }

  void _fetchCustomerDetails() {
    if (_userId != null && _token.isNotEmpty && widget.customerId != null) {
      _loadCustomerDetails(_userId!, _token);
    }
  }
}
