/// Represents a single photo upload in the persistent queue.
enum UploadStatus { queued, uploading, uploaded, failed }

class UploadQueueItem {
  final int? id;
  final String filePath; // permanent path in app docs (copied from picker temp)
  final int jobId;
  final String fileName;
  final String dedupKey; // filePath:fileSize:lastModified
  final UploadStatus status;
  final int retryCount;
  final DateTime createdAt;
  final String? errorMessage;

  const UploadQueueItem({
    this.id,
    required this.filePath,
    required this.jobId,
    required this.fileName,
    required this.dedupKey,
    this.status = UploadStatus.queued,
    this.retryCount = 0,
    required this.createdAt,
    this.errorMessage,
  });

  UploadQueueItem copyWith({
    int? id,
    String? filePath,
    int? jobId,
    String? fileName,
    String? dedupKey,
    UploadStatus? status,
    int? retryCount,
    DateTime? createdAt,
    String? errorMessage,
  }) {
    return UploadQueueItem(
      id: id ?? this.id,
      filePath: filePath ?? this.filePath,
      jobId: jobId ?? this.jobId,
      fileName: fileName ?? this.fileName,
      dedupKey: dedupKey ?? this.dedupKey,
      status: status ?? this.status,
      retryCount: retryCount ?? this.retryCount,
      createdAt: createdAt ?? this.createdAt,
      errorMessage: errorMessage ?? this.errorMessage,
    );
  }

  Map<String, dynamic> toMap() {
    return {
      if (id != null) 'id': id,
      'file_path': filePath,
      'job_id': jobId,
      'file_name': fileName,
      'dedup_key': dedupKey,
      'status': status.name,
      'retry_count': retryCount,
      'created_at': createdAt.toIso8601String(),
      'error_message': errorMessage,
    };
  }

  factory UploadQueueItem.fromMap(Map<String, dynamic> map) {
    return UploadQueueItem(
      id: map['id'] as int?,
      filePath: map['file_path'] as String,
      jobId: map['job_id'] as int,
      fileName: map['file_name'] as String,
      dedupKey: map['dedup_key'] as String,
      status: UploadStatus.values.firstWhere(
        (e) => e.name == map['status'],
        orElse: () => UploadStatus.queued,
      ),
      retryCount: map['retry_count'] as int? ?? 0,
      createdAt: DateTime.parse(map['created_at'] as String),
      errorMessage: map['error_message'] as String?,
    );
  }

  static const String createTableSQL = '''
    CREATE TABLE IF NOT EXISTS upload_queue (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      file_path TEXT NOT NULL,
      job_id INTEGER NOT NULL,
      file_name TEXT NOT NULL,
      dedup_key TEXT NOT NULL UNIQUE,
      status TEXT NOT NULL DEFAULT 'queued',
      retry_count INTEGER NOT NULL DEFAULT 0,
      created_at TEXT NOT NULL,
      error_message TEXT
    )
  ''';
}
