import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import 'dart:convert';
import 'package:http/http.dart' as http;
import 'home_page.dart';
import 'token_provider.dart';
import 'package:shared_preferences/shared_preferences.dart';

class LoginPage extends StatefulWidget {
  const LoginPage({super.key});

  @override
  _LoginPageState createState() => _LoginPageState();
}

class _LoginPageState extends State<LoginPage> {
  final TextEditingController _emailController = TextEditingController();
  final TextEditingController _passwordController = TextEditingController();
  bool _isLoading = false;

  @override
  void initState() {
    super.initState();
    _checkLoginStatus();
  }

  Future<void> _checkLoginStatus() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    String? token = prefs.getString('tokenid');

    // Set the token in TokenProvider
    Provider.of<TokenProvider>(context, listen: false).setToken(token!);

    // Navigate to the home page
    Navigator.pushReplacement(
      context,
      MaterialPageRoute(builder: (context) => HomePage()),
    );
  }

  Future<void> _login() async {
    setState(() {
      _isLoading = true;
    });

    try {
      final response = await http.post(
        Uri.parse('https://aeihawaii.com/photoappsch/loginapi/authenticate'),
        headers: {'Content-Type': 'application/json'},
        body: jsonEncode({
          'email': _emailController.text.trim(),
          'password': _passwordController.text,
        }),
      );

      // Debug (optional)
      // print("STATUS: ${response.statusCode}");
      // print("BODY: ${response.body}");

      if (response.statusCode != 200) {
        // Try to read server message if present
        String msg = "Login failed (${response.statusCode})";
        try {
          final err = jsonDecode(response.body);
          msg = err['message']?.toString() ?? err['error']?.toString() ?? msg;
        } catch (_) {
          // response body not JSON
          msg = response.body.isNotEmpty ? response.body : msg;
        }

        ScaffoldMessenger.of(context).showSnackBar(
          SnackBar(content: Text(msg)),
        );
        return;
      }

      // status 200 => parse JSON safely
      final data = jsonDecode(response.body);

      final token = data['data']?['token']?.toString() ?? "";
      final userId = data['data']?['user']?['id']?.toString() ?? "";

      final role =
          data['data']?['user']?['type']?.toString() ?? ""; // <-- FIXED

      if (token.isEmpty || userId.isEmpty) {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text("Login response missing token/user id")),
        );
        return;
      }

      final prefs = await SharedPreferences.getInstance();
      await prefs.setString('user_id', userId);
      await prefs.setString('tokenid', token); // IMPORTANT: tokenid
      await prefs.setString('role_tech', role);

      Provider.of<TokenProvider>(context, listen: false).setToken(token);

      if (!context.mounted) return;
      Navigator.pushReplacement(
        context,
        MaterialPageRoute(builder: (context) => const HomePage()),
      );
    } catch (e) {
      // This is for network/DNS/timeout/SSL issues mostly
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Network/Server error: $e')),
      );
    } finally {
      setState(() {
        _isLoading = false;
      });
    }
  }

  Future<void> _logout() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    await prefs.clear();

    // Navigate back to login page
    Navigator.pushReplacement(
      context,
      MaterialPageRoute(builder: (context) => LoginPage()),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Row(
          children: [
            Image.asset(
              'assets/logo.png', // Path to your logo
              height: 30, // Adjust the size as needed
            ),
            SizedBox(width: 8),
            Text('Login'),
          ],
        ),
      ),
      body: Padding(
        padding: const EdgeInsets.all(16.0),
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Image.asset(
              'assets/logo.png', // Display logo above the login form
              height: 100,
            ),
            SizedBox(height: 20.0),
            TextField(
              controller: _emailController,
              decoration: InputDecoration(
                labelText: 'Email/UserName',
                border: OutlineInputBorder(),
              ),
            ),
            SizedBox(height: 16.0),
            TextField(
              controller: _passwordController,
              decoration: InputDecoration(
                labelText: 'Password',
                border: OutlineInputBorder(),
              ),
              obscureText: true,
            ),
            SizedBox(height: 20.0),
            _isLoading
                ? CircularProgressIndicator()
                : ElevatedButton(
                    style: ElevatedButton.styleFrom(
                      backgroundColor:
                          Colors.green, // Set button color to green
                    ),
                    onPressed: _login,
                    child: Text('Login'),
                  ),
          ],
        ),
      ),
    );
  }
}
