import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'pages/today_job_page.dart';
import 'pages/my_job_page.dart';
import 'pages/help_page.dart';
import 'pages/job_page.dart';
import 'pages/technicians_tab.dart';
import 'session_service.dart';
import 'login_page.dart';
import 'widgets/clock_status_banner.dart';

class HomePage extends StatefulWidget {
  const HomePage({super.key});

  @override
  State<HomePage> createState() => _HomePageState();
}

class _HomePageState extends State<HomePage> {
  bool _canSeeTechnicians = false;
  bool _loadingRole = true;

  @override
  void initState() {
    super.initState();
    _loadRole();
  }

  Future<void> _loadRole() async {
    final prefs = await SharedPreferences.getInstance();

    final roleRaw = prefs.getString('role_tech');
    final role = (roleRaw ?? '').trim().toLowerCase();

    // Robust allow rule (handles "admin", "Admin ", "sales\n", "sales user", etc.)
    final allowed = role == 'admin' ||
        role == 'techm' ||
        role.contains('admin') ||
        role.contains('techm');

    if (!mounted) return;
    setState(() {
      _canSeeTechnicians = allowed;
      _loadingRole = false;
    });

}    

  Future<void> logout(BuildContext context) async {
    await SessionService.clearUserSession();
    ScaffoldMessenger.of(context).showSnackBar(
      const SnackBar(content: Text('You have been logged out.')),
    );
    if (!context.mounted) return;
    Navigator.pushAndRemoveUntil(
      context,
      MaterialPageRoute(builder: (context) => const LoginPage()),
      (Route<dynamic> route) => false,
    );
  }

  Future<void> _selectDateAndNavigate(BuildContext context) async {
    final pickedDate = await showDatePicker(
      context: context,
      initialDate: DateTime.now(),
      firstDate: DateTime(2020),
      lastDate: DateTime(2100),
    );
    if (pickedDate != null) {
      if (!context.mounted) return;
      Navigator.push(
        context,
        MaterialPageRoute(
          builder: (context) => JobPage(selectedDate: pickedDate),
        ),
      );
    }
  }

  @override
  Widget build(BuildContext context) {
    if (_loadingRole) {
      return const Scaffold(
        body: Center(child: CircularProgressIndicator()),
      );
    }

    final tabs = <Tab>[
      const Tab(text: 'Schedule'),
      const Tab(text: 'My Job'),
      if (_canSeeTechnicians) const Tab(text: 'Technicians'),
    ];

    final views = <Widget>[
      const TodayJobPage(),
      const MyJobPage(),
      if (_canSeeTechnicians) const TechniciansTab(),
    ];

    return DefaultTabController(
      length: tabs.length,
      child: Scaffold(
        appBar: PreferredSize(
          preferredSize: const Size.fromHeight(140),
          child: AppBar(
            backgroundColor: const Color(0xFF418E1C),
            automaticallyImplyLeading: false,
            title: Row(
              crossAxisAlignment: CrossAxisAlignment.center,
              children: [
                Image.asset('assets/logo.png', height: 36),
                const SizedBox(width: 8),
                const Text(
                  'Job Schedule',
                  style: TextStyle(
                    fontWeight: FontWeight.bold,
                    fontSize: 20,
                    color: Colors.white,
                  ),
                ),
              ],
            ),
            actions: [
              const ClockStatusBanner(),
              IconButton(
                icon: const Icon(Icons.help_outline, color: Colors.white),
                onPressed: () {
                  Navigator.push(
                    context,
                    MaterialPageRoute(builder: (context) => const HelpPage()),
                  );
                },
              ),
              IconButton(
                icon: const Icon(Icons.logout, color: Colors.white),
                onPressed: () async => await logout(context),
              ),
            ],
            bottom: PreferredSize(
              preferredSize: const Size.fromHeight(90),
              child: Column(
                children: [
                  Padding(
                    padding:
                        const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
                    child: GestureDetector(
                      onTap: () => _selectDateAndNavigate(context),
                      child: AbsorbPointer(
                        child: TextField(
                          decoration: InputDecoration(
                            hintText: 'Tap to select date',
                            filled: true,
                            fillColor: Colors.white,
                            contentPadding: const EdgeInsets.symmetric(
                              horizontal: 12,
                              vertical: 10,
                            ),
                            border: OutlineInputBorder(
                              borderRadius: BorderRadius.circular(8),
                            ),
                            prefixIcon: const Icon(Icons.calendar_today),
                          ),
                        ),
                      ),
                    ),
                  ),

                  // IMPORTANT: NOT const (because tabs are dynamic)
                  TabBar(
                    indicatorColor: Colors.white,
                    labelColor: Colors.white,
                    unselectedLabelColor: Colors.white70,
                    tabs: tabs,
                  ),
                ],
              ),
            ),
          ),
        ),

        // IMPORTANT: NOT const (because views are dynamic)
        body: TabBarView(children: views),
      ),
    );
  }
}
