# AEI Hawaii Scheduler - Maintenance Guide

## Overview

This document outlines the methodology for maintaining the AEI Hawaii Scheduler application across local development and remote production environments.

---

## Environment Details

### Local Development
- **Path:** `/var/www/html/dev_scheduler/SCHEDULER/`
- **URL:** `http://aeihawaii.com/scheduler/` (via hosts file)
- **Database:** MariaDB `mandhdesign_schedular`
- **PHP Version:** 8.2
- **Version Control:** Git (local repository)

### Remote Production
- **Server:** 18.225.0.90
- **Path:** `/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/`
- **Database:** MySQL 5.5 `mandhdesign_schedular`
- **SSH Access:** `ssh -i /tmp/aei_key Julian@18.225.0.90`
- **Version Control:** None (deploy from local)

---

## Git Workflow

### Repository Structure

```
/var/www/html/dev_scheduler/SCHEDULER/
├── .git/                    # Git repository
├── .gitignore              # Ignore logs, cache, local config
├── system/
│   └── application/
│       ├── controllers/    # Main PHP controllers
│       ├── models/
│       ├── views/
│       ├── libraries/
│       └── config/
└── ...
```

### Branch Strategy

```
main (production-synced)
  │
  ├── maint/XXX-description    # Feature branches for each maintenance item
  │
  └── hotfix/XXX-description   # Emergency fixes
```

- **main:** Always matches production. Never commit directly.
- **maint/XXX-description:** Branch for each maintenance item (e.g., `maint/011-optimize-calendar`)
- **hotfix/XXX-description:** Emergency production fixes

### Workflow Steps

#### 1. Sync Local with Production (Before Starting Work)

```bash
# Ensure main branch matches production
git checkout main
git status  # Should be clean

# If needed, sync specific files from production:
scp -i /tmp/aei_key Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php \
    /var/www/html/dev_scheduler/SCHEDULER/path/to/file.php
git add -A && git commit -m "Sync with production"
```

#### 2. Create Maintenance Branch

```bash
# Create new branch for maintenance item
git checkout -b maint/XXX-short-description

# Example:
git checkout -b maint/011-calendar-optimization
```

#### 3. Make Changes and Commit

```bash
# Make changes to files...

# Stage and commit with descriptive message
git add path/to/modified/file.php
git commit -m "MAINT-011: Description of change

- Detail 1
- Detail 2

Affected: controller_name.php function_name()"
```

#### 4. Test Locally

- Test all affected pages on local server
- Verify no regressions on related functionality
- Document test results

#### 5. Deploy to Production

```bash
# Create backup on production first
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php \
      /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php.bak.maintXXX"

# Deploy file
scp -i /tmp/aei_key /var/www/html/dev_scheduler/SCHEDULER/path/to/file.php \
    Julian@18.225.0.90:/tmp/file.php.maintXXX

ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "mv /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php \
      /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php.old && \
   cp /tmp/file.php.maintXXX \
      /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php"
```

#### 6. Verify and Merge

```bash
# After production verification, merge to main
git checkout main
git merge maint/XXX-short-description
git tag -a vMAINT-XXX -m "MAINT-XXX: Description"

# Delete feature branch (optional)
git branch -d maint/XXX-short-description
```

#### 7. Rollback (If Needed)

**Local rollback:**
```bash
git checkout main
git revert HEAD  # Revert last commit
# OR
git reset --hard <commit-hash>  # Reset to specific commit
```

**Production rollback:**
```bash
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php.bak.maintXXX \
      /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php"
```

---

## Maintenance Item Lifecycle

```
┌─────────────┐     ┌─────────────┐     ┌─────────────┐
│  PLANNING   │ --> │   LOCAL     │ --> │  TESTING    │
│             │     │   DEV       │     │             │
└─────────────┘     └─────────────┘     └─────────────┘
                                              │
                    ┌─────────────┐           │
                    │  DEPLOYED   │ <---------┘
                    │             │     (after approval)
                    └─────────────┘
                          │
              ┌───────────┴───────────┐
              v                       v
       ┌─────────────┐         ┌─────────────┐
       │  VERIFIED   │         │  REVERTED   │
       └─────────────┘         └─────────────┘
```

### Status Definitions

| Status | Description |
|--------|-------------|
| PLANNING | Issue identified, analyzing solution |
| LOCAL_DEV | Changes being developed on local branch |
| TESTING | Changes complete, testing on local server |
| DEPLOYED | Changes pushed to production |
| VERIFIED | Production verified working |
| REVERTED | Changes rolled back |
| ON_HOLD | Paused, waiting for decision |

---

## File Naming Conventions

### Git Branches
- `maint/XXX-short-description` (e.g., `maint/011-calendar-opt`)
- `hotfix/XXX-short-description` (e.g., `hotfix/012-500-error-fix`)

### Production Backups
- `filename.php.bak.maintXXX` (e.g., `admin.php.bak.maint011`)

### Commit Messages
```
MAINT-XXX: Short description (50 chars max)

- Detailed change 1
- Detailed change 2

Affected: file.php function_name()
```

---

## Quick Reference Commands

### Check Current State
```bash
cd /var/www/html/dev_scheduler/SCHEDULER
git status
git log --oneline -10
git branch -a
```

### Compare with Production
```bash
# Single file
ssh -i /tmp/aei_key Julian@18.225.0.90 "cat /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php" | diff - /var/www/html/dev_scheduler/SCHEDULER/path/to/file.php

# Checksum comparison
local_md5=$(md5sum /var/www/html/dev_scheduler/SCHEDULER/path/to/file.php | cut -d' ' -f1)
remote_md5=$(ssh -i /tmp/aei_key Julian@18.225.0.90 "md5sum /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php" | cut -d' ' -f1)
[ "$local_md5" = "$remote_md5" ] && echo "MATCH" || echo "DIFFERS"
```

### View Changes in Branch
```bash
git diff main..HEAD
git log main..HEAD --oneline
```

---

## Maintenance Log

| ID | Description | Branch | Status | Created | Deployed |
|----|-------------|--------|--------|---------|----------|
| 001 | PDF temp files cleanup | N/A | COMPLETED | 2026-01-07 | 2026-01-07 |
| 002 | Query performance optimization | maint/002 | REVERTED | 2026-01-08 | - |
| 003 | Comprehensive performance audit | maint/003 | REVERTED | 2026-01-08 | - |
| 004 | Admin calendar DATEDIFF optimization | maint/004 | REVERTED | 2026-01-08 | - |
| 005 | Job controllers optimization | maint/005 | REVERTED | 2026-01-08 | - |
| 006 | Job listing controllers + 500 fix | maint/006 | REVERTED | 2026-01-08 | - |
| 007 | Assign controller optimization | maint/007 | REVERTED | 2026-01-08 | - |
| 008 | Proposal performance optimization | maint/008 | REVERTED | 2026-01-08 | - |
| 009 | Controller performance review | maint/009 | ON_HOLD | 2026-01-08 | - |
| 010 | Job comment 500 error fix | N/A | RESOLVED | 2026-01-08 | 2026-01-08 |

---

## Important Notes

1. **Always sync with production before starting new work**
2. **Never push directly to production without local testing**
3. **Keep production backups for at least 30 days**
4. **Document all changes in commit messages**
5. **Test thoroughly - production has real user data**
6. **When in doubt, ask before deploying**

---

## Troubleshooting

### Git Issues

**Undo uncommitted changes:**
```bash
git checkout -- path/to/file.php
```

**Undo last commit (keep changes):**
```bash
git reset --soft HEAD~1
```

**View file at specific commit:**
```bash
git show <commit-hash>:path/to/file.php
```

### Production Issues

**Check Apache error log:**
```bash
ssh -i /tmp/aei_key Julian@18.225.0.90 "tail -50 /var/log/httpd/error_log"
```

**Verify file permissions:**
```bash
ssh -i /tmp/aei_key Julian@18.225.0.90 "ls -la /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php"
```

---

*Last Updated: 2026-01-08*
