# MAINT-006 Changes Log

## Status: DEPLOYED - 2026-01-08

---

### 2026-01-08 - Job Listing Controllers Review

**Problem:** Review job listing controllers for performance optimization and fix 500 error.

**URLs to Investigate:**
- http://aeihawaii.com/scheduler/admin/myjobs
- http://aeihawaii.com/scheduler/admin/unscheduled_jobs
- http://aeihawaii.com/scheduler/admin/mycompletejobs
- http://aeihawaii.com/scheduler/cancel_job/cancel_jobs (500 ERROR)

**Reported By:** User

---

## Analysis Tasks

- [x] Investigate cancel_jobs 500 error - **MISSING FUNCTION**
- [x] Analyze `myjobs()` function in admin.php
- [x] Analyze `unscheduled_jobs()` function in admin.php
- [x] Analyze `mycompletejobs()` function in admin.php
- [x] Analyze `cancel_jobs()` function in cancel_job.php
- [x] Check for DATE_FORMAT in WHERE clauses
- [x] Check for N+1 query patterns
- [x] Check for full table scans

---

## Findings

### Issue 1: cancel_job.php - CRITICAL 500 ERROR

**Location:** `cancel_job.php:248-279`

The `cancel_jobs()` function calls `$this->getUserIntial()` but this function is **NOT DEFINED** in the Cancel_job class. It only exists in admin.php.

```php
// Lines 255, 265, 266, 277 - function does not exist in this class!
$salesuser[$user] = $this->getUserIntial($user, true)."(De)";
$salesuser[$user] = $this->getUserIntial($user, true)."(C)";
$temp_elec[] = $this->getUserIntial($user, false);
$salesuser[$user] = $this->getUserIntial($user, true)."(I)";
```

**Impact:** Page returns 500 Internal Server Error

---

### Issue 2: DATE_FORMAT() in WHERE Clauses (Index Killer)

All 4 controllers use `DATE_FORMAT()` in WHERE clauses, which prevents index usage:

**myjobs() - Line 5278:**
```php
$sql_date = " AND DATE_FORMAT(jobs.job_date,'%Y%m%d') >= '".$data['yearmonth'].$data['day']."'...";
```

**unscheduled_jobs() - Line 5713:**
```php
$sql_date = " AND DATE_FORMAT(jobs.job_date,'%Y%m%d') >= '".$data['yearmonth'].$data['day']."'...";
```

**mycompletejobs() - Line 6144:**
```php
$sql_date = " AND DATE_FORMAT(jobs.job_date,'%Y%m%d') >= '".$data['yearmonth'].$data['day']."'...";
```

**service_calls queries (Lines 5292, 5734, 6166):**
```php
DATE_FORMAT(service_calls.date,'%Y%m%d') >= ...
```

**Impact:** Full table scans on jobs table (84,160+ rows) and service_calls table

---

### Issue 3: Full Table Scans on File Tables

All 3 admin.php functions load ENTIRE file tables into memory:

**myjobs() - Lines 5308-5341:**
```php
$ppfile = $this->db->get("genral_files")->result_array();      // ALL genral_files
$pmfile = $this->db->get("presale_files")->result_array();     // ALL presale_files
$permitfile = $this->db->get("permit_files")->result_array();  // ALL permit_files
$skfile = $this->db->get("sketch_files")->result_array();      // ALL sketch_files
```

**Same pattern in:**
- `unscheduled_jobs()` - Lines 5750-5781
- `mycompletejobs()` - Lines 6182-6213
- `cancel_jobs()` - Lines 231-232 (only genral_files)

**Impact:** Loading potentially 100,000+ rows into memory when only ~30 are needed

---

### Issue 4: N+1 Query Pattern with getUserIntial()

Each job listing loops through jobs and calls `getUserIntial()` multiple times per job:

```php
foreach($jobs as $key => $job) {
    // For each designer_id (could be multiple comma-separated)
    $salesuser[$user] = $this->getUserIntial($user,true)."(De)";

    // For each contractor_id (could be multiple comma-separated)
    $salesuser[$user] = $this->getUserIntial($user,true)."(C)";
    $temp_elec[] = $this->getUserIntial($user,false);

    // For each installer_id (could be multiple comma-separated)
    $salesuser[$user] = $this->getUserIntial($user,true)."(I)";
}
```

**getUserIntial() at admin.php:8138:**
```php
function getUserIntial($id,$fullname=false){
    $j = $this->db->query("SELECT users.* FROM users WHERE id =$id ")->result();
    // individual query per user
}
```

**Impact:** With 30 jobs, each having 2-3 assigned users = 60-90+ extra queries per page

---

## Implementation Plan

### Phase 1: Fix cancel_job.php 500 Error (CRITICAL)

Add the missing `getUserIntial()` function to the Cancel_job class:

```php
function getUserIntial($id, $fullname = false) {
    // Use cached user lookup
    $this->_loadUsersCache();
    if (isset($this->_users_by_id[$id])) {
        $user = $this->_users_by_id[$id];
        if ($fullname) {
            return ucfirst($user->first_name) . " " . ucfirst($user->last_name);
        }
        return ucfirst(substr($user->first_name, 0, 1)) . ucfirst(substr($user->last_name, 0, 1));
    }
    return "-";
}
```

### Phase 2: Fix DATE_FORMAT Queries

Convert DATE_FORMAT comparisons to direct date comparisons:

```php
// BEFORE (prevents index usage)
$sql_date = " AND DATE_FORMAT(jobs.job_date,'%Y%m%d') >= '".$data['yearmonth'].$data['day']."'";

// AFTER (allows index usage)
$start_date = substr($data['yearmonth'], 0, 4) . '-' . substr($data['yearmonth'], 4, 2) . '-' . $data['day'];
$sql_date = " AND jobs.job_date >= '$start_date'";
```

### Phase 3: Replace Full Table Scans with Filtered Queries

Instead of loading entire file tables, filter by job IDs:

```php
// BEFORE: Load entire table
$ppfile = $this->db->get("genral_files")->result_array();

// AFTER: Filter by job IDs from the current page
$job_ids = array_column($jobs, 'job_pid');
if (!empty($job_ids)) {
    $this->db->where_in('job_id', $job_ids);
}
$ppfile = $this->db->get("genral_files")->result_array();
```

### Phase 4: Add User Cache to Eliminate N+1

Apply the same `_loadUsersCache()` pattern from admin.php/job_status_tab.php.

---

## File Modification Log

| Date | File | Change | Status |
|------|------|--------|--------|
| 2026-01-08 | cancel_job.php | Add getUserIntial() and user cache | **DEPLOYED** |
| 2026-01-08 | admin.php | Fix DATE_FORMAT in myjobs(), unscheduled_jobs(), mycompletejobs() | **DEPLOYED** |
| 2026-01-08 | admin.php | Optimize file table queries (filter by job_pids) | **DEPLOYED** |

---

## Changes Made

### 1. cancel_job.php - Fixed 500 Error

**Added private cache properties:**
```php
private $_users_cache = null;
private $_users_by_id = array();
```

**Added _loadUsersCache() function:**
```php
private function _loadUsersCache() {
    if ($this->_users_cache === null) {
        $this->_users_cache = $this->db->query("SELECT * FROM users")->result();
        foreach ($this->_users_cache as $user) {
            $this->_users_by_id[$user->id] = $user;
        }
    }
}
```

**Added getUserIntial() function:**
```php
function getUserIntial($id, $fullname = false) {
    if (!$id) return 0;
    $this->_loadUsersCache();
    if (isset($this->_users_by_id[$id])) {
        $user = $this->_users_by_id[$id];
        if ($fullname) {
            return ucfirst($user->first_name) . " " . ucfirst($user->last_name);
        }
        return ucfirst($user->first_name[0]) . strtoupper($user->last_name[0] . $user->last_name[0]);
    }
    return "";
}
```

**Backup:** cancel_job.php.bak.maint006

---

## Rollback Plan

All modified files will be backed up before changes.

