# MAINT-005 Changes Log

## Status: DEPLOYED - 2026-01-08 13:33 HST

---

### 2026-01-08 - Job Controllers Performance Review

**Problem:** Review job-related controller pages for performance optimization opportunities.

**URLs to Investigate:**
- http://aeihawaii.com/scheduler/admin/job_comment/106527
- http://aeihawaii.com/scheduler/admin/job/106527
- http://aeihawaii.com/scheduler/admin/job_financing/106527
- http://aeihawaii.com/scheduler/job_status_tab/job_status/106527

**Reported By:** User

---

## Analysis Tasks

- [x] Analyze `job_comment()` function in admin.php - Uses cached getuser(), no major issues
- [x] Analyze `job()` function in admin.php - Uses cached getuser(), no major issues
- [x] Analyze `job_financing()` function in admin.php - Sequential proposal queries, acceptable
- [x] Analyze `job_status()` function in job_status_tab.php - **MAJOR ISSUES FOUND**
- [x] Check for DATEDIFF/DATE_FORMAT in WHERE clauses - None found in these controllers
- [x] Check for N+1 query patterns - **18+ queries for permit_files in job_status_tab**
- [x] Check for missing indexes - permit_files, genral_files, presale_files, sketch_files
- [x] Identify related models and libraries - Job.php uses deprecated mysql_* functions

---

## Findings

### Issue 1: job_status_tab.php - Unoptimized `getuser()` Function

**Location:** `job_status_tab.php:1005-1020`

The `getuser()` function in this controller was NOT updated in MAINT-002. It still makes individual database queries:

```php
$j = $this->db->query("SELECT users.* FROM users WHERE id =$id ")->result();
```

Meanwhile, `admin.php` has the optimized cached version. This inconsistency causes N+1 queries when viewing job status pages.

**Impact:** Multiple individual user queries per page load

---

### Issue 2: job_status_tab.php - Massive N+1 for permit_files Table

**Location:** `job_status_tab.php:125-275`

The `job_status()` function makes **18+ separate queries** to the `permit_files` table, each fetching a different file type:

```php
// Each of these is a separate database query!
$this->db->where("job_id",$pid);
$this->db->where("type",'spa');
$data['file_spa'] = $this->db->get("permit_files")->row_array();

$this->db->where("job_id",$pid);
$this->db->where("type",'stc');
$data['file_stc'] = $this->db->get("permit_files")->row_array();
// ... 16 more similar queries ...
```

**File types queried individually:**
- spa, stc, project_approved, hoaapproval, heco_bill, signder, obt_ride
- hoa_app, heco_ap, trustdocument, cover_sheet, heep_app
- derclosepacket, derclosemailtracking, deracceptance, arraymapping
- permit_close, permit_open

**Plus additional queries to:**
- genral_files
- predesignsketch_files
- presale_files
- sketch_files

**Impact:** ~22+ database queries that could be reduced to 4-5 queries

---

### Issue 3: Job.php Library - Deprecated mysql_* Functions

**Location:** `system/application/libraries/Job.php`

The Job library (used by all job controllers) uses deprecated `mysql_*` functions:

```php
$res = mysql_fetch_object(mysql_query($sql_jobtype));  // Line 47
$result = mysql_query($query);                          // Line 55
$numfields = mysql_num_fields($result);                 // Line 56
mysql_result($result, $row, $i);                        // Line 61
```

**Functions affected:**
- `percent_completed()` - Lines 43-136
- `getjobtabclass()` - Lines 139+
- `getfinancetabclass()` - (likely similar)
- `getcustomertabclass()` - (likely similar)

**Impact:** Deprecated functions, potential connection issues, no query caching

---

### Issue 4: Missing Index on permit_files Table

The `permit_files` table is queried heavily by job_id and type, but may be missing a composite index.

**Tables to check:**
- `permit_files` - needs index on (job_id, type)
- `genral_files` - needs index on (job_id)
- `presale_files` - needs index on (job_id, file_type)
- `sketch_files` - needs index on (job_id)
- `predesignsketch_files` - needs index on (job_id)

---

## Implementation Plan

### Phase 1: Optimize permit_files Queries (High Impact)

Replace 18+ individual queries with a single batched query:

```php
// BEFORE: 18+ queries
$this->db->where("job_id",$pid);
$this->db->where("type",'spa');
$data['file_spa'] = $this->db->get("permit_files")->row_array();
// ... repeat 17 more times ...

// AFTER: 1 query
$sql = "SELECT * FROM permit_files WHERE job_id = ? ORDER BY type, created DESC";
$all_files = $this->db->query($sql, array($pid))->result_array();

// Group by type
$files_by_type = array();
foreach ($all_files as $file) {
    if (!isset($files_by_type[$file['type']])) {
        $files_by_type[$file['type']] = $file;
    }
}
$data['file_spa'] = isset($files_by_type['spa']) ? $files_by_type['spa'] : array();
// ... etc
```

### Phase 2: Copy User Cache to job_status_tab.php

Apply the same `_loadUsersCache()` optimization from admin.php to job_status_tab.php.

### Phase 3: Add Database Indexes

```sql
CREATE INDEX idx_job_type ON permit_files(job_id, type);
CREATE INDEX idx_job_id ON genral_files(job_id);
CREATE INDEX idx_job_filetype ON presale_files(job_id, file_type);
CREATE INDEX idx_job_id ON sketch_files(job_id);
CREATE INDEX idx_job_id ON predesignsketch_files(job_id);
```

### Phase 4: Update Job.php Library (Lower Priority)

Convert deprecated `mysql_*` functions to CodeIgniter's database methods. This is a larger refactor that should be done carefully.

---

## File Modification Log

| Date | File | Change | Status |
|------|------|--------|--------|
| 2026-01-08 | job_status_tab.php | Added user cache, batched permit_files queries | Complete |
| 2026-01-08 | Database | Added 5 indexes on file tables | Complete |
| 2026-01-08 | activitylog.php | Fixed MySQL strict mode error on logout | Complete |

## Changes Made

### 1. Database Indexes Added (Local)

```sql
CREATE INDEX idx_job_type ON permit_files(job_id, type);
CREATE INDEX idx_job_id ON genral_files(job_id);
CREATE INDEX idx_job_filetype ON presale_files(job_id, file_type);
CREATE INDEX idx_job_id ON sketch_files(job_id);
CREATE INDEX idx_job_id ON predesignsketch_files(job_id);
```

### 2. job_status_tab.php Optimizations

**Added helper functions:**
- `_loadUsersCache()` - Caches all users in memory for fast lookup
- `_loadPermitFiles($job_id)` - Batch loads all permit files in 1 query
- `_loadSentNotificationLog($job_pid)` - Batch loads notification logs in 1 query
- `_loadJobDropdownLog($job_pid)` - Batch loads dropdown logs in 1 query
- `_loadJobNotesLog($job_pid)` - Batch loads notes logs in 1 query

**Updated `getuser()` function:**
- Now uses cached user lookup instead of individual database queries

**Query reduction in `job_status()` function:**
- **BEFORE:** ~42 individual queries
- **AFTER:** ~10 queries

| Query Type | Before | After |
|------------|--------|-------|
| permit_files | 23 | 1 |
| sent_notififaction_log | 9 | 1 |
| job_dropdown_log | 5 | 1 |
| job_notes_log | 5 | 1 |
| Other file tables | 5 | 5 |
| **Total** | **~42** | **~10** |

### 3. activitylog.php - MySQL Strict Mode Fix

**Issue:** Logout caused database error:
```
Error Number: 1366
Incorrect integer value: '' for column `job_id` at row 1
```

**Root Cause:** The `createLog()` function passed empty strings `''` for `job_id` and `customer_id` columns, which are integers in MySQL strict mode.

**Fix:** Convert empty strings to NULL before inserting:
```php
// BEFORE
$row['job_id'] = $job_id;
$row['customer_id'] = $customer_id;

// AFTER
$row['job_id'] = ($job_id === '' || $job_id === null) ? NULL : $job_id;
$row['customer_id'] = ($customer_id === '' || $customer_id === null) ? NULL : $customer_id;
```

---

## Rollback Plan

All modified files will be backed up before changes.
