# MAINT-003: Comprehensive Performance Audit

## Executive Summary

A thorough site-wide performance audit of the AEI Scheduler codebase has identified **systemic performance issues** that significantly impact page load times and server resources. This audit focuses on code-level changes only (no database schema modifications).

### Issue Statistics

| Category | Count | Severity |
|----------|-------|----------|
| Remaining DATE_FORMAT in WHERE | 24+ locations | Critical |
| N+1 Query Patterns | 20+ patterns | Critical |
| Deprecated mysql_* functions | 200+ calls | High |
| External HTTP Requests (blocking) | 150+ calls | Critical |
| Inefficient Array Operations | 100+ patterns | High |
| Queries Without LIMIT | 50+ queries | High |

---

## CATEGORY 1: Remaining DATE_FORMAT() Issues

### Status: Not Yet Fixed (MAINT-002 only covered admin.php)

**Files Requiring DATE_FORMAT Optimization:**

| File | Lines | Functions Affected |
|------|-------|-------------------|
| `controllers/ticket.php` | 125 | reportgraph() |
| `controllers/assign.php` | 163, 366, 528, 560, 763, 926 | myassignments(), unscheduled_jobs(), mycompletejobs() |
| `controllers/inventory.php` | 140, 201, 1273 | Multiple inventory functions |
| `controllers/acjobs.php` | 82 | Index function |
| `controllers/swhjobs.php` | 108, 302 | Multiple view functions |
| `controllers/pvjobs.php` | 98, 265, 450, 3177, 3461, 3651 | Multiple view functions |
| `controllers/customerajax.php` | 159 | Customer search |
| `controllers/mycustomerpage.php` | 65 | Customer page |

**Solution:** Apply same DATE_FORMAT → date range conversion as MAINT-002.

---

## CATEGORY 2: N+1 Query Patterns

### Critical Issues in Active Controllers

**1. Notes Queries Inside Job Loops (admin.php)**
- Lines: 638-669 (controllers_lat/admin.php)
- Impact: Each job type loop (PV, SWH, SAF, service_calls) executes a notes query per job
- Estimated: 40-100+ extra queries per day view page

```php
// Current problematic pattern (repeated 4 times):
foreach($data['pv_jobs'] as $j) {
    $notes = $this->db->query("SELECT notes.* FROM notes WHERE notes.job_id='".$j['id']."'");
}
```

**Solution:** Fetch all notes in one query using `IN()` clause, then map to jobs in PHP.

**2. Nested Configuration Loops (admin.php)**
- Lines: 2170-2224
- Pattern: for() loop × foreach() × foreach() with queries
- Impact: Potential O(n³) query multiplication

**3. Count Aggregation Loop (ajax.php)**
- Line: 652-657
- Issue: PHP loop used instead of SQL SUM()

---

## CATEGORY 3: Deprecated mysql_* Functions

### Critical - Uses Separate Database Connection

**High Impact Files:**

| File | Occurrences | Priority |
|------|-------------|----------|
| `libraries/Job.php` | 58 | CRITICAL |
| `controllers/derreport.php` | 58 | HIGH |
| `controllers/derreport_job.php` | 58 | HIGH |
| `controllers/admin.php` | 3 | MEDIUM |

**Functions Used:**
- `mysql_query()` - ~200+ instances
- `mysql_fetch_object()` - ~150+ instances
- `mysql_result()` - ~50+ instances
- `mysql_num_fields()` - ~30+ instances

**Problems:**
1. Uses separate connection from CodeIgniter's `$this->db`
2. No connection pooling benefits
3. No automatic escaping (security risk)
4. Will break on PHP 7.0+ (functions removed)

**Solution:** Migrate all mysql_* calls to CodeIgniter's Query Builder.

---

## CATEGORY 4: External HTTP Requests (Blocking)

### Critical Performance Bottlenecks

**1. NREL PVWatts API (proposal.php)**
- Lines: 7349, 7351
- Type: Synchronous `file_get_contents()`
- Blocking Time: 1-3+ seconds per call
- Impact: SEVERE - Blocks proposal generation

```php
$contents = @file_get_contents("http://developer.nrel.gov/api/pvwatts/v5.json?...");
```

**Files Affected:** proposal.php, customers_pv.php, esign.php (22+ instances)

**2. Weather Underground API**
- Files: lead_weather.php, complete_lead_graph.php, weatherapic.php
- Critical Issue: Called IN LOOPS (lines 31-44 in weatherapic.php)
- Impact: 30+ API calls per page load possible

```php
for($i=0; $i<$days; $i++){
    $json_string = file_get_contents("http://api.wunderground.com/...");
}
```

**3. Honolulu Building Permit Scraping (permitapi.php)**
- Lines: 363-396, 422-451, 802-831
- Type: Multiple `curl_exec()` calls
- Impact: Government websites are slow, unpredictable

**4. Self-Referential HTTP Calls (proposal.php)**
- Line: 2455
- Issue: Calls own server via HTTP instead of direct function call

```php
$html = file_get_contents('http://www.aeihawaii.com/dev/scheduler/pdfhtml/createpdfhtml');
```

**5. osTicket API (osticket_helper.php)**
- Lines: 1013-1261
- 5 separate `curl_exec()` calls for ticket operations

**Solutions:**
1. Queue external API calls asynchronously
2. Cache API responses (consider Redis/Memcached)
3. Replace self-referential HTTP with direct function calls
4. Remove loop-based API calls; batch requests

---

## CATEGORY 5: Inefficient Array Operations

### Critical Issues

**1. in_array() Inside Loops - O(n²) Complexity**
- File: `controllers/admin.php`
- Lines: 1369-1387, 9121, 9135, 10633, 13024, 13435
- Impact: 14+ occurrences, each causing quadratic time complexity

```php
// Current O(n²) pattern:
foreach($_cached_installers as $row) {
    if (in_array($row['id'], $installer_assign_array)) {
        // Process
    }
}

// Optimized O(n) pattern:
$lookup = array_flip($installer_assign_array);
foreach($_cached_installers as $row) {
    if (isset($lookup[$row['id']])) {
        // Process
    }
}
```

**2. array_merge() Inside Nested Loops**
- File: `controllers/ticket.php`
- Lines: 330-353
- Impact: Creates new arrays in O(n) time per merge, inside 3 levels of loops

**3. count() in Loop Conditions**
- File: `controllers/ticket.php`
- Lines: 350-352
- Issue: `count()` called on every iteration instead of caching

**4. Repeated explode() Calls**
- File: `controllers/login.php`
- Lines: 197-215
- Issue: Same string exploded 9 times for role checking

---

## CATEGORY 6: Queries Without LIMIT

### Critical - Memory/Performance Impact

**1. SELECT * FROM jobs (No WHERE)**
- File: `admin.php`
- Lines: 7443, 9054
- Impact: CRITICAL - Fetches entire jobs table (thousands of records)

**2. SELECT * FROM customers (No WHERE)**
- File: `controllers/acajax.php`
- Line: 303
- Impact: HIGH - Used in autocomplete, fetches all customers

**3. SELECT * FROM users (Repeated)**
- Files: Multiple (40+ locations)
- Impact: HIGH - User table fetched repeatedly without LIMIT

**4. File/Attachment Tables (No LIMIT)**
- Tables: elecphoto_files, files, meter_files, permit_files
- File: `controllers/admin.php`
- Lines: 28+ occurrences
- Impact: MEDIUM-HIGH - Attachment tables grow large

---

## IMPLEMENTATION PRIORITY

### Phase 1: Critical Quick Wins (Immediate Impact)

| Task | Files | Est. Queries Saved |
|------|-------|-------------------|
| Fix remaining DATE_FORMAT queries | 8 files | 50-100 per request |
| Add LIMIT to critical queries | 5 files | Prevent memory issues |
| Convert in_array() to isset() lookups | admin.php | 1000+ comparisons |

### Phase 2: High Priority (Major Impact)

| Task | Files | Impact |
|------|-------|--------|
| Fix N+1 notes queries | admin.php | 40-100 queries saved |
| Cache external API responses | proposal.php, weather files | 1-10 seconds saved |
| Replace self-referential HTTP | proposal.php | 500ms-2s saved |

### Phase 3: Medium Priority (Cleanup)

| Task | Files | Impact |
|------|-------|--------|
| Migrate mysql_* to CodeIgniter | Job.php, derreport.php | Better connection handling |
| Fix array_merge in loops | ticket.php | Memory/CPU savings |
| Remove loop-based API calls | weatherapic.php | Major for weather pages |

### Phase 4: Long-term (Technical Debt)

| Task | Files | Impact |
|------|-------|--------|
| Implement API response caching | Multiple | Site-wide improvement |
| Add async job queue for external calls | Multiple | Non-blocking operations |
| Audit all queries for proper indexing | Multiple | Database performance |

---

## FILES REQUIRING CHANGES (By Priority)

### Tier 1 - Highest Impact
1. `system/application/controllers/admin.php` - N+1, array operations
2. `system/application/controllers/proposal.php` - External API calls
3. `system/application/libraries/Job.php` - mysql_* functions
4. `system/application/controllers/ajax.php` - Aggregation queries

### Tier 2 - High Impact
5. `system/application/controllers/assign.php` - DATE_FORMAT
6. `system/application/controllers/pvjobs.php` - DATE_FORMAT (6 locations)
7. `system/application/controllers/ticket.php` - Array operations
8. `system/application/controllers/derreport.php` - mysql_* functions

### Tier 3 - Medium Impact
9. `system/application/controllers/inventory.php` - DATE_FORMAT
10. `system/application/controllers/swhjobs.php` - DATE_FORMAT
11. `system/application/controllers/acjobs.php` - DATE_FORMAT
12. `system/application/controllers/weatherapic.php` - Loop API calls
13. `system/application/controllers/permitapi.php` - curl blocking

### Tier 4 - Lower Priority
14. `system/application/controllers/customerajax.php` - DATE_FORMAT
15. `system/application/controllers/mycustomerpage.php` - DATE_FORMAT
16. `system/application/controllers/login.php` - Repeated explode
17. `system/application/helpers/osticket_helper.php` - curl blocking

---

## ESTIMATED PERFORMANCE IMPROVEMENTS

| Metric | Current (Est.) | After Optimization |
|--------|----------------|-------------------|
| Queries per complex page | 150-300 | 30-50 |
| External API blocking time | 3-15 seconds | <500ms (cached) |
| Array operation complexity | O(n²) in many places | O(n) |
| Memory usage (large pages) | 50-200MB | 20-50MB |

---

## TESTING REQUIREMENTS

### Before Implementation
- [ ] Profile current page load times
- [ ] Enable MySQL slow query log
- [ ] Document current memory usage

### After Each Phase
- [ ] Measure page load improvements
- [ ] Verify no functionality regressions
- [ ] Test with production-like data volumes

### Specific Tests
- [ ] Proposal generation with API calls
- [ ] Day view with many jobs
- [ ] Customer search/autocomplete
- [ ] Report generation (derreport)
- [ ] Weather graph pages

---

## NOTES

- **No database schema changes** in this maintenance item
- Focus on code-level optimizations only
- MAINT-002 already addressed admin.php DATE_FORMAT and caching for day()/jobschedule()
- This audit expands optimization to the entire codebase
