# PDF Temp Files Accumulation Issue

**Date:** 2026-01-07
**Location:** `httpdocs/dev/scheduler/pdftemp/`

---

## Executive Summary

The `pdftemp` folder has accumulated **50,853+ files** totaling **6GB** of disk space. This is caused by a bug in the cleanup code and missing scheduled cleanup processes.

---

## Current State

| Metric | Value |
|--------|-------|
| Total Files | 50,853+ |
| Total Size | 6 GB |
| HTML Files | 33,515 |
| PDF Files | 18,562 |
| Intermediate Temp Files | 38,069 |

### File Types Present
- `--proposal*.pdf` - Generated proposal PDFs
- `*doc1.pdf` - Document PDFs
- `temp*-*.html` - HTML files used for PDF generation
- `footer*-*.html` - Footer HTML templates
- `partpdf*-*.html/pdf` - Parts list PDFs
- `commission*-*.html/pdf` - Commission PDFs

---

## Root Causes

### 1. Bug in Cleanup Code (PRIMARY ISSUE)

Files are created with one naming pattern but `unlink()` tries to delete files with a different pattern.

**Example from `proposal.php` (lines 4165-4191):**

```php
// FILE CREATION - includes $proposal_id in filename
$fp = fopen(".../pdftemp/temp$t-$proposal_id.html", "w+");
exec("... /pdftemp/proposal$t-$proposal_id.pdf");

// CLEANUP - MISSING $proposal_id in filename (BUG!)
unlink(".../pdftemp/temp$t.html");      // Should be: temp$t-$proposal_id.html
unlink(".../pdftemp/proposal$t.pdf");   // Should be: proposal$t-$proposal_id.pdf
```

**This bug exists in multiple controllers:**
- `proposal.php` (lines 4190-4191, 7367-7368, 7489-7490)
- `esign.php` (lines 1764-1765, 5031-5032, 5146-5147)
- `cron.php` (lines 1730-1731)
- `acproposal*.php` (multiple versions)
- `acpdf.php` (lines 741-742)

### 2. No Cleanup for Email Workflow

When `$is_mail = true`, the function returns the file path early and **no cleanup occurs**:

```php
if ($is_mail) {
    return "/path/to/pdftemp/proposal$t-$proposal_id.pdf";  // Returns here, no cleanup
} else {
    // ... cleanup code only runs in else branch
    unlink(...);
}
```

Files created for email attachments are never deleted.

### 3. Footer HTML Files Never Cleaned

Footer HTML files are created but never deleted in any code path:

```php
$footer_fp = fopen(".../pdftemp/footer$footer_t-$proposal_id.html", "w+");
// No corresponding unlink() for footer files anywhere
```

### 4. No Scheduled Cleanup Process

There is **no cron job or scheduled task** to clean up old temp files. The cron files in the server backup are empty.

### 5. Multiple pdftemp Locations

PDF temp files are also written to:
- `/httpdocs/dev/scheduler/pdftemp/` (main location - 6GB)
- `/httpdocs/esign/upload/pdftemp/` (secondary location)
- `/httpdocs/dev/scheduler/temppdf/` (576KB - relatively small)

---

## Affected Controllers

| Controller | Function | Issue |
|------------|----------|-------|
| `proposal.php` | `createpdfdesign()` | Bug in unlink filename |
| `proposal.php` | `beforeschedulecpf()` | Bug in unlink filename |
| `proposal.php` | `sendproposalemail()` | No cleanup after email |
| `esign.php` | `createpdfdesign()` | Bug in unlink filename |
| `esign.php` | `beforeschedulecpf()` | Bug in unlink filename |
| `cron.php` | `createpdfdesign()` | Bug in unlink filename |
| `acproposal.php` | `createpdfdesign()` | Bug in unlink filename |
| `acpdf.php` | PDF generation | Bug in unlink filename |
| Multiple | Footer creation | Never cleaned |
| Multiple | Email workflow | Never cleaned |

---

## Recommended Solutions

### Solution 1: Fix the Unlink Bug (Code Fix)

Fix all instances where `unlink()` has incorrect filenames:

**Before:**
```php
unlink("/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp/temp$t.html");
unlink("/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp/proposal$t.pdf");
```

**After:**
```php
unlink("/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp/temp$t-$proposal_id.html");
unlink("/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp/footer$footer_t-$proposal_id.html");
unlink("/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp/proposal$t-$proposal_id.pdf");
```

### Solution 2: Add Footer Cleanup

Add `unlink()` for footer files wherever they are created:

```php
// After PDF generation completes
unlink($footerhtml);  // Clean up footer HTML
```

### Solution 3: Cleanup After Email Workflow

Move cleanup outside the if/else or add cleanup for email path:

```php
if ($is_mail) {
    $return_path = "/path/to/pdftemp/proposal$t-$proposal_id.pdf";
    // Don't delete PDF since it's needed for email
    // But DO delete intermediate files:
    @unlink("/path/to/pdftemp/temp$t-$proposal_id.html");
    @unlink("/path/to/pdftemp/footer$footer_t-$proposal_id.html");
    return $return_path;
}
```

### Solution 4: Create Scheduled Cleanup Cron Job

Add a cron job to clean up old temp files. Example script:

```bash
#!/bin/bash
# cleanup_pdftemp.sh
# Run daily via cron: 0 2 * * * /path/to/cleanup_pdftemp.sh

PDFTEMP_DIR="/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp"

# Delete HTML files older than 1 day
find "$PDFTEMP_DIR" -name "*.html" -type f -mtime +1 -delete

# Delete PDF files older than 7 days (keep recent ones for potential re-download)
find "$PDFTEMP_DIR" -name "*.pdf" -type f -mtime +7 -delete

# Log cleanup
echo "$(date): Cleaned pdftemp folder" >> /var/log/pdftemp_cleanup.log
```

**Crontab entry:**
```
0 2 * * * /path/to/cleanup_pdftemp.sh
```

### Solution 5: Immediate Manual Cleanup

To immediately free up disk space (one-time cleanup):

```bash
# Preview what would be deleted (dry run)
find /var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp -name "*.html" -type f -mtime +1 | wc -l
find /var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp -name "*.pdf" -type f -mtime +30 | wc -l

# Delete old HTML files (safe - these are just intermediate files)
find /var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp -name "*.html" -type f -mtime +1 -delete

# Delete old PDF files (keep last 30 days for safety)
find /var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/pdftemp -name "*.pdf" -type f -mtime +30 -delete
```

---

## Implementation Priority

| Priority | Task | Effort | Impact |
|----------|------|--------|--------|
| 1 | Manual cleanup of existing files | Low | High (immediate disk space recovery) |
| 2 | Add cron job for scheduled cleanup | Low | High (prevents future accumulation) |
| 3 | Fix unlink bug in code | Medium | Medium (proper cleanup going forward) |
| 4 | Add footer cleanup | Low | Low (small file size) |
| 5 | Fix email workflow cleanup | Medium | Medium (depends on email volume) |

---

## Files to Modify

```
system/application/controllers/proposal.php
system/application/controllers/esign.php
system/application/controllers/cron.php
system/application/controllers/acproposal.php
system/application/controllers/acpdf.php
system/application/controllers/adminpartpdf.php
system/application/controllers/actabpartpdf.php
system/application/controllers/drwproposal.php
system/application/controllers/roofproposal.php
system/application/controllers/safproposal.php
system/application/controllers/swhproposal.php
system/application/controllers/gcesign.php
system/application/controllers/evesign.php
system/application/controllers/roofesign.php
system/application/controllers/safesign.php
system/application/controllers/swhesign.php
system/application/controllers/acesign.php
```

---

## Testing Checklist

After implementing fixes:

- [ ] Generate a new proposal PDF and verify temp files are cleaned
- [ ] Send a proposal via email and verify intermediate files are cleaned
- [ ] Verify footer HTML files are being deleted
- [ ] Run cron cleanup script and verify old files are removed
- [ ] Monitor pdftemp folder size over 1 week to confirm fix effectiveness

---

## Notes

- The `mail_sent/` folder stores copies of emailed PDFs (intentional, for history)
- The `temppdf/` folder is small (576KB) and appears to be managed differently
- Consider adding monitoring/alerting for pdftemp folder size
