# Enhancement Workflow

How to structure new enhancement projects for the AEI Scheduler. Based on the
`AEI_PHOTO_API_PROJECT/` template.

---

## Project Structure

Each enhancement gets its own directory under a parent folder:

```
ENHANCEMENTS/
├── ENH-001_PHOTO_UPLOAD/
│   ├── README.md          # Problem statement, solution, current status
│   ├── ORIGINAL/          # Backup copies of original files before changes
│   ├── NEW/               # Modified/new files ready for deployment
│   └── CHANGES.md         # What changed, why, and how
│
├── ENH-002_NEXT_PROJECT/
│   ├── README.md
│   ├── ORIGINAL/
│   ├── NEW/
│   └── CHANGES.md
```

### Reference: AEI_PHOTO_API_PROJECT

The Photo API project at `AEI_PHOTO_API_PROJECT/` demonstrates this pattern
with a full implementation including documentation, original file backups,
and deployment-ready new files.

---

## Lifecycle Phases

```
PLANNING  →  LOCAL_DEV  →  TESTING  →  DEPLOYED  →  VERIFIED
                                            │
                                            └──→  REVERTED (if issues)
```

| Phase | Description |
|-------|-------------|
| **PLANNING** | Problem identified, analyzing solution approach |
| **LOCAL_DEV** | Changes being developed and tested locally |
| **TESTING** | Changes complete, undergoing testing on local site |
| **DEPLOYED** | Changes pushed to production |
| **VERIFIED** | Production verified working correctly |
| **REVERTED** | Changes rolled back due to issues |
| **ON_HOLD** | Paused, waiting for decision or dependency |

---

## Step-by-Step

### 1. Create the Project Directory

```bash
mkdir -p ENHANCEMENTS/ENH-XXX_SHORT_NAME/{ORIGINAL,NEW}
```

### 2. Write the README

Create `README.md` with:

```markdown
# ENH-XXX: Short Description

## Status: PLANNING

## Problem
What's broken or missing? Include specific symptoms, affected users, and impact.

## Solution
Proposed approach. Which files need to change and why.

## Files Affected
| File | Change Type |
|------|-------------|
| `controllers/example.php` | Modified |
| `views/example_view.php` | New |

## Dependencies
- Any prerequisites or related items

## Testing
How to verify this works correctly.
```

### 3. Backup Original Files

Before modifying anything, copy the original files:

```bash
cp /var/www/html/dev_scheduler/SCHEDULER/path/to/file.php \
   ENHANCEMENTS/ENH-XXX_SHORT_NAME/ORIGINAL/file.php
```

### 4. Develop Changes

Work in the local dev environment. When ready, copy modified files to `NEW/`:

```bash
cp /var/www/html/dev_scheduler/SCHEDULER/path/to/file.php \
   ENHANCEMENTS/ENH-XXX_SHORT_NAME/NEW/file.php
```

### 5. Document Changes

Create `CHANGES.md`:

```markdown
# ENH-XXX Changes

## Modified Files

### controllers/example.php
- Added `new_function()` at line 150
- Modified `existing_function()` to handle edge case
- Reason: [why this change was needed]

## New Files
- `views/new_view.php` - New view for the feature

## Database Changes
- None (or describe schema changes)

## Configuration Changes
- None (or describe config changes)
```

### 6. Test Locally

Follow `TESTING_GUIDE.md` for the testing procedure.

### 7. Deploy

```bash
scripts/deploy_remote.sh path/to/modified_file.php
```

### 8. Verify and Update Status

Update the README status from `TESTING` to `DEPLOYED`, then to `VERIFIED`
after production verification.

---

## Naming Convention

- Directory: `ENH-XXX_SHORT_NAME` (e.g., `ENH-001_PHOTO_UPLOAD`)
- Use sequential numbers
- Keep the short name descriptive but brief (2-4 words, underscored)

---

## Git Integration

If using git in the local dev environment:

```bash
# Create branch for the enhancement
git checkout -b enh/XXX-short-name

# Develop and commit
git add -A && git commit -m "ENH-XXX: Description"

# After production verification, merge to main
git checkout main && git merge enh/XXX-short-name
```
