# Deployment Guide

Step-by-step process for developing, testing, and deploying changes to the
AEI Scheduler.

---

## Workflow Summary

```
1. Make changes   →  2. Deploy locally     →  3. Test locally       →  4. Deploy to production  →  5. Verify
   (edit code)        (aei_deploy.sh local)    (dev.aeihawaii.com)      (aei_deploy.sh prod)        (production)
```

---

## Unified Deploy Script (Recommended)

The `aei_deploy.sh` script handles the entire pipeline:

```bash
# Step 1: Deploy + pre-flight check locally
scripts/aei_deploy.sh local admin.php cancel_job.php

# Step 2: Test at http://dev.aeihawaii.com/scheduler/admin

# Step 3: Deploy to production (after local testing passes)
scripts/aei_deploy.sh production admin.php cancel_job.php

# Emergency: Rollback production
scripts/aei_deploy.sh rollback admin.php
```

### What it does automatically
- PHP syntax checking (`php -l`)
- PHP 7.3 compatibility scanning
- Protected file blocking (database.php, config.php, .htaccess)
- Timestamped backups before every overwrite
- PHP-FPM restart after local deploy (clears OPcache)
- Error log monitoring before/after deployment
- Checksum verification after production deploy
- File ownership restoration on remote (ec2-user:ec2-user)

### Options
- `--confirm` — Enable interactive prompts (for terminal use)
- `--from DIR` — Specify source directory (default: local dev site)
- `--verbose` — Detailed output

### File path shortcuts
You can use shorthand controller names:
```bash
scripts/aei_deploy.sh local admin.php          # resolves to system/application/controllers/admin.php
scripts/aei_deploy.sh local system/application/controllers/admin.php   # full path also works
```

---

## Prerequisites

### SSH Key
```bash
# Convert PPK key (one-time, or use the script)
scripts/ssh_connect.sh "echo 'Connected'"
```

### Local DNS
Ensure `/etc/hosts` contains:
```
127.0.0.1   dev.aeihawaii.com
```

### Verify Local Site
Visit `http://dev.aeihawaii.com/scheduler/admin` — login page should load.

---

## Step 1: Make Changes

Work in the local dev environment:
```
/var/www/html/dev_scheduler/SCHEDULER/
```

### Find the right file
- Use `SCHEDULER_DOCS/11-AI_CONTEXT_MAP/ACTIVE_ROUTES.csv` to map URLs to controllers
- Use `SCHEDULER_DOCS/11-AI_CONTEXT_MAP/CONTROLLER_INDEX.json` for feature lookups
- Controller code: `system/application/controllers/`
- Views: `system/application/views/`
- Models: `system/application/models/`

### PHP Compatibility
Production runs **PHP 7.3.30** (upgraded from 5.3.29 on 2026-02-23). Apache 2.4.61 with mod_php7.
- DB driver is `mysqli` (changed from `mysql`; backup at `database.php.bak_php53`)
- `[]` array syntax, `::class`, generators, traits, closures with `$this`, and `finally` blocks are now supported
- Still no PHP 7.4+ features (typed properties, arrow functions, null coalescing assignment)
- See `SCHEDULER_DOCS/09-ISSUES_AND_DEBT/PHP_COMPATIBILITY.md` for full list

---

## Step 2: Deploy & Test Locally

```bash
scripts/aei_deploy.sh local system/application/controllers/your_file.php
```

The script will:
1. Run PHP syntax check
2. Scan for PHP compatibility
3. Check for protected files
4. Create backup (`.bak.YYYYMMDD_HHMM`)
5. Copy to local dev site
6. Restart PHP 7.4 FPM
7. Check error logs for new errors
8. Print test URLs

### Manual testing
1. Visit the affected page on `http://dev.aeihawaii.com/scheduler/`
2. Test the specific functionality you changed
3. Check browser console for JavaScript errors
4. Check Apache error log: `tail -20 /var/log/apache2/error.log`

### Thorough Test (for significant changes)
See `TESTING_GUIDE.md` for the full testing checklist.

---

## Step 3: Deploy to Production

```bash
scripts/aei_deploy.sh production system/application/controllers/your_file.php
```

The script will:
1. Re-run all pre-flight checks
2. Verify file differs from production (skip if identical)
3. Create backup on remote (`.bak.YYYYMMDD_HHMM`)
4. Upload via SCP to `/tmp`
5. Copy to destination
6. Restore `ec2-user:ec2-user` ownership
7. Verify checksum matches (local vs remote)
8. Check remote error logs

### Legacy scripts (still available)
- `scripts/deploy_remote.sh` — Interactive single-file deploy
- `scripts/deploy_local.sh` — Interactive local deploy
- `scripts/verify_deployment.sh` — Post-deploy verification

---

## Step 4: Verify on Production

Visit `https://aeihawaii.com/dev/scheduler/` and test the affected functionality.

For detailed verification:
```bash
scripts/verify_deployment.sh remote system/application/controllers/your_file.php
```

---

## Rollback Procedure

If something goes wrong after deployment:

```bash
# Automatic: finds most recent backup and restores it
scripts/aei_deploy.sh rollback your_file.php

# Or use multiple files:
scripts/aei_deploy.sh rollback admin.php cancel_job.php
```

### Manual rollback
```bash
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "cp /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php.bak.YYYYMMDD_HHMM \
      /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php && \
   sudo chown ec2-user:ec2-user /var/www/vhosts/aeihawaii.com/httpdocs/scheduler/path/to/file.php"
```

---

## Protected Files

These files MUST NOT be synced between environments:

| File | Why |
|------|-----|
| `system/application/config/database.php` | Different DB credentials per environment |
| `system/application/config/config.php` | Different base_url per environment |
| `.htaccess` | Environment-specific settings |

If accidentally overwritten, see `CONNECTORS/DATABASE/README.md` for correct local values.

---

## Syncing FROM Remote

To pull latest production code to your local dev environment:

```bash
# Preview what would change
scripts/sync_from_remote.sh --dry-run

# Execute sync (protects config files automatically)
scripts/sync_from_remote.sh

# Skip database sync
scripts/sync_from_remote.sh --skip-db
```

See `LOCAL_SYNC/README.md` for the detailed manual procedure.
