# Update Local Scheduler from Remote Server

Step-by-step guide to update the local development copy of the AEI Scheduler
to match the remote production server. This process is read-only on the remote
server -- nothing is changed on production.

---

## Quick Reference

| Item | Value |
|------|-------|
| **Remote Server** | 18.225.0.90 (Julian@) |
| **Remote Scheduler Path** | `/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/` |
| **Local Dev Path** | `/var/www/html/dev_scheduler/SCHEDULER/` |
| **Backup Mirror Path** | `/mnt/e/remote_scheduler_backup/source/` |
| **SSH Key (PPK)** | `/var/www/html/AEI_REMOTE/CONNECTORS/SSH/schedular_server_private_key.ppk` |
| **SSH Key (converted)** | `/tmp/aei_key` |
| **Local DB** | localhost / root / scheduler / mandhdesign_schedular |
| **Remote DB** | 18.225.0.90 / AEI_User / P@55w02d7777 / mandhdesign_schedular |

---

## Overview

The update process has three steps:

```
Remote Server ──(1)──> Backup Mirror ──(2)──> Local Dev
                                                  │
Remote Database ──────────(3)──────────> Local Database
```

1. **Sync source code** from remote to backup mirror
2. **Copy source code** from backup mirror to local dev
3. **Sync the database** from remote to local

---

## Prerequisites

### SSH Key Setup

The remote server uses a PuTTY (.ppk) key that must be converted to OpenSSH
format before use on Linux:

```bash
puttygen /var/www/html/AEI_REMOTE/CONNECTORS/SSH/schedular_server_private_key.ppk \
  -O private-openssh -o /tmp/aei_key
chmod 600 /tmp/aei_key
```

The sync scripts handle this conversion automatically, but if running manual
commands you may need to do it yourself.

### Verify SSH Connection

```bash
ssh -i /tmp/aei_key -o StrictHostKeyChecking=no Julian@18.225.0.90 "echo OK && hostname"
```

You should see `OK` and the hostname `ip-172-31-39-5`.

---

## Step 1: Sync Source Code from Remote to Backup Mirror

This pulls the latest scheduler application code from the remote server into
the local backup mirror at `/mnt/e/remote_scheduler_backup/source/`.

### Preview Changes (Recommended)

```bash
/mnt/e/remote_scheduler_backup/scripts/sync_source.sh --dry-run
```

This shows what files would be updated without making any changes.

### Execute the Sync

```bash
/mnt/e/remote_scheduler_backup/scripts/sync_source.sh
```

**What gets synced:** All scheduler PHP code, views, assets, libraries, etc.

**What is excluded** (large/binary content not needed for development):
- `s3uploads/` -- user uploaded files (photos, documents)
- `pdftemp/`, `temppdf/` -- temporary PDF files
- `mail_sent/` -- copies of sent emails
- `uploads/` -- uploaded files
- `*.zip`, `*.tar.gz`, `*.log` -- archives and log files

### Check the Sync Log

```bash
cat /mnt/e/remote_scheduler_backup/source/.sync.log
```

---

## Step 2: Copy Source Code to Local Dev

This copies the synced source code from the backup mirror into the local
development environment.

### CRITICAL: Exclude Config Files

The local development environment uses different configuration than the remote
server. Two config files **MUST be excluded** from the copy to prevent breaking
the local site:

| File | Local Value | Remote Value |
|------|-------------|--------------|
| `system/application/config/database.php` | localhost / root / scheduler | 18.225.0.90 / AEI_User |
| `system/application/config/config.php` | `http://aeihawaii.com/scheduler/` | `http://aeihawaii.com/dev/scheduler/` |

**If these files get overwritten**, the local site will try to connect to the
remote database and use the wrong base URL. The local site will not function.

### Preview Changes (Recommended)

```bash
rsync -avz --dry-run --stats \
  --exclude="system/application/config/database.php" \
  --exclude="system/application/config/config.php" \
  --exclude=".sync.log" \
  "/mnt/e/remote_scheduler_backup/source/" \
  "/var/www/html/dev_scheduler/SCHEDULER/"
```

### Execute the Copy

```bash
rsync -avz --stats \
  --exclude="system/application/config/database.php" \
  --exclude="system/application/config/config.php" \
  --exclude=".sync.log" \
  "/mnt/e/remote_scheduler_backup/source/" \
  "/var/www/html/dev_scheduler/SCHEDULER/"
```

### What If Config Files Get Overwritten?

If the config files are accidentally overwritten, restore them:

**database.php** -- must contain these local settings:
```php
$db['default']['hostname'] = "localhost";
$db['default']['username'] = "root";
$db['default']['password'] = "scheduler";
$db['default']['database'] = "mandhdesign_schedular";
```

**config.php** -- must contain this local base URL:
```php
$config['base_url'] = "http://aeihawaii.com/scheduler/";
```

---

## Step 3: Sync the Database

This pulls a fresh copy of the production database and imports it locally.

### Option A: Use the Latest Weekly Backup (Faster)

Weekly backups run automatically every Sunday at 2:00 AM and are stored at
`/mnt/e/remote_scheduler_backup/database/`.

```bash
# List available backups
ls -lh /mnt/e/remote_scheduler_backup/database/full_backup_*.sql.gz

# Import the most recent backup
zcat /mnt/e/remote_scheduler_backup/database/full_backup_YYYY-MM-DD.sql.gz \
  | mysql -u root -pscheduler mandhdesign_schedular
```

Replace `YYYY-MM-DD` with the date of the most recent backup file.

**Pros:** Fast, no load on remote server.
**Cons:** Data may be up to 7 days old.

### Option B: Pull a Fresh Dump (Most Current)

This connects to the remote server via SSH and dumps the database directly.

```bash
# Convert SSH key if needed
puttygen /var/www/html/AEI_REMOTE/CONNECTORS/SSH/schedular_server_private_key.ppk \
  -O private-openssh -o /tmp/aei_key
chmod 600 /tmp/aei_key

# Dump remote database (takes ~40 minutes, creates ~640MB compressed file)
ssh -i /tmp/aei_key \
  -o StrictHostKeyChecking=no \
  -o ConnectTimeout=60 \
  -o ServerAliveInterval=60 \
  -o ServerAliveCountMax=120 \
  -o TCPKeepAlive=yes \
  Julian@18.225.0.90 \
  "mysqldump -u root \
    --single-transaction \
    --quick \
    --routines \
    --triggers \
    --max_allowed_packet=512M \
    mandhdesign_schedular" \
  | gzip > /mnt/e/remote_scheduler_backup/database/full_backup_$(date +%Y-%m-%d).sql.gz

# Import into local database (~2 minutes)
zcat /mnt/e/remote_scheduler_backup/database/full_backup_$(date +%Y-%m-%d).sql.gz \
  | mysql -u root -pscheduler mandhdesign_schedular
```

**Pros:** Most current data.
**Cons:** Takes ~40 minutes to dump, puts some load on the remote server.
Prefer running during off-hours.

### Monitor Dump Progress

While the dump is running, you can monitor the file size:

```bash
# Watch the dump file grow (Ctrl+C to stop watching)
watch -n 10 ls -lh /mnt/e/remote_scheduler_backup/database/full_backup_$(date +%Y-%m-%d).sql.gz
```

Expected final size: ~620-640 MB compressed.

### Verify the Import

```bash
# Check table count (should be 453 as of Feb 2026)
mysql -u root -pscheduler -e \
  "SELECT COUNT(*) as table_count FROM information_schema.tables
   WHERE table_schema = 'mandhdesign_schedular';"

# Check largest tables have data
mysql -u root -pscheduler -e \
  "SELECT table_name, table_rows FROM information_schema.tables
   WHERE table_schema = 'mandhdesign_schedular'
   AND table_rows > 10000
   ORDER BY table_rows DESC LIMIT 10;"
```

---

## Verification

After completing all three steps, verify everything is working:

### 1. Check Source Code

```bash
# Confirm file counts are similar
find /var/www/html/dev_scheduler/SCHEDULER/ -type f | wc -l
# Expected: ~9,000+ files
```

### 2. Check Database

```bash
mysql -u root -pscheduler -e \
  "SELECT COUNT(*) FROM information_schema.tables
   WHERE table_schema='mandhdesign_schedular';"
# Expected: 453+ tables
```

### 3. Check Local Site

Visit `http://aeihawaii.com/scheduler/admin` in a browser. The login page
should load. If it shows a database error, the `database.php` config file
was likely overwritten (see Step 2 recovery instructions above).

---

## Important Notes

### Do Not Modify Config Files

The two config files (`database.php` and `config.php`) contain settings
specific to the local environment. They must remain different from the
remote versions. Never sync or overwrite them with the remote copies.

### Local-Only Database Tables

The local database may have a few extra tables not present on the remote
server (e.g., `esign_requests`, `esign_signatures`, `form_drafts`). These
are local development tables and will persist across database imports since
the import only drops/recreates tables that exist in the dump.

### No Changes to Remote

All operations in this guide are **read-only on the remote server**. The
sync scripts pull data FROM remote, they never push TO remote.

### Excluded Files Are Intentional

Photos, PDFs, uploads, and other large binary files are excluded from the
source sync. These files are stored in `s3uploads/` on the remote server
and are not needed for local development. If you need specific uploaded
files for testing, you can pull them individually:

```bash
rsync -avz -e "ssh -i /tmp/aei_key" \
  "Julian@18.225.0.90:/var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/s3uploads/specific_file.jpg" \
  "/var/www/html/dev_scheduler/SCHEDULER/s3uploads/"
```

---

## File Locations Reference

```
/var/www/html/
├── scheduler/               -> symlink to dev_scheduler/SCHEDULER/
├── dev_scheduler/
│   └── SCHEDULER/           <- LOCAL DEV (update target)
└── AEI_REMOTE/
    ├── LOCAL_SYNC/           <- THIS DOCUMENTATION
    ├── CONNECTORS/SSH/       <- SSH keys
    └── _backups/             -> /mnt/e/remote_scheduler_backup/
        ├── source/           <- Backup mirror (intermediate step)
        ├── database/         <- Weekly database backups
        └── scripts/
            ├── sync_source.sh
            └── weekly_db_backup.sh
```

---

## Troubleshooting

### SSH Connection Refused

```bash
# Verify the key is converted and has correct permissions
ls -la /tmp/aei_key
# Should show: -rw------- (600 permissions)

# Re-convert if needed
puttygen /var/www/html/AEI_REMOTE/CONNECTORS/SSH/schedular_server_private_key.ppk \
  -O private-openssh -o /tmp/aei_key
chmod 600 /tmp/aei_key
```

### Sync Script Fails

```bash
# Check the sync log for errors
cat /mnt/e/remote_scheduler_backup/source/.sync.log

# Verify remote path exists
ssh -i /tmp/aei_key Julian@18.225.0.90 \
  "ls /var/www/vhosts/aeihawaii.com/httpdocs/dev/scheduler/"
```

### Database Import Fails

```bash
# Check MariaDB is running
service mysql status

# Check disk space
df -h

# Try importing with verbose errors
zcat /mnt/e/remote_scheduler_backup/database/full_backup_YYYY-MM-DD.sql.gz \
  | mysql -u root -pscheduler mandhdesign_schedular --verbose 2>&1 | tail -20
```

### Local Site Shows Errors After Update

Most likely cause: config files were overwritten. Check:

```bash
# Verify database.php points to localhost
grep hostname /var/www/html/dev_scheduler/SCHEDULER/system/application/config/database.php

# Verify config.php has correct base_url
grep base_url /var/www/html/dev_scheduler/SCHEDULER/system/application/config/config.php
```

If either shows remote values, restore the local values (see Step 2).

---

*Last synced: February 7, 2026*
*Remote tables: 453 | Local tables: 456 (3 local-only dev tables)*
