# Permit Tracking Plan - Salesforce Integration

## Goal

Replace the broken DPP HTML scraper with a Salesforce UI API integration
that automatically tracks permit status, inspections, fees, and reviews
for all AEI projects.

## Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                     AEI SCHEDULER                           │
│                                                             │
│  jobs table                    permit_sync table (NEW)      │
│  ┌──────────────────┐          ┌────────────────────────┐   │
│  │ id                │         │ id                     │   │
│  │ building_permit   │────────▶│ job_id                 │   │
│  │ customer_id       │         │ sf_permit_id           │   │
│  │ ...               │         │ sf_permit_name         │   │
│  └──────────────────┘          │ phase                  │   │
│                                │ status                 │   │
│  permit_details table          │ issue_date             │   │
│  (RETIRED - keep for           │ expiration_date        │   │
│   historical data)             │ total_fees             │   │
│                                │ total_payments         │   │
│  permit_files table            │ total_balance          │   │
│  (UNCHANGED - doc mgmt)        │ last_synced            │   │
│                                │ sync_status            │   │
│                                │ raw_json (TEXT)        │   │
│                                └────────────────────────┘   │
│                                          ▲                  │
│                                          │                  │
│                                  ┌───────┴───────┐         │
│                                  │  CRON JOB     │         │
│                                  │  (PHP script) │         │
│                                  └───────┬───────┘         │
└──────────────────────────────────────────┼──────────────────┘
                                           │
                                    HTTPS/JSON
                                           │
                                           ▼
┌──────────────────────────────────────────────────────────────┐
│               SALESFORCE (honolulu.my.site.com)              │
│                                                              │
│  1. Login via Aura endpoint → get sid cookie                 │
│  2. GET /services/data/v62.0/ui-api/records/{id}             │
│     with Bearer {sid} token                                  │
│                                                              │
│  Objects accessible:                                         │
│  ├── MUSW__Permit2__c  (529 fields) ← PRIMARY               │
│  ├── MUSW__Inspection__c (94 fields)                         │
│  ├── MUSW__Fee__c (61 fields)                                │
│  ├── MUSW__Review__c (51 fields)                             │
│  ├── MUSW__Address__c (55 fields)                            │
│  └── MUSW__Parcel__c (169 fields)                            │
└──────────────────────────────────────────────────────────────┘
```

## Phase 1: Database Schema

### New Table: `permit_sync`

Primary table for Salesforce-synced permit data.

```sql
CREATE TABLE permit_sync (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    job_id          INT NOT NULL,
    permit_number   VARCHAR(50) NOT NULL COMMENT 'AEI permit number (e.g. 189630)',
    sf_record_id    VARCHAR(18) COMMENT 'Salesforce record ID (e.g. a1Bcx000001Evx7EAC)',
    sf_permit_name  VARCHAR(100) COMMENT 'Full SF name (e.g. BP-2025-189630)',

    -- Status
    phase           VARCHAR(50) COMMENT 'Review, Inspection, etc.',
    status          VARCHAR(50) COMMENT 'In Progress, Complete, etc.',
    permit_type     VARCHAR(50) COMMENT 'Building, Electrical, etc.',
    work_type       VARCHAR(50) COMMENT 'Repair, New Construction, etc.',
    use_class       VARCHAR(50) COMMENT 'Residential, Commercial',
    use_type        VARCHAR(100) COMMENT 'Single-Unit Dwelling, etc.',

    -- Dates
    application_accepted DATE,
    application_expiration DATE,
    issue_date      DATE,
    completion_date DATE,
    expiration_date DATE,

    -- Fees
    total_fees      DECIMAL(10,2) DEFAULT 0,
    total_payments  DECIMAL(10,2) DEFAULT 0,
    total_balance   DECIMAL(10,2) DEFAULT 0,
    valuation       DECIMAL(12,2) DEFAULT 0,

    -- Location
    address_display VARCHAR(255),
    parcel_tmk      VARCHAR(50),

    -- CO
    co_required     VARCHAR(10),
    co_issued       TINYINT(1) DEFAULT 0,

    -- Sync metadata
    last_synced     DATETIME,
    sync_status     ENUM('ok','error','pending','not_found') DEFAULT 'pending',
    sync_error      VARCHAR(500),
    raw_json        MEDIUMTEXT COMMENT 'Full JSON response from SF for debugging',

    -- Timestamps
    created         DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated         DATETIME ON UPDATE CURRENT_TIMESTAMP,

    INDEX idx_job_id (job_id),
    INDEX idx_permit_number (permit_number),
    INDEX idx_sf_record_id (sf_record_id),
    INDEX idx_sync_status (sync_status),
    INDEX idx_status (status),
    INDEX idx_phase (phase)
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
```

### New Table: `permit_sync_inspections`

```sql
CREATE TABLE permit_sync_inspections (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    permit_sync_id  INT NOT NULL,
    sf_record_id    VARCHAR(18),
    inspection_name VARCHAR(100),
    inspection_type VARCHAR(100),
    status          VARCHAR(50),
    scheduled_date  DATE,
    result_date     DATE,
    inspector       VARCHAR(100),
    comments        TEXT,
    last_synced     DATETIME,
    created         DATETIME DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_permit_sync_id (permit_sync_id),
    FOREIGN KEY (permit_sync_id) REFERENCES permit_sync(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
```

### New Table: `permit_sync_fees`

```sql
CREATE TABLE permit_sync_fees (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    permit_sync_id  INT NOT NULL,
    sf_record_id    VARCHAR(18),
    fee_name        VARCHAR(200),
    fee_type        VARCHAR(100),
    amount          DECIMAL(10,2),
    status          VARCHAR(50),
    paid_by         VARCHAR(100),
    last_synced     DATETIME,
    created         DATETIME DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_permit_sync_id (permit_sync_id),
    FOREIGN KEY (permit_sync_id) REFERENCES permit_sync(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
```

### New Table: `permit_sync_reviews`

```sql
CREATE TABLE permit_sync_reviews (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    permit_sync_id  INT NOT NULL,
    sf_record_id    VARCHAR(18),
    review_name     VARCHAR(200),
    review_type     VARCHAR(100),
    status          VARCHAR(50),
    reviewer        VARCHAR(100),
    due_date        DATE,
    comments        TEXT,
    last_synced     DATETIME,
    created         DATETIME DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_permit_sync_id (permit_sync_id),
    FOREIGN KEY (permit_sync_id) REFERENCES permit_sync(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
```

### New Table: `permit_sync_log`

```sql
CREATE TABLE permit_sync_log (
    id              INT AUTO_INCREMENT PRIMARY KEY,
    permit_sync_id  INT,
    action          VARCHAR(50) COMMENT 'login, fetch, search, error',
    message         TEXT,
    created         DATETIME DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_permit_sync_id (permit_sync_id),
    INDEX idx_created (created)
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
```

## Phase 2: Sync Script

### Authentication Flow

```
1. POST https://honolulu.my.site.com/s/sfsites/aura
   Action: applauncher.LoginForm.login
   Params: username, password

2. Extract 'sid' cookie from response Set-Cookie header

3. Use sid as Bearer token for all UI API calls:
   Authorization: Bearer {sid}
```

### Sync Strategy

**For each AEI job with a building_permit number:**

1. **Search** - Use the permit number to find the Salesforce record
   - Try UI API: `GET /services/data/v62.0/ui-api/records/{sf_record_id}`
   - If sf_record_id not yet known, need to discover it (see Discovery below)

2. **Fetch** - Pull all relevant fields:
   ```
   GET /services/data/v62.0/ui-api/records/{id}?fields=
     MUSW__Permit2__c.Name,
     MUSW__Permit2__c.MUSW__Phase__c,
     MUSW__Permit2__c.MUSW__Status__c,
     MUSW__Permit2__c.MUSW__Type2__c,
     MUSW__Permit2__c.MUSW__Work_Type2__c,
     MUSW__Permit2__c.MUSW__Use_Class__c,
     MUSW__Permit2__c.MUSW__Use_Type__c,
     MUSW__Permit2__c.MUSW__Description__c,
     MUSW__Permit2__c.MUSW__Valuation__c,
     MUSW__Permit2__c.Application_Accepted_Date__c,
     MUSW__Permit2__c.Application_Expiration_Date__c,
     MUSW__Permit2__c.MUSW__Issue_Date__c,
     MUSW__Permit2__c.MUSW__DateCompleted__c,
     MUSW__Permit2__c.MUSW__Expiration_Date__c,
     MUSW__Permit2__c.MUSW__Total_Fees__c,
     MUSW__Permit2__c.MUSW__Total_Payments__c,
     MUSW__Permit2__c.MUSW__Total_Balance__c,
     MUSW__Permit2__c.MUSW__Address__r,
     MUSW__Permit2__c.CO_Required__c,
     MUSW__Permit2__c.Final_CO_Issued__c
   ```

3. **Compare** - Check if status/phase changed since last sync

4. **Update** - Write to `permit_sync` table

5. **Notify** - If status changed, trigger email notification

### Record Discovery

The UI API requires a Salesforce record ID. Since we don't have SOQL access,
we need to discover IDs. Options:

**Option A: Portal Search (Aura)**
- Use the portal's search functionality via Aura API
- Search for permit number through the community search bar
- Parse results for the record ID

**Option B: Manual Initial Mapping**
- For existing permits, manually look up each permit in the portal
- Record the SF record ID in permit_sync.sf_record_id
- Future permits entered via portal will have IDs captured at creation

**Option C: List View API**
- `GET /services/data/v62.0/ui-api/list-ui/MUSW__Permit2__c`
- Iterate through list views to find permits
- Limited by what list views are available to the community user

**Recommended: Option A + B hybrid**
- Batch-map existing active permits manually (one-time)
- Use Aura search for ongoing discovery of new permits

## Phase 3: Cron Schedule

```
# Sync all active permits every 4 hours during business hours
0 6,10,14,18 * * 1-5  /usr/local/bin/php5.3 /path/to/permit_sync_cron.php

# Full sync (including completed) once weekly
0 2 * * 0  /usr/local/bin/php5.3 /path/to/permit_sync_cron.php --full
```

## Phase 4: UI Integration

### Permit Tab Enhancement

Add a "DPP Status" section to the existing permit tab showing:

- **Phase**: Review / Inspection / etc. (from Salesforce)
- **Status**: In Progress / Complete (from Salesforce)
- **Last Synced**: timestamp
- **Fees**: Total / Paid / Balance
- **Key Dates**: Accepted, Issued, Expires
- **Link**: Direct link to DPP portal page

### Dashboard Widget

Add a permit status dashboard showing:

| Permit # | Job | Customer | Phase | Status | Balance | Last Updated |
|----------|-----|----------|-------|--------|---------|-------------|
| BP-2025-189630 | 45123 | Kahumana | Review | In Progress | $0.00 | 2h ago |

With filters for: All / In Review / Inspection / Completed / Has Balance

### Email Notifications

Trigger notifications when:
- Permit phase changes (e.g., Review → Inspection)
- Permit status changes (e.g., In Progress → Complete)
- New inspection scheduled
- Fee balance becomes due
- Permit approaching expiration

## Implementation Priority

| Priority | Task | Effort | Impact |
|----------|------|--------|--------|
| **P1** | Create permit_sync tables | 1 day | Foundation |
| **P1** | Build PHP auth + fetch script | 2-3 days | Core sync |
| **P2** | Map existing active permits to SF IDs | 1-2 days | Data |
| **P2** | Add DPP Status section to permit tab | 1-2 days | Visibility |
| **P3** | Cron job setup | 0.5 day | Automation |
| **P3** | Email notifications | 1 day | Alerts |
| **P4** | Dashboard widget | 1-2 days | Overview |
| **P4** | Inspection/Fee/Review sync | 2-3 days | Deep data |

## Technical Constraints

- **PHP 5.3.29** on production - must use `curl` for HTTP, no modern HTTP
  libraries. `json_decode/json_encode` available.
- **No OAuth client_id** - must use browser-session-style auth via Aura login
- **Session expiry** - SF sessions expire; script must handle re-auth
- **Rate limits** - SF API limits apply; batch fetches wisely
- **Record access** - Can only see permits associated with this account
  (shared permits or owned permits). May need DPP to share more permits
  with the AEI account.

## Open Questions

1. **Account scope** - Does this SF account have visibility to ALL AEI permits
   or just the ones manually shared? We saw only 1 shared permit (BP-2025-189630).
   AEI may need to request DPP share all their permits with this account.

2. **Multiple accounts** - Does AEI file permits under different applicant
   names? If so, may need multiple SF logins or request DPP consolidation.

3. **Electrical permits** - Are electrical permits in the same SF object
   (MUSW__Permit2__c) with a different `RecordTypeId`? Need to verify.

4. **Historical data** - Should we backfill permit_sync with data from the
   old permit_details table for pre-Salesforce permits?

5. **Two-way sync** - Is there a need to push data back to Salesforce
   (e.g., uploading documents, updating contractor info)?
