# Photovoltaic (PV) System Data Mapping Documentation

## Overview

This document provides a comprehensive mapping of all customer information, sales information, and system specifications for Photovoltaic (PV) systems to their corresponding database tables and columns.

**Database:** `mandhdesign_schedular`
**Data Sources:** proposals table, jobs table, customers table, and related lookup tables

---

## Proposals vs Jobs Relationship

### Understanding the Workflow

```
LEAD → PROPOSAL → JOB(S)
                    ├── Premeasure (job_type_id = 7)
                    ├── Prelag (job_type_id = 6)
                    └── PV Installation (job_type_id = 2)
```

### Key Relationship Rules

1. **Proposals are stored in the `proposal` table** - This is where sales creates quotes/estimates
2. **Jobs are created when a proposal is sold** - The `jobs.proposal_id` field links back to the proposal
3. **One proposal can generate multiple jobs** - Typically a PM (Premeasure), PL (Prelag), and PV (Installation) job
4. **Proposal status does NOT change to "sold"** - The status remains "new"; the presence of linked jobs indicates it was sold
5. **Job type 2 = Photovoltaic Installation** - This is the main PV installation job

### Identifying Sold Proposals

```sql
-- Find proposals that have been converted to jobs
SELECT p.id as proposal_id, p.dpp_firstname, p.dpp_lastname,
       j.id as job_id, j.job_type_id, jt.type as job_type
FROM proposal p
JOIN jobs j ON j.proposal_id = p.id
JOIN job_types jt ON j.job_type_id = jt.id
WHERE p.is_delete = 1  -- 1 = active, 0 = deleted
ORDER BY p.id DESC;
```

---

## Job Types Reference

| ID | Type | Code | Description |
|----|------|------|-------------|
| 1 | Solar Water Heating | SWH | SWH system installation |
| 2 | **Photovoltaic** | **PV** | **PV system installation** |
| 3 | Solar Attic Fan | SAF | Solar attic fan installation |
| 5 | Service Call | SC | Service/maintenance call |
| 6 | Prelag | PL | Pre-installation lag/preparation |
| 7 | Premeasure | PM | Site measurement before install |
| 8 | Electrical Upgrades | EL | Electrical service upgrades |
| 16 | AC | AC | Air conditioning |
| 21 | EV | EV | Electric vehicle charger |
| 29 | Roofing | RF | Roofing work |
| 30 | Storage | ST | Battery storage systems |

---

## Customer Information Mapping

### Table: `customers`

| Field | Type | Description | Example |
|-------|------|-------------|---------|
| `id` | int(11) | Primary key | 12345 |
| `first_name` | varchar(255) | Customer first name | "John" |
| `last_name` | varchar(255) | Customer last name | "Smith" |
| `first_name1` | varchar(255) | Secondary contact first name (spouse) | "Jane" |
| `last_name1` | varchar(255) | Secondary contact last name | "Smith" |
| `home_phone` | varchar(255) | Primary phone number | "808-555-1234" |
| `secondary_phone` | varchar(255) | Alternate phone | "808-555-5678" |
| `email` | varchar(255) | Email address | "john@example.com" |
| `address` | varchar(255) | Street address | "123 Aloha St" |
| `city` | varchar(255) | City | "Honolulu" |
| `state` | varchar(255) | State | "HI" |
| `zip_code` | varchar(255) | ZIP code | "96813" |
| `neighborhood` | varchar(255) | Neighborhood name | "Kaimuki" |
| `association` | varchar(255) | HOA name if applicable | "Sunset HOA" |
| `customer_type_id` | int(11) | 1=Residential, 2=Commercial | 1 |
| `household_type_id` | int(11) | 1=Single-Family, 2=Townhouse/Condo | 1 |
| `avg_electric_bill` | varchar(255) | Average monthly electric bill | "350" |
| `sun_zone` | varchar(255) | Solar zone for production estimates | "Zone A" |
| `heco_account_number` | varchar(255) | Hawaiian Electric account | "1234567890" |
| `heco_meter_number` | varchar(255) | HECO meter number | "M12345678" |
| `cust_tmk` | varchar(200) | Tax Map Key | "1-2-3-456-789" |
| `customer_status` | int(11) | 0=active | 0 |

### Customer Type Reference

| ID | Type |
|----|------|
| 1 | Residential |
| 2 | Commercial |

### Household Type Reference

| ID | Type |
|----|------|
| 1 | Single-Family |
| 2 | Townhouse/Condo |

---

## Proposal Table Mapping (Sales Information)

### Table: `proposal`

#### Customer/Contact Information (on Proposal)

| Field | Type | Description | Maps To |
|-------|------|-------------|---------|
| `id` | int(10) | Proposal ID (Primary Key) | - |
| `customer_id` | int(11) | Link to existing customer | customers.id |
| `customer_exist` | enum('y','n') | Whether customer exists | - |
| `dpp_firstname` | varchar(150) | First name on proposal | - |
| `dpp_lastname` | varchar(150) | Last name on proposal | - |
| `dpp_ot_firstname` | varchar(150) | Other first name (spouse) | - |
| `dpp_ot_lastname` | varchar(150) | Other last name | - |
| `address` | varchar(150) | Installation address | - |
| `city` | varchar(100) | City | - |
| `state` | varchar(10) | State | - |
| `zip` | varchar(50) | ZIP code | - |
| `phone` | varchar(100) | Primary phone | - |
| `sphone` | varchar(100) | Secondary phone | - |
| `email` | varchar(100) | Email address | - |
| `is_married` | tinyint(4) | Married status | 0=No, 1=Yes |

#### System Specifications

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `job_type` | tinyint(4) | System type (2=PV) | job_types |
| `system_size` | varchar(50) | System size in kW | - |
| `nom` | int(11) | Number of modules | - |
| `module_type` | int(11) | Module/panel brand | module_brands |
| `invertert` | int(11) | Inverter type ID | inverter_types |
| `not` | int(11) | Number of inverters | - |
| `noi` | int(11) | Number of inverters (alternate) | - |
| `roof_type` | int(11) | Roof type | roof_types |
| `storage` | tinyint(1) | Storage system included | 0=No, 1=Yes |
| `qoute_nopowerwalls` | tinyint(4) | **Number of Powerwall batteries** | - |
| `extra_inv` | varchar(50) | Additional inverter type ID | inverter_types |

#### Pricing Information

| Field | Type | Description |
|-------|------|-------------|
| `price` | varchar(50) | Total quoted price |
| `pv_price` | varchar(50) | PV system price only |
| `storage_price` | varchar(50) | Storage system price |
| `estimate` | int(11) | Estimate amount |
| `discount` | int(11) | Discount amount |
| `acharge` | int(11) | Additional charges |
| `addedcharge` | int(11) | Added charges |
| `cash_check_discount` | varchar(50) | Cash/check discount |
| `permit_fee` | varchar(11) | Permit fee |
| `assessed_value` | varchar(11) | Assessed property value |

#### Lead Source / Sales Information

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `user_id` | int(11) | Sales representative | users |
| `lead_source_id` | int(11) | Lead source | lead_origin |
| `lead_source` | varchar(50) | Lead source (text) | - |
| `status` | varchar(100) | Proposal status | "new" |
| `created` | datetime | Proposal creation date | - |
| `modified_date` | datetime | Last modified date | - |
| `form_state` | tinyint(4) | Form completion state | - |

#### Payment Method

| Field | Type | Description |
|-------|------|-------------|
| `mop` | varchar(100) | Method of payment code |
| `other_mop` | varchar(100) | Other payment method |
| `finacing_mop` | varchar(100) | Financing method |
| `select_financier` | varchar(40) | Selected financier |
| `name_financier` | varchar(100) | Financier name |
| `asb_financing` | int(11) | ASB financing flag |

#### HOA Information

| Field | Type | Description |
|-------|------|-------------|
| `home_owner_assoc` | tinyint(4) | Has HOA (0=No, 1=Yes) |
| `home_owner_assoc_name` | varchar(100) | HOA name |
| `hoa_id` | int(11) | HOA lookup ID |

#### Referral Information

| Field | Type | Description |
|-------|------|-------------|
| `referral_job` | int(11) | Referral job ID |
| `reff_first_name` | varchar(50) | Referrer first name |
| `reff_last_name` | varchar(50) | Referrer last name |
| `refer_customer_id` | int(11) | Referring customer ID |
| `refer_first_name` | varchar(100) | Referred by first name |
| `refer_last_name` | varchar(100) | Referred by last name |
| `refer_phone` | varchar(100) | Referrer phone |
| `refer_address` | varchar(100) | Referrer address |

---

## Jobs Table Mapping (PV Installation)

### Table: `jobs` (for job_type_id = 2)

#### Core Job Fields

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `id` | int(11) | Job ID (Primary Key) | - |
| `customer_id` | int(11) | Customer reference | customers |
| `proposal_id` | int(11) | **Link to original proposal** | proposal |
| `job_type_id` | int(11) | Job type (2 = PV) | job_types |
| `user_id` | int(11) | Sales rep who created | users |
| `job_date` | date | Scheduled installation date | - |
| `current_status` | varchar(100) | Current status text | - |
| `job_phase_status` | varchar(50) | Phase status | - |
| `job_status` | tinyint(2) | 0=active, 1=cancelled | - |

#### PV System Equipment

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `module_brand_id` | int(11) | Solar panel brand | module_brands |
| `module_count` | varchar(255) | **Number of panels** | - |
| `system_size` | varchar(255) | **System size in kW** | - |
| `module_power_id` | int(11) | Panel wattage | module_powers |
| `other_module_brand` | varchar(255) | Other brand (text) | - |
| `other_module_power` | varchar(255) | Other power (text) | - |

#### Inverter Information

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `inverter_type_id` | int(11) | **Inverter type** | inverter_types |
| `inverter_brand_id` | int(11) | Inverter brand | inverter_brands |
| `inverter_count` | varchar(255) | Number of inverters/optimizers | - |
| `other_inverter_type` | varchar(255) | Other type (text) | - |
| `other_inverter_brand` | varchar(255) | Other brand (text) | - |
| `inverter_accessible` | enum('y','n') | Inverter accessible | - |

#### Roof & Installation Details

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `roof_type_id` | int(11) | Roof type | roof_types |
| `rafter_type_id` | int(11) | Rafter type | rafter_types |
| `rafter_distance` | varchar(255) | Rafter spacing | - |
| `roof_orientation` | varchar(255) | Roof orientation | - |
| `roof_pitch` | varchar(255) | Roof pitch/angle | - |
| `roof_color` | varchar(255) | Roof color | - |
| `tilt_kit_needed` | enum('y','n') | Tilt kit required | - |
| `stories` | tinyint(4) | Number of stories | - |
| `other_roof_type` | varchar(255) | Other roof type (text) | - |

#### Electrical Details

| Field | Type | Description | Lookup Table |
|-------|------|-------------|--------------|
| `service_panel_amperage_id` | int(11) | Panel amperage | service_panel_amperages |
| `wire_run_type_id` | int(11) | Wire run type | wire_run_types |
| `wire_run_distance` | int(255) | Wire run distance | - |
| `subpanel_needed` | enum('y','n') | Subpanel needed | - |
| `subpanel_exist` | enum('y','n') | Subpanel exists | - |
| `breaker_box_location` | varchar(255) | Breaker box location | - |
| `meter_to_module_distance` | varchar(255) | Meter to module distance | - |
| `meter_location` | varchar(255) | Meter location | - |
| `servicepanelupgrade` | tinyint(3) | Service panel upgrade needed | - |

#### Permit Status Fields

| Field | Type | Description |
|-------|------|-------------|
| `electrical_permit` | int(11) | Electrical permit status (0=pending) |
| `building_permit` | int(11) | Building permit status |
| `nemstatus_permit` | int(11) | NEM (Net Energy Metering) status |
| `electrical_permit_status` | varchar(50) | Status text |
| `building_permit_status` | varchar(50) | Status text |
| `nemstatus_permit_status` | varchar(50) | Status text |
| `electrical_permit_completeiondate` | varchar(50) | Completion date |
| `building_permit_completeiondate` | date | Completion date |
| `nemstatus_permit_completeiondate` | date | NEM completion date |
| `permitting_status` | varchar(50) | Overall permitting status |
| `heco_status` | varchar(50) | HECO status |
| `hoa_status` | varchar(50) | HOA approval status |

#### Key Dates

| Field | Type | Description |
|-------|------|-------------|
| `job_date` | date | Scheduled date |
| `created` | datetime | Job creation date |
| `installed_date` | datetime | Installation completed |
| `paperwork_completed` | datetime | Paperwork done |
| `array_map` | datetime | Array map date |
| `nem_executed` | datetime | NEM executed date |
| `system_turn_on` | datetime | System activation date |

#### Pricing (on Job)

| Field | Type | Description |
|-------|------|-------------|
| `system_list_price` | varchar(255) | System list price |
| `net_total_price` | varchar(50) | Net total |
| `sub_total_price` | varchar(50) | Subtotal |
| `referral_amount` | varchar(255) | Referral credit |
| `state_tax_credit` | varchar(255) | State tax credit |
| `federal_tax_credit` | varchar(255) | Federal tax credit |
| `total_tax_credits` | varchar(255) | Total tax credits |

---

## Financial Data & Deal Types

### Deal Type (HECO Interconnection Program)

**Key Field: `proposal.deal_type`**

The `deal_type` field determines the HECO interconnection program and affects pricing calculations, system design, and export capabilities.

#### Active Deal Types Reference

| ID | Name | Description | Storage? |
|----|------|-------------|----------|
| 1 | NEM | Net Energy Metering (Legacy) | No |
| 5 | CGS+ / HESS CGS+ w/ Storage | Customer Grid-Supply Plus with Storage | Yes |
| 8 | NEM w/ Storage | Net Energy Metering with Storage | Yes |
| 9 | CGS | Customer Grid-Supply | No |
| 10 | CSS w/ Storage | Customer Self-Supply with Storage | Yes |
| 11 | CSS Non-Export | Customer Self-Supply Non-Export | No |
| 19 | CGS+ | Customer Grid-Supply Plus | No |
| **20** | **NEM+ Non Export** | NEM+ without grid export | No |
| **21** | **NEM+ w/Storage** | **NEM+ with Storage (Current Popular)** | **Yes** |
| **22** | **Smart DER Export w/ Storage** | **Smart DER with Export and Storage** | **Yes** |
| 23 | Smart DER Non-Export | Smart DER without export | No |
| **24** | **Smart DER Non-Export w/ Storage** | Smart DER Non-Export with Storage | Yes |
| 25 | Smart DER Export | Smart DER with Export | No |
| 51 | NEM+ w/Storage-PPA | NEM+ Storage via PPA financing | Yes |
| 52 | Smart DER Export w/ Storage-PPA | Smart DER Export Storage via PPA | Yes |

#### Deal Type Text Field

**Field: `proposal.deal_type_text`**

Stores the human-readable deal type name. Common values:
- "NEM+ w/Storage"
- "Smart DER Export w/ Storage"
- "HESS CGS+ w/ Storage"
- "CSS Non-Export"
- "CGS+"

#### Deal Type Usage Statistics

| Deal Type | Count |
|-----------|-------|
| NEM (1) | 14,830 |
| HESS CGS+ w/ Storage (5) | 9,260 |
| NEM w/ Storage (8) | 3,306 |
| CGS (9) | 3,453 |
| NEM+ w/Storage (21) | 1,904 |
| Smart DER Export w/ Storage (22) | 2,345 |
| CGS+ (19) | 1,297 |
| NEM+ Non Export (20) | 912 |

### Pricing Fields (Proposal Table)

#### Primary Pricing

| Field | Type | Description |
|-------|------|-------------|
| `price` | varchar(50) | **Total system price** |
| `pv_price` | varchar(50) | PV system price only |
| `storage_price` | varchar(50) | Storage/battery system price |
| `estimate` | int(11) | Initial estimate amount |
| `price_per_watt` | varchar(50) | Price per watt calculation |

#### Adjustments & Discounts

| Field | Type | Description |
|-------|------|-------------|
| `discount` | int(11) | Discount amount |
| `acharge` | int(11) | Additional charges (electrical work, etc.) |
| `addedcharge` | int(11) | Added charges |
| `discountadd` | int(11) | Additional discount |
| `adjust_price` | varchar(50) | Price adjustment amount |
| `cost_adj_option` | varchar(50) | Cost adjustment option selected |
| `cash_check_discount` | varchar(50) | Discount for cash/check payment |
| `cash_check_discount_new` | varchar(50) | Updated cash/check discount |
| `qoute_upcharge` | varchar(50) | Promotional code/upcharge |

#### Financing-Related Pricing

| Field | Type | Description |
|-------|------|-------------|
| `cpfdeal_listprice` | varchar(11) | CPF deal list price |
| `cpfdeal_partpdflistprice` | varchar(11) | CPF partial PDF list price |
| `cpfdeal_adminfee` | varchar(11) | CPF admin fee |
| `mitsubishi_listprice` | varchar(50) | Mitsubishi module list price |
| `lg_listprice` | varchar(50) | LG module list price |
| `hyundai_listprice` | varchar(50) | Hyundai module list price |

#### Permit & Assessment

| Field | Type | Description |
|-------|------|-------------|
| `permit_fee` | varchar(11) | Permit fee amount |
| `assessed_value` | varchar(11) | Property assessed value |

### Payment Method Fields

**Field: `proposal.mop` (Method of Payment)**

| Value | Description |
|-------|-------------|
| cash | Cash payment |
| check | Check payment |
| financing | General financing |
| financing_cpb | Central Pacific Bank financing |
| financing_usa | Hawaii USA financing |
| sunrun | Sunrun PPA/Lease |
| gemprogram | GEMS Program |
| union | Credit Union financing |
| other | Other payment method |

#### Related Payment Fields

| Field | Type | Description |
|-------|------|-------------|
| `mop` | varchar(100) | Method of payment code |
| `other_mop` | varchar(100) | Other payment method description |
| `finacing_mop` | varchar(100) | Financing method details |
| `select_financier` | varchar(40) | Selected financing company |
| `name_financier` | varchar(100) | Financier name |
| `asb_financing` | int(11) | ASB financing flag |
| `amount_deposit` | varchar(25) | Deposit amount |
| `total_amount_collected` | float | Total amount collected |
| `payment_note` | text | Payment notes |

### DML (Design Meeting Level / Offset Percentage)

**Field: `proposal.dml`**

Indicates the system's offset percentage relative to customer's electricity usage.

| Value | Meaning |
|-------|---------|
| 6 | <75% offset |
| 7 | >75-120% offset |
| 8 | >120-250% offset |
| 9 | >250% offset |

### Tax Credits (Jobs Table)

#### Primary Tax Credit Fields

| Field | Type | Description |
|-------|------|-------------|
| `state_tax_credit` | varchar(255) | State tax credit (primary system) |
| `federal_tax_credit` | varchar(255) | Federal tax credit (primary system) |
| `total_tax_credits` | varchar(255) | Combined tax credits |
| `state_excise_tax` | varchar(255) | State excise tax |

#### Multi-System Tax Credits

For systems with multiple installations (e.g., multiple rooftops):

| Field | Description |
|-------|-------------|
| `state_tax_credit_1` through `state_tax_credit_9` | State credits for systems 2-10 |
| `federal_tax_credit_1` through `federal_tax_credit_9` | Federal credits for systems 2-10 |

**Tax Credit Calculation Logic:**
```
State Tax Credit = 35% of system price (capped at $5,000)
Federal Tax Credit = 30% of system price (varies by tax_year field)
```

**Field: `proposal.tax_year`**

Stores the federal tax credit percentage (default .30 = 30%).

### Payment Tracking (Jobs Table)

| Field | Type | Description |
|-------|------|-------------|
| `payment_type_id` | int(11) | Payment type (FK to payment_types) |
| `payment_dpp` | tinyint(4) | Down payment received |
| `payment_amdpp` | varchar(20) | Down payment amount |
| `payment_fpp` | tinyint(4) | Final payment received |
| `payment_amfpp` | varchar(11) | Final payment amount |
| `cpd_payment1` | tinyint(4) | CPD payment 1 received |
| `cpd_payment1_value` | varchar(20) | CPD payment 1 amount |
| `cpd_payment2` | tinyint(4) | CPD payment 2 received |
| `cpd_payment2_value` | varchar(20) | CPD payment 2 amount |
| `full_paid` | tinyint(4) | Fully paid flag |
| `payment_recieved` | tinyint(4) | Payment received flag |
| `final_payment_recieved` | tinyint(4) | Final payment received flag |
| `final_amount_deposit` | varchar(25) | Final deposit amount |
| `deposit_der_amount` | varchar(50) | DER deposit amount |

### Commission Fields (Jobs Table)

| Field | Type | Description |
|-------|------|-------------|
| `sales_commission` | varchar(255) | Sales commission amount |
| `base_commsion` | float | Base commission |
| `sale_commission_status` | tinyint(4) | Commission paid status |
| `sales_commi_time` | timestamp | Commission payment timestamp |
| `contract_adjust_comm` | varchar(100) | Contract adjustment commission |
| `comm_method` | tinyint(4) | Commission calculation method |
| `comm_amt_per` | varchar(50) | Commission amount/percentage |
| `comm_user_id` | int(11) | Commission user ID |

### Pricing Summary (Jobs Table)

| Field | Type | Description |
|-------|------|-------------|
| `system_list_price` | varchar(255) | System list price |
| `list_price` | varchar(255) | List price |
| `net_total_price` | varchar(50) | Net total (after tax credits) |
| `sub_total_price` | varchar(50) | Subtotal (with GET tax) |
| `referral_amount` | varchar(255) | Referral credit amount |
| `qoute_upcharge` | double | Quote upcharge |
| `qoute_addwork` | double | Additional work cost |
| `discount` | double | Discount amount |
| `addedcharge` | double | Added charges |

### AEI Specials & Promotions

**Table: `aei_special`**

Stores promotional specials that can be applied to proposals.

| Field | Type | Description |
|-------|------|-------------|
| `id` | int | Special ID |
| `special_name` | varchar | Name (e.g., "33rd Anniversary Special") |
| `job_type_id` | int | Applicable job type |
| `discount_amount` | int | Discount amount |
| `profit_margin` | int | Profit margin percentage |
| `commision` | int | Commission percentage |
| `status` | int | 1=active, 0=inactive |

**Field: `proposal.aei_specials`** - Links to applied special.

### Financial Query Examples

```sql
-- Get proposal with full financial details
SELECT
    p.id,
    p.dpp_firstname,
    p.dpp_lastname,
    p.deal_type,
    p.deal_type_text,
    p.price as total_price,
    p.pv_price,
    p.storage_price,
    p.discount,
    p.acharge as additional_charges,
    p.mop as payment_method,
    p.select_financier,
    p.dml as offset_percentage,
    p.tax_year,
    p.qoute_nopowerwalls as battery_count
FROM proposal p
WHERE p.is_delete = 1
AND p.job_type = 2
ORDER BY p.created DESC;

-- Get job with payment and commission details
SELECT
    j.id,
    j.proposal_id,
    j.system_list_price,
    j.net_total_price,
    j.sub_total_price,
    j.state_tax_credit,
    j.federal_tax_credit,
    j.total_tax_credits,
    j.sales_commission,
    j.full_paid,
    j.payment_recieved,
    pt.type as payment_method
FROM jobs j
LEFT JOIN payment_types pt ON j.payment_type_id = pt.id
WHERE j.job_type_id = 2
AND j.job_status = 0
ORDER BY j.id DESC;

-- Get deal type distribution
SELECT
    p.deal_type,
    p.deal_type_text,
    COUNT(*) as count,
    AVG(CAST(p.price AS DECIMAL(10,2))) as avg_price
FROM proposal p
WHERE p.is_delete = 1
AND p.deal_type > 0
GROUP BY p.deal_type, p.deal_type_text
ORDER BY count DESC;
```

---

## Storage/Battery System Mapping

### Powerwall Type Identification

The `inverter_types` table contains battery/storage systems identified by the `type1` column = 'storage'.

#### Storage Inverter Types Reference

| ID | Type | Category | Capacity (kWh) | Output (kW) |
|----|------|----------|----------------|-------------|
| 298 | TESLA POWERWALL 6.4 KWH | Legacy (Powerwall 1) | 6.4 | 3.3 |
| 299 | TESLA POWERWALL 12.8 KWH | Legacy (Powerwall 1) | 12.8 | 5.0 |
| 329 | TESLA POWERWALL 2.0 (13.5 kWh) | Powerwall 2 | 13.5 | 5.0 / 7.0 peak |
| **334** | **Tesla Powerwall 2 AC (14.0 kWh)** | **Powerwall 2 (Current)** | **13.5** | **5.0 / 7.0 peak** |
| 502 | Powerwall+ | Powerwall+ | 13.5 | 5.0 / 7.0 peak |
| **528** | **Tesla Powerwall 3 AC (14.0 kWh)** | **Powerwall 3 (Current)** | **13.5** | **11.5 continuous** |
| 400 | LG Chem RESU10H | LG Chem | 8.8 | 5.0 / 7.0 peak |
| 417 | TABUCHI ELECTRIC | Tabuchi | 7.4 | 5.5 |
| 422 | Generac PWRcell | Generac | 3.0 per module | 7.6 |
| 448 | Enphase Encharge | Enphase | 3.36 | varies |
| 526 | Enphase 5P | Enphase | 5.0 | varies |
| 534 | Panasonic EVERVOLT | Panasonic | varies | varies |

### Battery Storage Capacity & Output Specifications

#### Database Fields for Storage Specifications

**Proposal Table - Storage Capacity Fields:**

| Field | Type | Description |
|-------|------|-------------|
| `qoute_nopowerwalls` | tinyint(4) | **Number of battery units** |
| `max_storage_capacity_kWh` | varchar(50) | **Maximum storage capacity (kWh)** |
| `hess_available_capacity` | varchar(50) | **Usable storage capacity (kWh)** |
| `available_capacity_kwh` | varchar(50) | Available capacity in kWh |
| `hess_capacity_supplied` | varchar(50) | HESS capacity supplied |
| `per_hess_capacity_supplied` | varchar(50) | Percentage of HESS capacity supplied |
| `storage` | tinyint(1) | Storage included flag (0/1) |
| `back_up_power` | int(11) | Backup power option (0=Select, 1=Yes, 2=No) |

**Proposal Table - Power Output Fields:**

| Field | Type | Description |
|-------|------|-------------|
| `available_kw_from_pv_system` | varchar(50) | Available kW from PV system |
| `production__kw_available_per_day` | varchar(50) | Production kW available per day |
| `pv_power_required_for_hess` | varchar(50) | PV power required for HESS |
| `daytime_load_kWh` | varchar(50) | Daytime load in kWh |

#### Capacity Calculations by Battery Type

**Tesla Powerwall 3 (ID 528):**
```
Usable Capacity per Unit: 13.5 kWh (stored as 11.5 in some calculations)
Total Capacity = qoute_nopowerwalls × 11.5 kWh
Continuous Output: 11.5 kW (integrated inverter)
```

**Tesla Powerwall 2 (ID 334):**
```
Usable Capacity per Unit: 13.5 kWh
Total Capacity = qoute_nopowerwalls × 13.5 kWh
Output: 5.0 kW continuous / 7.0 kW peak (requires separate inverter)
```

**LG Chem RESU10H (ID 400):**
```
Usable Capacity per Unit: 8.8 kWh
Total Capacity = qoute_nopowerwalls × 8.8 kWh
Output: 5.0 kW / 7.0 kW peak
```

**Generac PWRcell (ID 422):**
```
Capacity per Module: 3.0 kWh
Total Capacity = qoute_nopowerwalls × 3.0 kWh
Output: 7.6 kW
```

**Enphase Encharge (ID 448):**
```
Usable Capacity per Unit: 3.36 kWh
Total Capacity = qoute_nopowerwalls × 3.36 kWh
Output: Based on microinverters (349W × nom / 1000)
```

#### Inverter Output Power by Extra Inverter Type

When using storage with a separate inverter (`extra_inv` field), the output power varies:

| Extra Inverter ID | Model | Output (kW) |
|-------------------|-------|-------------|
| 293, 427 | SolarEdge SE3800H | 3.8 |
| 238, 428 | SolarEdge SE5000H | 5.0 |
| 294 | SolarEdge SE6000H | 6.0 |
| 239, 429 | SolarEdge SE7600H | 7.6 |
| 295, 431 | SolarEdge SE10000H | 10.0 |
| 296, 432 | SolarEdge SE11400H | 11.4 |
| 430 | SolarEdge SE9000H | 9.0 |
| 419 | SolarEdge EV 3800 | 3.8 |
| 420 | SolarEdge EV 7600 | 7.6 |
| 529 | Tesla 11.5 kWh | 11.5 |

#### Sample Storage Data from Database

| Batteries | Inverter Type | Max Capacity (kWh) | Usable Capacity (kWh) |
|-----------|---------------|--------------------|-----------------------|
| 1 | Powerwall 3 | 11.5 - 13.5 | 11.5 |
| 2 | Powerwall 3 | 23.0 | 23.0 |
| 3 | Powerwall 3 | 34.5 | 34.5 |
| 1 | Powerwall 2 | 13.5 | 13.5 |
| 2 | Powerwall 2 | 27.0 | 27.0 |

### Storage Specification Queries

```sql
-- Get storage proposals with capacity and power details
SELECT
    p.id,
    p.dpp_firstname,
    p.dpp_lastname,
    it.type as battery_type,
    p.qoute_nopowerwalls as battery_count,
    p.max_storage_capacity_kWh as max_capacity_kwh,
    p.hess_available_capacity as usable_capacity_kwh,
    p.available_kw_from_pv_system as available_kw,
    p.back_up_power,
    -- Calculate total capacity based on battery type
    CASE
        WHEN p.invertert = 528 THEN p.qoute_nopowerwalls * 11.5
        WHEN p.invertert = 334 THEN p.qoute_nopowerwalls * 13.5
        WHEN p.invertert = 400 THEN p.qoute_nopowerwalls * 8.8
        WHEN p.invertert = 422 THEN p.qoute_nopowerwalls * 3.0
        ELSE NULL
    END as calculated_capacity_kwh,
    -- Determine output power
    CASE
        WHEN p.invertert = 528 THEN 11.5
        WHEN p.invertert IN (334, 329) THEN 5.0
        WHEN p.invertert = 400 THEN 5.0
        WHEN p.invertert = 422 THEN 7.6
        ELSE NULL
    END as output_power_kw
FROM proposal p
LEFT JOIN inverter_types it ON p.invertert = it.id
WHERE p.invertert IN (334, 528, 400, 422, 448)
AND p.is_delete = 1
AND p.qoute_nopowerwalls > 0
ORDER BY p.created DESC;

-- Get complete storage system details with inverter info
SELECT
    p.id as proposal_id,
    p.qoute_nopowerwalls as battery_units,
    it.type as battery_type,
    p.max_storage_capacity_kWh,
    p.hess_available_capacity,
    ei.type as extra_inverter,
    p.storage_price,
    p.deal_type_text
FROM proposal p
LEFT JOIN inverter_types it ON p.invertert = it.id
LEFT JOIN inverter_types ei ON p.extra_inv = ei.id
WHERE p.storage = 1
AND p.is_delete = 1
ORDER BY p.id DESC
LIMIT 50;
```

### How to Identify Powerwall 3 vs Powerwall 2

```sql
-- Powerwall 3 Proposals
SELECT id, dpp_firstname, dpp_lastname, qoute_nopowerwalls, system_size
FROM proposal
WHERE invertert = 528  -- Powerwall 3 inverter type ID
AND is_delete = 1;

-- Powerwall 2 Proposals
SELECT id, dpp_firstname, dpp_lastname, qoute_nopowerwalls, system_size
FROM proposal
WHERE invertert = 334  -- Powerwall 2 inverter type ID
AND is_delete = 1;

-- Jobs with Powerwall systems
SELECT j.id, j.customer_id, j.inverter_type_id, it.type as inverter_name,
       p.qoute_nopowerwalls as battery_count
FROM jobs j
JOIN proposal p ON j.proposal_id = p.id
JOIN inverter_types it ON j.inverter_type_id = it.id
WHERE j.inverter_type_id IN (334, 528)  -- Powerwall 2 or 3
AND j.job_type_id = 2;  -- PV installation
```

### Battery Count Field

**Key Field: `proposal.qoute_nopowerwalls`**

This field stores the number of battery units:
- For **Powerwall 2**: Each unit is a separate 14.0 kWh battery
- For **Powerwall 3**: See interpretation below

### Powerwall 3 Inverter vs Expansion Unit Logic

**Important:** Powerwall 3 has an integrated inverter. When multiple battery units are installed, some are full Powerwall 3 units (with inverter) and others are Expansion Units (battery only, no inverter).

#### Key Relationship: Inverters vs Batteries

**The `inverter_count` field in the jobs table tells you the number of Powerwall 3 units with integrated inverters. The remaining batteries are Expansion Units.**

```
Expansion Units = Battery Count - Inverter Count
```

#### Examples from Database

| Inverter Count | Battery Count | Interpretation |
|----------------|---------------|----------------|
| 1 | 1 | 1 Powerwall 3 (with inverter), 0 Expansion Units |
| 1 | 2 | 1 Powerwall 3 (with inverter), **1 Expansion Unit** |
| 1 | 3 | 1 Powerwall 3 (with inverter), **2 Expansion Units** |
| 1 | 4 | 1 Powerwall 3 (with inverter), **3 Expansion Units** |
| 2 | 3 | 2 Powerwall 3 (with inverters), **1 Expansion Unit** |
| 3 | 4 | 3 Powerwall 3 (with inverters), **1 Expansion Unit** |
| 2 | 4 | 2 Powerwall 3 (with inverters), **2 Expansion Units** |

#### Reading the Data

**Example 1:** Database shows `inverter_count = 3`, `qoute_nopowerwalls = 4`
- **3 Powerwall 3 units** (each with integrated 11.5 kW inverter)
- **1 Expansion Unit** (battery only, no inverter)
- Total storage: 4 × 13.5 = 54 kWh
- Total inverter output: 3 × 11.5 = 34.5 kW

**Example 2:** Database shows `inverter_count = 1`, `qoute_nopowerwalls = 2`
- **1 Powerwall 3 unit** (with integrated 11.5 kW inverter)
- **1 Expansion Unit** (battery only, no inverter)
- Total storage: 2 × 13.5 = 27 kWh
- Total inverter output: 1 × 11.5 = 11.5 kW

#### Calculation Formulas

```
Number of PW3 Inverters = jobs.inverter_count (when inverter_type_id = 528)
Number of Expansion Units = proposal.qoute_nopowerwalls - jobs.inverter_count
Total Storage Capacity (kWh) = proposal.qoute_nopowerwalls × 13.5
Total Inverter Output (kW) = jobs.inverter_count × 11.5
```

#### SQL Query to Calculate Inverters vs Expansion Units

```sql
-- Calculate PW3 inverters and expansion units from jobs table
SELECT
    j.id as job_id,
    j.proposal_id,
    p.qoute_nopowerwalls as total_batteries,
    j.inverter_count as pw3_inverters,
    (p.qoute_nopowerwalls - j.inverter_count) as expansion_units,
    (p.qoute_nopowerwalls * 13.5) as total_capacity_kwh,
    (j.inverter_count * 11.5) as total_output_kw
FROM jobs j
JOIN proposal p ON j.proposal_id = p.id
WHERE j.inverter_type_id = 528  -- Powerwall 3
AND j.job_type_id = 2  -- PV installation
AND j.job_status = 0  -- Active
AND p.qoute_nopowerwalls > 0
ORDER BY j.id DESC;
```

#### Note on Data Sources

- **Battery Count (`qoute_nopowerwalls`)**: Stored in the `proposal` table
- **Inverter Count (`inverter_count`)**: Stored in the `jobs` table
- To get both values, you must JOIN proposals and jobs tables

### Battery Count Distribution (Powerwall 3)

| Batteries | Count of Proposals |
|-----------|-------------------|
| 1 | 1,371 |
| 2 | 1,286 |
| 3 | 271 |
| 4 | 44 |
| 5 | 16 |
| 6 | 14 |

---

## Lookup Tables Reference

### Module Brands (Solar Panels)

| ID | Brand |
|----|-------|
| 410 | Panasonic 325W |
| 436 | Panasonic 340W |
| 447 | REC 370W Alpha Series |
| 451 | REC 445W Alpha Series |
| 456 | Tesla 420W Black |
| 500 | Sunpower A-series 410 |
| 501 | Tesla 400W Black |
| 516 | Panasonic 410W |
| 523 | SunPower 400W Equinox Black |
| 532 | Panasonic 420W Black |

### Inverter Brands

| ID | Brand |
|----|-------|
| 4 | Enphase |
| 5 | PV Powered |
| 6 | SMA |

### Inverter Types (Non-Storage)

| ID | Type |
|----|------|
| 3 | Enphase M215 |
| 203 | M250 |
| 413 | Enphase IQ7 250W |
| 414 | IQ7+ |
| 415 | IQ7X |
| 504 | Enphase IQ8A |
| 505 | Enphase IQ8H |
| 38 | SOLAR EDGE SE3800A-US-HI |
| 238 | SOLAR EDGE SE5000H-US-HI |
| 239 | SOLAR EDGE SE7600H-US-HI |
| 295 | SOLAR EDGE SE10000H-US-HI |

### Roof Types

| ID | Type |
|----|------|
| 2 | Metal Roof - Standing Seam |
| 5 | Monier Tile |
| 6 | Metal/Aluminium |
| 8 | Asphalt Shingle |
| 9 | WoodShake |
| 10 | White/Flat Roof |
| 11 | Japanese Tile |
| 12 | Rolled Asphalt |
| 15 | Flat Tile |
| 240 | Decra Tile (metal) |

### Rafter Types

| ID | Type |
|----|------|
| 4 | Wood |
| 5 | Metal |
| 8 | Monier |

### Service Panel Amperages

| ID | Amperage |
|----|----------|
| 3 | 200A |
| 4 | 60A |
| 5 | 100A |
| 7 | 125A |

### Wire Run Types

| ID | Type |
|----|------|
| 4 | Attic Run |
| 5 | Direct |
| 9 | Over Roof |
| 11 | Under Eaves |

### Payment Types

| ID | Type |
|----|------|
| 1 | AMEX |
| 2 | Check/Cash |
| 3 | Cash |
| 4 | Financing |
| 5 | VISA |
| 6 | Mastercard |
| 15 | Financing - Central Pacific Bank |
| 16 | Financing - Hawaii USA |
| 17 | Sunrun |
| 27 | GEMS Program |
| 28 | Financing - Hawaii State Fed. Credit Union |

### Lead Sources

| ID | Origin |
|----|--------|
| 1 | Phone |
| 2 | Referral |
| 26 | Previous/Current customer |
| 30 | Internet |
| 31 | Social Media |
| 7 | Facebook |
| 8 | Yelp |
| 41 | Company Website |
| 44 | Company Vehicle |
| 4 | TV |

---

## Complete Query Examples

### Get Full PV Proposal Details

```sql
SELECT
    p.id as proposal_id,
    p.dpp_firstname,
    p.dpp_lastname,
    p.address,
    p.city,
    p.phone,
    p.email,
    p.system_size,
    p.nom as module_count,
    mb.brand as module_brand,
    it.type as inverter_type,
    p.qoute_nopowerwalls as battery_count,
    p.price,
    p.storage,
    lo.origin as lead_source,
    u.first_name as sales_rep,
    p.created,
    p.status
FROM proposal p
LEFT JOIN module_brands mb ON p.module_type = mb.id
LEFT JOIN inverter_types it ON p.invertert = it.id
LEFT JOIN lead_origin lo ON p.lead_source_id = lo.id
LEFT JOIN users u ON p.user_id = u.id
WHERE p.job_type = 2  -- PV
AND p.is_delete = 1
ORDER BY p.created DESC;
```

### Get PV Job with All Equipment Details

```sql
SELECT
    j.id as job_id,
    j.proposal_id,
    c.first_name,
    c.last_name,
    c.address,
    c.city,
    j.system_size,
    j.module_count,
    mb.brand as module_brand,
    it.type as inverter_type,
    j.inverter_count,
    rt.type as roof_type,
    spa.amperage as panel_amperage,
    j.job_date,
    j.current_status,
    j.installed_date,
    j.electrical_permit,
    j.building_permit,
    j.nemstatus_permit
FROM jobs j
JOIN customers c ON j.customer_id = c.id
LEFT JOIN module_brands mb ON j.module_brand_id = mb.id
LEFT JOIN inverter_types it ON j.inverter_type_id = it.id
LEFT JOIN roof_types rt ON j.roof_type_id = rt.id
LEFT JOIN service_panel_amperages spa ON j.service_panel_amperage_id = spa.id
WHERE j.job_type_id = 2  -- PV
AND j.job_status = 0  -- Active
ORDER BY j.job_date DESC;
```

### Get Powerwall 3 Systems with Battery Breakdown

```sql
SELECT
    p.id as proposal_id,
    p.dpp_firstname,
    p.dpp_lastname,
    p.system_size as pv_system_size,
    p.qoute_nopowerwalls as total_batteries,
    CEIL(p.qoute_nopowerwalls / 4) as pw3_inverter_count,
    (p.qoute_nopowerwalls - CEIL(p.qoute_nopowerwalls / 4)) as expansion_unit_count,
    (p.qoute_nopowerwalls * 13.5) as total_storage_kwh,
    p.price as total_price,
    p.pv_price,
    p.storage_price,
    j.id as job_id,
    j.job_date as scheduled_date,
    j.installed_date
FROM proposal p
LEFT JOIN jobs j ON j.proposal_id = p.id AND j.job_type_id = 2
WHERE p.invertert = 528  -- Powerwall 3
AND p.is_delete = 1
AND p.qoute_nopowerwalls > 0
ORDER BY p.created DESC;
```

---

## Field Mapping Summary Table

### Customer & System Information

| Business Concept | Proposal Field | Jobs Field | Notes |
|-----------------|----------------|------------|-------|
| Customer Name | dpp_firstname, dpp_lastname | via customer_id | Proposal has copy, job links to customer |
| Address | address, city, state, zip | via customer_id | |
| Phone | phone, sphone | via customer_id | |
| Email | email | via customer_id | |
| System Size (kW) | system_size | system_size | Copied from proposal |
| Module Count | nom | module_count | |
| Module Brand | module_type | module_brand_id | Different field names |
| Inverter Type | invertert | inverter_type_id | Different field names |
| Inverter Count | not | inverter_count | |
| **Battery Count** | **qoute_nopowerwalls** | - | **Proposal only** |
| Roof Type | roof_type | roof_type_id | |
| Sales Rep | user_id | user_id | |
| Lead Source | lead_source_id | - | Proposal only |
| Status | status | current_status, job_phase_status | Different status tracking |

### Financial & Deal Information

| Business Concept | Proposal Field | Jobs Field | Notes |
|-----------------|----------------|------------|-------|
| **Deal Type (HECO Program)** | **deal_type** | - | **Key field for interconnection** |
| Deal Type Name | deal_type_text | - | Human-readable deal name |
| **Total Price** | **price** | **system_list_price** | Main price field |
| PV Price | pv_price | - | PV-only portion |
| Storage Price | storage_price | - | Battery portion |
| Discount | discount | discount | |
| Additional Charges | acharge | addedcharge | Electrical work, etc. |
| Payment Method | mop | payment_type_id | Different formats |
| Financier | select_financier, name_financier | - | |
| Cash/Check Discount | cash_check_discount | - | |
| Offset % | dml | - | Design Meeting Level |
| State Tax Credit | - | state_tax_credit | Job only |
| Federal Tax Credit | - | federal_tax_credit | Job only |
| Tax Credit Year | tax_year | - | Federal credit % |
| Net Price | - | net_total_price | After credits |
| Commission | - | sales_commission | Job only |
| Deposit | amount_deposit | final_amount_deposit | |
| Payment Status | - | full_paid, payment_recieved | Job tracking |

### Storage/Battery System Information

| Business Concept | Proposal Field | Jobs Field | Notes |
|-----------------|----------------|------------|-------|
| **Battery Count** | **qoute_nopowerwalls** | - | Number of battery units |
| Battery Type | invertert | inverter_type_id | Links to inverter_types |
| **Max Capacity (kWh)** | **max_storage_capacity_kWh** | - | Total storage capacity |
| **Usable Capacity (kWh)** | **hess_available_capacity** | - | Usable storage capacity |
| Available Capacity | available_capacity_kwh | - | Available capacity |
| **Output Power (kW)** | available_kw_from_pv_system | - | Available kW from PV |
| Extra Inverter | extra_inv | - | Additional inverter for storage |
| Storage Flag | storage | - | 0/1 storage included |
| Backup Power | back_up_power | - | Backup power option |
| Daytime Load | daytime_load_kWh | - | Daytime load in kWh |

---

## Notes and Caveats

1. **Many fields are varchar**: Fields like system_size and price are stored as varchar - cast to numeric for calculations
2. **Proposal status stays "new"**: The presence of linked jobs indicates a sold proposal
3. **Battery count is proposal-only**: The `qoute_nopowerwalls` field exists only in the proposal table
4. **Multiple jobs per proposal**: A single proposal typically generates PM, PL, and PV jobs
5. **Soft deletes**: Use `is_delete = 1` for active proposals (counter-intuitive naming)
6. **Historical backups**: Ignore tables with dates in names (e.g., `system_price_13jan2015`)

---

*Last Updated: January 13, 2026*
