# Email System Documentation

This document describes how the AEI Scheduler system sends and receives emails, based on analysis of `/var/www/html/dev_scheduler/`.

---

## 1. Email Sending

### Primary Method: CodeIgniter Email Library + Mailjet SMTP

The application uses CodeIgniter's built-in Email library configured with **Mailjet** as the SMTP provider.

**SMTP Configuration:**
- Host: `in-v3.mailjet.com`
- Port: `587`
- Protocol: `smtp`
- Authentication: Enabled

**Key Configuration File:**
- `/var/www/html/dev_scheduler/SCHEDULER/system/libraries/Email.php`

### Alternative: PHPMailer Library

PHPMailer v5.2.6 is also available as an alternative sending method.

**PHPMailer Files:**
- `/SCHEDULER/assets/mailer/class.phpmailer.php`
- `/SCHEDULER/assets/mailer/class.smtp.php`

**Features:**
- SMTP authentication (PLAIN, LOGIN, NTLM, CRAM-MD5)
- TLS/SSL support via STARTTLS
- Full RFC 821 SMTP compliance

### Legacy Configuration (Commented Out)

Amazon SES was previously configured but is no longer active:
- Host: `email-smtp.us-east-1.amazonaws.com`

---

## 2. Email Receiving

### Method: IMAP Protocol via Gmail

The system polls Gmail accounts using PHP's IMAP functions to receive and process incoming emails.

**Account 1: Lead Creation**
- Purpose: Monitors incoming emails and creates leads in the system
- Server: `imap.gmail.com:993` (SSL/TLS)
- Handler: `checkGmailAccount()` function in `newcron.php`
- Storage: `lead` table with fields `gmailtimestamp`, `gmailmessage`, `lead_source = 'GMAIL'`

**Account 2: Support Ticket Creation**
- Purpose: Monitors support emails and creates tickets
- Server: `imap.gmail.com:993` (SSL/TLS)
- Handler: `save_ticket_imap()` function in `ticketmap.php`
- Filter: Searches for UNSEEN emails
- Parses: From, To, Subject, Date, Message body
- Storage: `osticket_detail` table

---

## 3. Email Sending Controllers

| Controller | Purpose | From Address |
|------------|---------|--------------|
| `emailaccount.php` | Invoice/accounting emails | `accounting@alternateenergyhawaii.com` |
| `emailcustomer.php` | Customer review/feedback emails | `admin@alternateenergyhawaii.com` |
| `newcron.php` | Automated cron job notifications | `sales@aeihawaii.com`, `do-not-reply@aeihawaii.com` |
| `ticketmap.php` | Ticket confirmation emails | `support@alternateenergyhawaii.com` |
| `emailtest.php` | Test email functionality | - |

**Controller Locations:** `/SCHEDULER/system/application/controllers/`

---

## 4. Email Templates

### Template Storage
- Database tables: `ost_email_template` and `ost_email_template_group`

### Template Variables
```
%{ticket.number}      - Ticket ID
%{ticket.subject}     - Ticket subject
%{ticket.topic.name}  - Help topic
%{customer.firstname} - Customer first name
%{customer.lastname}  - Customer last name
%{company.name}       - Company name
%{signature}          - Email signature
%{baseurl}            - Application base URL
```

### Template Types
- `ticket.autoresp` - Automatic ticket response
- `ticket.autoreply` - Auto-reply with ticket details
- Custom HTML templates with company branding

---

## 5. Email Tracking

### Tracking Methods
- **Read Tracking:** 1x1 pixel tracking method embedded in HTML emails
- **Database Tracking:** Records sent emails and read status

### Tracking Tables
| Table | Purpose |
|-------|---------|
| `invoicetracking` | Tracks invoice emails (with read status) |
| `customerreviewtracking` | Tracks customer review emails |
| `proposal_mail_history` | Proposal email history |
| `acproposal_mail_history` | AC proposal email history |
| `gc_mail_history` | GC email history |
| `homecharging_mail_history` | Home charging email history |

---

## 6. Email Configuration Settings

```
Charset:        UTF-8
Content Type:   text/html (configurable)
Word Wrap:      Enabled (76 characters)
Multipart:      mixed
Priority:       3 (Normal)
Timeout:        5 seconds
BCC Batch Mode: Disabled
Attachments:    Supported (PDF, images)
```

---

## 7. Key Files Reference

| File | Purpose |
|------|---------|
| `/SCHEDULER/system/libraries/Email.php` | Main Email class with SMTP config |
| `/SCHEDULER/assets/mailer/class.phpmailer.php` | PHPMailer library |
| `/SCHEDULER/assets/mailer/class.smtp.php` | SMTP protocol handler |
| `/SCHEDULER/system/helpers/email_helper.php` | Email helper functions |
| `/SCHEDULER/system/application/controllers/emailaccount.php` | Invoice emails |
| `/SCHEDULER/system/application/controllers/emailcustomer.php` | Customer emails |
| `/SCHEDULER/system/application/controllers/newcron.php` | Cron notifications + IMAP |
| `/SCHEDULER/system/application/controllers/ticketmap.php` | Ticket IMAP handler |

---

## 8. Dependencies

### PHP Extensions Required
- `imap` - For IMAP email receiving
- `openssl` - For SSL/TLS connections
- `mbstring` - For character encoding

### Third-Party Services
- **Mailjet** - SMTP email delivery service (primary)
- **Gmail IMAP** - Email receiving (polling-based)

### Libraries
- CodeIgniter Email Library (built-in)
- PHPMailer v5.2.6

---

## 9. Implementation Notes for New Project

### To Send Emails
1. Set up Mailjet account (or similar SMTP provider)
2. Configure SMTP settings in your email library
3. Use HTML templates with tracking pixels if needed

### To Receive Emails
1. Set up a Gmail account with IMAP enabled
2. Generate an App Password (if using 2FA)
3. Create a cron job or polling mechanism to check for new emails
4. Use PHP's `imap_*` functions to connect and fetch messages

### IMAP Connection Example (PHP)
```php
$hostname = '{imap.gmail.com:993/imap/ssl}INBOX';
$username = 'your-email@gmail.com';
$password = 'your-app-password';

$inbox = imap_open($hostname, $username, $password);
$emails = imap_search($inbox, 'UNSEEN');

foreach ($emails as $email_number) {
    $overview = imap_fetch_overview($inbox, $email_number, 0);
    $message = imap_fetchbody($inbox, $email_number, 1);
    // Process email...
}

imap_close($inbox);
```

### SMTP Sending Example (CodeIgniter)
```php
$this->load->library('email');

$this->email->from('sender@example.com', 'Sender Name');
$this->email->to('recipient@example.com');
$this->email->subject('Email Subject');
$this->email->message('<html>Email body</html>');
$this->email->attach('/path/to/file.pdf');

$this->email->send();
```
