#!/usr/bin/env python3
"""
Sync uploaded photo to local server (background process).
Called by upload.php via nohup to avoid blocking the mobile app response.

Retries up to INLINE_RETRIES times with exponential backoff before
falling back to the queue/ directory for cron-based retry via
process_retry_queue.py (every 15 minutes).

The inline retry handles transient network failures between the
remote and local servers.

Usage: python3 sync_to_local.py <file_path> <file_name> <job_id> <customer_id> \
           <job_type> <last_name> <first_name> <job_date> <photo_type>

Requires: requests (pip3 install requests)
"""

import os
import sys
import json
import time
import random
import logging
from datetime import datetime

UPLOAD_URL = 'https://upload.aeihawaii.com/uploadlocallat_kuldeep.php'
AUTH_TOKEN = 'remote_token'
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_DIR = os.path.join(SCRIPT_DIR, 'logs')
LOG_FILE = os.path.join(LOG_DIR, 'sync_to_local.log')
QUEUE_DIR = os.path.join(SCRIPT_DIR, 'queue')
CONNECT_TIMEOUT = 10
READ_TIMEOUT = 60
INLINE_RETRIES = 3
RETRY_DELAYS = [2, 4, 8]  # seconds between attempts


def setup_logging():
    """Configure file logging."""
    if not os.path.isdir(LOG_DIR):
        os.makedirs(LOG_DIR, mode=0o777, exist_ok=True)
    logging.basicConfig(
        filename=LOG_FILE,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S',
    )


def sync_file(file_path, file_name, job_id, customer_id, job_type,
              last_name, first_name, job_date, photo_type):
    """POST file and metadata to local server.

    Returns (success, error_reason) tuple.
    error_reason is None on success, or a string describing the failure.
    """
    import requests

    if not os.path.isfile(file_path):
        msg = "File not found: %s" % file_path
        logging.error(msg)
        return False, msg

    try:
        with open(file_path, 'rb') as f:
            files = {'file': (file_name, f)}
            data = {
                'auth_token':  AUTH_TOKEN,
                'file_name':   file_name,
                'job_id':      job_id,
                'customer_id': customer_id,
                'job_type':    job_type,
                'last_name':   last_name,
                'first_name':  first_name,
                'job_date':    job_date,
                'photo_type':  photo_type,
            }
            resp = requests.post(
                UPLOAD_URL,
                files=files,
                data=data,
                timeout=(CONNECT_TIMEOUT, READ_TIMEOUT),
                verify=False,
            )

        if resp.status_code == 200:
            logging.info("OK %s -> HTTP %d (%s)", file_name, resp.status_code, job_id)
            return True, None
        else:
            msg = "HTTP %d body=%s" % (resp.status_code, resp.text[:200])
            logging.warning("FAIL %s -> %s", file_name, msg)
            return False, msg

    except Exception as e:
        msg = "%s: %s" % (type(e).__name__, str(e))
        logging.error("ERROR %s -> %s", file_name, msg)
        return False, msg


def enqueue_failed(file_path, file_name, job_id, customer_id, job_type,
                   last_name, first_name, job_date, photo_type, error_reason):
    """Save failed sync payload to queue/ directory as JSON for later retry."""
    if not os.path.isdir(QUEUE_DIR):
        os.makedirs(QUEUE_DIR, mode=0o777, exist_ok=True)

    now = datetime.now()
    rand = '%06d' % random.randint(0, 999999)
    filename = 'sync_%s_%s.json' % (now.strftime('%Y%m%d_%H%M%S'), rand)

    payload = {
        'file_path':    file_path,
        'file_name':    file_name,
        'job_id':       job_id,
        'customer_id':  customer_id,
        'job_type':     job_type,
        'last_name':    last_name,
        'first_name':   first_name,
        'job_date':     job_date,
        'photo_type':   photo_type,
        'queued_at':    now.strftime('%Y-%m-%d %H:%M:%S'),
        'retry_count':  0,
        'last_error':   error_reason or 'Unknown error',
    }

    queue_path = os.path.join(QUEUE_DIR, filename)
    with open(queue_path, 'w') as f:
        json.dump(payload, f, indent=2)

    logging.info("QUEUED %s -> %s (reason: %s)", file_name, filename, error_reason)


if __name__ == "__main__":
    if len(sys.argv) != 10:
        print("Usage: sync_to_local.py <file_path> <file_name> <job_id> <customer_id>"
              " <job_type> <last_name> <first_name> <job_date> <photo_type>")
        sys.exit(1)

    setup_logging()

    file_path = sys.argv[1]
    file_name = sys.argv[2]
    job_id = sys.argv[3]
    customer_id = sys.argv[4]
    job_type = sys.argv[5]
    last_name = sys.argv[6]
    first_name = sys.argv[7]
    job_date = sys.argv[8]
    photo_type = sys.argv[9]

    kwargs = dict(
        file_path=file_path, file_name=file_name, job_id=job_id,
        customer_id=customer_id, job_type=job_type, last_name=last_name,
        first_name=first_name, job_date=job_date, photo_type=photo_type,
    )

    # Inline retry with exponential backoff.
    success = False
    error_reason = None
    for attempt in range(1, INLINE_RETRIES + 1):
        success, error_reason = sync_file(**kwargs)
        if success:
            break
        if attempt < INLINE_RETRIES:
            delay = RETRY_DELAYS[attempt - 1] if attempt - 1 < len(RETRY_DELAYS) else RETRY_DELAYS[-1]
            logging.info("RETRY %s -> attempt %d/%d failed, retrying in %ds",
                         file_name, attempt, INLINE_RETRIES, delay)
            time.sleep(delay)

    if not success and os.path.isfile(file_path):
        # All inline retries exhausted — queue for cron-based retry.
        # Only queue if source file still exists (network/server failure).
        # If file is gone, there's nothing to retry.
        enqueue_failed(error_reason=error_reason, **kwargs)

    sys.exit(0 if success else 1)
